package com.acecounter.android.acetm.common.queue;

import android.content.Context;

import androidx.annotation.Nullable;

import com.acecounter.android.acetm.common.config.ACECommonStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig.ACECONSTANT_INTEGER;
import com.acecounter.android.acetm.common.config.ACEStaticConfig.ACELOG_JSON_KEY;
import com.acecounter.android.acetm.common.file.ACEFileUtil;
import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.parameter.IACECommonAPI;
import com.acecounter.android.acetm.common.parameter.ICallbackOfTask;
import com.acecounter.android.acetm.common.parameter.Task;
import com.acecounter.android.acetm.common.util.TimeUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Date;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Queue;

public final class ACEWaitQueueManager implements ACEQueueManager {
    private final String TAG = ACEWaitQueueManager.class.getSimpleName();

    private Queue<Task> waitQueue;

    private ACEWaitQueueManager() {
        waitQueue = new LinkedList<>();
    }

    private static class Singleton {
        private static final ACEWaitQueueManager INSTANCE = new ACEWaitQueueManager();
    }

    public static ACEWaitQueueManager getInstance() {
        return ACEWaitQueueManager.Singleton.INSTANCE;
    }

    public int count() {
        return waitQueue.size();
    }

    public void enqueue(@Nullable final Task task) {
        if (task == null) {
            return;
        }

        if (waitQueue.size() >= ACEStaticConfig.ACECONSTANT_INTEGER.QUEUE_MAX_WAITING_COUNT) {
            ACELog.d(TAG,
                    String.format(
                            Locale.getDefault(),
                            "%d, 대기큐가 가득찼습니다.", task.getTaskHash()));
            return;
        }

        ACELog.d(TAG,
                String.format(
                        Locale.getDefault(),
                        "%d, 대기큐에 Task 를 추가합니다.", task.getTaskHash()));
        ACELog.d(TAG,
                String.format(
                        Locale.getDefault(),
                        "%d, desc: %s, create time: %s", task.getTaskHash(), task.getDescription(), task.getCreateTimeString()));
        waitQueue.add(task);
    }

    public void dequeue() {
        ACELog.d(TAG, "in dequeue");
        final Context _context = ACECommonStaticConfig.getContext();
        if (_context == null) {
            return;
        }

        if (waitQueue.isEmpty()) {
            return;
        }

        ACELog.d(TAG, "save task in wait queue.");
        final JSONArray _jsonArray = new JSONArray();
        while (!waitQueue.isEmpty()) {
            final Task _task = waitQueue.remove();
            _task.doWork(new ICallbackOfTask() {
                @Override
                public void callback() {
                    ACELog.d(TAG,
                            String.format(
                                    Locale.getDefault(),
                                    "%d, desc: %s, create time: %s", _task.getTaskHash(), _task.getDescription(), _task.getCreateTimeString()));
                    try {
                        JSONObject _json = _task.getJSONobject();
                        int count = ACEStaticConfig.ACECONSTANT_INTEGER.INIT_FAILED_LOG_COUNT;
                        if (_json.has(ACEStaticConfig.ACELOG_JSON_KEY.FAILED_COUNT)) {
                            count = _json.getInt(ACEStaticConfig.ACELOG_JSON_KEY.FAILED_COUNT) + 1;
                        }
                        _json.put(ACEStaticConfig.ACELOG_JSON_KEY.FAILED_COUNT, count);
                        _jsonArray.put(_json);
                    }
                    catch (JSONException e) {
                        e.printStackTrace();
                    }
                    _task.doneWork();
                }
            });
        }

        if (_jsonArray.length() > 0) {
            ACELog.d(TAG,
                    String.format(
                            Locale.getDefault(),
                            "_jsonArray.length(): %d", _jsonArray.length()));
            ACEFileUtil.appendFailedLogFile(_context, _jsonArray);
        }
    }

    public void enqueueStop() {
        ACELog.d(TAG, "Start process wait queue task.");
        while (!waitQueue.isEmpty()) {
            Task task = waitQueue.remove();
            ACELog.d(TAG,
                    String.format(
                            Locale.getDefault(),
                            "%d, desc: %s, create time: %s", task.getTaskHash(), task.getDescription(), task.getCreateTimeString()));
            ACESendQueueManager.getInstance().enqueue(task);
        }
    }

    public void loadFailLogs() {
        ACELog.d(TAG, "In process failed log.");
        Context _context = ACECommonStaticConfig.getContext();
        if (_context == null) {
            return;
        }

        @Nullable JSONArray _loadJsonArray = ACEFileUtil.readFailedLogFile(_context);
        if (_loadJsonArray == null || _loadJsonArray.length() == 0) {
            ACELog.d(TAG, "not found failed log.");
            return;
        }

        final long now = System.currentTimeMillis();
        final Date _now = new Date(now);
        IACECommonAPI commonAPI = ACECommonStaticConfig.getCommonAPI();
        if (commonAPI != null) {
            ACELog.i(TAG, "Start process failed log.");
            ACEFileUtil.overwriteFailedLogFile(_context, new JSONArray());
            for (int i = 0; i < _loadJsonArray.length(); ++i) {
                final JSONObject obj = _loadJsonArray.optJSONObject(i);
                if (obj != null) {
                    try {
                        ACELog.d(TAG, String.format(Locale.getDefault(), "%d::%s", i, obj.toString(2)));

                        if (obj.has(ACELOG_JSON_KEY.CREATETIME)) {
                            final Date _date = new Date(obj.getLong(ACELOG_JSON_KEY.CREATETIME));
                            ACELog.d(TAG,
                                    String.format(
                                            Locale.getDefault(),
                                            "failed log create time: '%s'", TimeUtils.dateToACEDateFormatString(_date)));
                            final long diff = _now.getTime() - _date.getTime();
                            final long diffDays = diff / ACECONSTANT_INTEGER.OneDayMilliseconds;
                            ACELog.d(TAG, String.format(Locale.getDefault(), "diffDays: %d", diffDays));
                            if (diffDays < 1) {
                                ACELog.d(TAG, "proceed failed log");
                                commonAPI.send(ACEStaticConfig.ACEofAPI.FailedLogSend, _context, obj);
                            }
                            else {
                                ACELog.d(TAG, "abandon failed log");
                            }
                        }
                        else {
                            ACELog.d(TAG,
                                    String.format(
                                            Locale.getDefault(),
                                            "FailLog not have '%s' key, But proceed failed log", ACELOG_JSON_KEY.CREATETIME));
                            commonAPI.send(ACEStaticConfig.ACEofAPI.FailedLogSend, _context, obj);
                        }
                    } catch (JSONException e) {
                        ACEDebugLog.wtf(TAG, new ACEException(e, "was occur exception on processing failed log.").toString());
                    } catch (Exception e) {
                        ACEDebugLog.wtf(TAG, new ACEException(e, "was occur exception on processing failed log.").toString());
                    }
                }
            }
        }
        else {
            ACELog.w(TAG, "failed not implementation common API.");
        }
    }
}
