package com.acecounter.android.acetm.common.queue;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.common.config.ACECommonStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig.ACECONSTANT_INTEGER;
import com.acecounter.android.acetm.common.config.ACEStaticConfig.ACELOG_JSON_KEY;
import com.acecounter.android.acetm.common.file.ACEFileUtil;
import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.parameter.IACECommonAPI;
import com.acecounter.android.acetm.common.parameter.ICallbackOfTask;
import com.acecounter.android.acetm.common.parameter.Task;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Date;
import java.util.Locale;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;

public final class ACESendQueueManager implements ACEQueueManager {
    private final String TAG = ACESendQueueManager.class.getSimpleName();

    @NonNull
    private final ExecutorService _executorService;

    private ACESendQueueManager() {
        _executorService = Executors.newSingleThreadExecutor();
    }

    private static class Singleton {
        private static final ACESendQueueManager INSTANCE = new ACESendQueueManager();
    }

    public static ACESendQueueManager getInstance() {
        return ACESendQueueManager.Singleton.INSTANCE;
    }

    public void enqueue(@Nullable final Task task) {
        if (task == null) {
            return;
        }

        Runnable runnable = new Runnable() {
            public void run() {
                task.doWork(new ICallbackOfTask() {
                    @Override
                    public void callback() {
                        ACELog.v(TAG,
                                String.format(
                                        Locale.getDefault(),
                                        "%d, in Task runnable callback.", task.getTaskHash()));
                        task.didWork();
                    }
                });
            }
        };

        try {
            if (!_executorService.isShutdown() && !_executorService.isTerminated()) {
                ACELog.d(TAG,
                        String.format(
                                Locale.getDefault(),
                                "%d, 전송큐에 Task 를 추가합니다.", task.getTaskHash()));
                ACELog.d(TAG,
                        String.format(
                                Locale.getDefault(),
                                "%d, desc: %s, create time: %s", task.getTaskHash(), task.getDescription(), task.getCreateTimeString()));

                _executorService.submit(runnable);
            }
            else {
                if (_executorService.isShutdown()) {
                    task.failed(new Exception("executorService is shutdowned."));
                }
                else if (_executorService.isTerminated()) {
                    task.failed(new Exception("executorService is terminated."));
                }
            }
        } catch (RejectedExecutionException e) {
            task.failed(e);
        } catch (NullPointerException e) {
            task.failed(e);
        }
    }

    public void dequeue() {
        ACELog.d(TAG, "in dequeue");
    }

    public void loadFailLogs() {
        ACELog.d(TAG, "In process failed log.");
        Context _context = ACECommonStaticConfig.getContext();
        if (_context == null) {
            return;
        }

        @Nullable JSONArray _loadJsonArray = ACEFileUtil.readFailedLogFile(_context);
        if (_loadJsonArray == null || _loadJsonArray.length() == 0) {
            ACELog.d(TAG, "not found failed log.");
            return;
        }

        final long now = System.currentTimeMillis();
        final Date _now = new Date(now);
        IACECommonAPI commonAPI = ACECommonStaticConfig.getCommonAPI();
        if (commonAPI != null) {
            ACELog.i(TAG, "Start process failed log.");
            ACEFileUtil.overwriteFailedLogFile(_context, new JSONArray());
            for (int i = 0; i < _loadJsonArray.length(); ++i) {
                final JSONObject obj = _loadJsonArray.optJSONObject(i);
                if (obj != null) {
                    try {
                        ACELog.d(TAG, String.format(Locale.getDefault(), "%d::%s", i, obj.toString(2)));

                        if (obj.has(ACELOG_JSON_KEY.CREATETIME)) {
                            final Date _date = new Date(obj.getLong(ACELOG_JSON_KEY.CREATETIME));
                            final long diff = _now.getTime() - _date.getTime();
                            final long diffDays = diff / ACECONSTANT_INTEGER.OneDayMilliseconds;
                            if (diffDays < 1) {
                                commonAPI.send(ACEStaticConfig.ACEofAPI.FailedLogSend, _context, obj);
                            }
                        }
                        else {
                            commonAPI.send(ACEStaticConfig.ACEofAPI.FailedLogSend, _context, obj);
                        }
                    } catch (JSONException e) {
                        ACEDebugLog.wtf(TAG, new ACEException(e, "was occur exception on processing failed log.").toString());
                    }
                }
            }
        }
        else {
            ACELog.w(TAG, "failed not implementation common API.");
        }
    }
}
