package com.acecounter.android.acetm.common.policy;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.common.config.ACECommonStaticConfig;
import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.parameter.IACECommonAPI;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public final class ACEPolicy {
    private static final String TAG = ACEPolicy.class.getSimpleName();

    private static final long REPEAT_PULLING_INTERVAL_SECOND_DEFAULT = 6 * 60 * 60;
    static long REPEAT_PULLING_INTERVAL_SECOND = REPEAT_PULLING_INTERVAL_SECOND_DEFAULT;

    private static ScheduledFuture<?> future;

    @Nullable
    private ScheduledThreadPoolExecutor exec;

    //region 싱글턴
    private static class Singleton {
        private static final ACEPolicy INSTANCE = new ACEPolicy();
    }

    @NonNull
    private static ACEPolicy getInstance() {
        return ACEPolicy.Singleton.INSTANCE;
    }
    //endregion 싱글턴

    private ACEPolicy() {
        exec = new ScheduledThreadPoolExecutor(1);
    }

    public synchronized static void requestPolicy(final Context context,
                                                  final long initialDelay) {
        if (ACEPolicy.getInstance().exec == null) {
            ACEDebugLog.wtf(TAG, "ScheduledThreadPoolExecutor is released. Can not request policy.");
            return;
        }

        if (future != null) {
            ACELog.i(TAG, "enabled repeat policy function.");
        }

        future = ACEPolicy.getInstance().exec.scheduleWithFixedDelay(new Runnable() {
            public void run() {
                try {
                    ACELog.i(TAG, "Start request policy.");
                    IACECommonAPI commonAPI = ACECommonStaticConfig.getCommonAPI();
                    if (commonAPI != null) {
                        commonAPI.requestPolicy(context);
                    }
                } catch (Exception e) {
                    ACEDebugLog.wtf(TAG, new ACEException(e, "was occur exception on request policy.").toString());
                    ACEPolicy.shutdownPolicyScheduledExecutor();
                }
            }
        }, initialDelay, ACEPolicy.REPEAT_PULLING_INTERVAL_SECOND, TimeUnit.SECONDS);
    }

    synchronized static void reRequestPolicy(final Context context) {
        if (ACEPolicy.getInstance().exec == null) {
            ACEDebugLog.wtf(TAG, "ScheduledThreadPoolExecutor is released. Can not request policy.");
            return;
        }

        boolean res = future.cancel(false);
        ACELog.d(TAG, "Previous task canceled: " + res);
        ACEPolicy.requestPolicy(context, ACEPolicy.REPEAT_PULLING_INTERVAL_SECOND);
    }

    static void shutdownPolicyScheduledExecutor() {
        if (ACEPolicy.getInstance().exec == null) {
            ACEDebugLog.wtf(TAG, "ScheduledThreadPoolExecutor is released. Can not shutdown.");
            return;
        }

        if (ACEPolicy.getInstance().exec.isTerminating()) {
            ACELog.d(TAG, "releasing repeat policy function.");
        }
        else if (!ACEPolicy.getInstance().exec.isShutdown() || !ACEPolicy.getInstance().exec.isTerminated()) {
            ACELog.i(TAG, "shutdown repeat policy function.");
            ACEPolicy.getInstance().exec.shutdown();
            future = null;
        }
        else {
            if (ACEPolicy.getInstance().exec.isShutdown()) {
                ACELog.d(TAG, "shutdown repeat policy function.");
            }
            else if (ACEPolicy.getInstance().exec.isTerminated()) {
                ACELog.d(TAG, "terminated repeat policy function.");
            }
            else {
                ACEDebugLog.wtf(TAG, "failed shutdown repeat policy function.");
            }
        }
    }
}
