package com.acecounter.android.acetm.common.parameter;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Build;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.BuildConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig.ICallbackDoneAndFail_Result_JSON_KEY;
import com.acecounter.android.acetm.common.gms.AdvertisingIdClient;
import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.policy.ACEPolicyParameters;
import com.acecounter.android.acetm.common.util.NetworkUtils;
import com.acecounter.android.acetm.common.util.StringUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;

import static com.acecounter.android.acetm.common.config.ACEStaticConfig.ACECONSTANT.DOMAIN_DUMMY;

public final class ACEParameterUtil {
    static final private String TAG = ACEParameterUtil.class.getSimpleName();

    //region common
    @NonNull
    public static String getRandomNumber6Length() {
        int value = new Random().nextInt(1000000);
        return String.format(Locale.getDefault(), "%06d", value);
    }
    //endregion common

    //region Utils
    private static long apkUpdateTime(@NonNull PackageManager packageManager,
                                      @NonNull String packageName) {
        long timestamp;
        try {
            ApplicationInfo info = packageManager.getApplicationInfo(packageName, 0);

            File apkFile = new File(info.sourceDir);
            if (apkFile.exists()) {
                timestamp = apkFile.lastModified();
            } else {
                timestamp = 1;
            }
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "apkUpdateTime 실패 했습니다.").toString());
            timestamp = 1;
        }
        return timestamp;
    }

    @Nullable
    public static String convertJSONToGetURL(@NonNull JSONObject params) {
        Uri.Builder builder = new Uri.Builder()
                .scheme(ACEStaticConfig.PROTOCOL_TYPE.HTTPS)
                .encodedAuthority(DOMAIN_DUMMY);
        try {
            Iterator keys = params.keys();
            while (keys.hasNext()) {
                String key = (String)keys.next();
                builder.appendQueryParameter(key, params.getString(key));
            }
        }
        catch (JSONException e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "convertJSONToGetURL 과정에 예외 발생").toString());
        }

        return builder.build().getQuery();
    }

    @NonNull
    private static String filterOnlyLetter(@Nullable String str) {
        try {
            if (TextUtils.isEmpty(str)) {
                return "";
            }

            StringBuilder result = new StringBuilder();
            char[] arr = str.toCharArray();
            for (char c : arr) {
                if (Character.isLetter(c)) {
                    result.append(c);
                }
            }
            return result.toString();
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "filterOnlyLetter 실패 했습니다.").toString());
            return "";
        }
    }

    public static void getAdvertisingIdInfo(@NonNull final Context context,
                                            @Nullable final ICallbackDoneAndFail callback) {
        AsyncTask.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    AdvertisingIdClient.AdInfo advertisingIdInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);

                    JSONObject result = new JSONObject();
                    result.put(ICallbackDoneAndFail_Result_JSON_KEY.IsAdTrackingEnabled, !advertisingIdInfo.isLimitAdTrackingEnabled());

                    if (!advertisingIdInfo.isLimitAdTrackingEnabled()) {
                        final String _adid = advertisingIdInfo.getId();
                        if (!TextUtils.isEmpty(_adid)) {
                            if (callback != null) {
                                callback.done(result.put(ICallbackDoneAndFail_Result_JSON_KEY.ADID, _adid));
                            }
                        }
                        else {
                            if (callback != null) {
                                callback.done(result.put(ICallbackDoneAndFail_Result_JSON_KEY.FAIL_MESSAGE, "_adid is null or empty"));
                            }
                        }
                    }
                    else {
                        if (callback != null) {
                            callback.done(result.put(ICallbackDoneAndFail_Result_JSON_KEY.ADID, ACEStaticConfig.ACECONSTANT.DEFAULT_ADID));
                        }
                    }
                }
                catch (Exception _ex) {
                    ACEDebugLog.wtf(TAG, new ACEException(_ex, "adid 를 얻는 과정에 예외 발생").toString());
                    if (callback != null) {
                        callback.failed(null);
                    }
                }
            }
        });
    }

    public static String getApplicationName(@NonNull Context context) {
        try {
            ApplicationInfo applicationInfo = context.getApplicationInfo();
            int stringId = applicationInfo.labelRes;
            return stringId == 0 ? applicationInfo.nonLocalizedLabel.toString() : context.getString(stringId);
        } catch (Exception e) {
            return "";
        }
    }

    public static String getApplicationVersion(@NonNull Context context) {
        try {
            PackageInfo i = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            return i.versionName;
        } catch (Exception e) {
            return "";
        }
    }

    @NonNull
    public static String getContextName(@NonNull Context context) {
        try {
            return context
                    .getClass()
                    .getName()
                    .replace(context.getPackageName() + ".", "")
                    .replaceAll("\\.", "/");
        } catch (Exception e) {
            ACELog.e(TAG, new ACEException(e, "context 이름을 얻지 못 했습니다.").toString());
            return "";
        }
    }

    @NonNull
    public static String getDeviceNormalizeName() {
        try {
            String manufacturer = Build.MANUFACTURER;
            String model = Build.MODEL;
            if (model.startsWith(manufacturer)) {
                return filterOnlyLetter(model)
                        .replace(" ", "")
                        .replace("+", "");
            }
            return (filterOnlyLetter(manufacturer) + model)
                    .replace(" ", "")
                    .replace("+", "");
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "디바이스 이름을 얻지 못 했습니다.").toString());
            return "";
        }
    }

    public static long getInstallUnixTimeStamp(@NonNull Context context) {
        long installTime = 0L;

        try {
            PackageManager manager = context.getApplicationContext().getPackageManager();
            String packageName = context.getPackageName();
            installTime = installTimeFromPackageManager(manager, packageName);
            if (installTime < 0) {
                installTime = apkUpdateTime(manager, packageName);
            }
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "getInstallUnixTimeStamp 실패 했습니다.").toString());
        }
        return installTime;
    }

    @NonNull
    public static String getIpAddress() {
        String cpSourceId = ACEPolicyParameters.getInstance().getCpSourceIP();
        if (!TextUtils.isEmpty(cpSourceId)) {
            return cpSourceId;
        }
        else {
            try {
                String ipv4 = NetworkUtils.getLocalIpv4Address();
                if (!TextUtils.isEmpty(ipv4)) {
                    return ipv4;
                }
                else {
                    // ipv6는 포맷이 v4와 달라 당분간 v4만 사용
                    // example: >>fe80::f0ac:71ff:febe:faa%dummy0<<
//                    String ipv6 = NetworkUtils.getLocalIpv6Address();
//                    if (!TextUtils.isEmpty(ipv6)) {
//                        return ipv6;
//                    }
//                    else {
                    return "0.0.0.0";
//                    }
                }
            } catch (Exception e) {
                ACEDebugLog.wtf(TAG, new ACEException(e, "ip 를 얻지 못 했습니다.").toString());
                return "0.0.0.0";
            }
        }
    }

    @NonNull
    public static String getLanguage() {
        try {
            String lang = Locale.getDefault().getLanguage();
            if (!TextUtils.isEmpty(lang)) {
                return lang;
            }
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "언어 정보를 얻지 못 했습니다.").toString());
        }
        return "ko";
    }

    @NonNull
    public static String getNetworkProvider(Context context) {
        try {
            TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            if (telephonyManager != null) {
                return telephonyManager.getSimOperator();
            }
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "네트워크 사업자 정보를 얻지 못 했습니다.").toString());
        }
        return "";
    }

    @NonNull
    public static String getOS() {
        String osName = "Android";
        String releaseNumber = Build.VERSION.RELEASE;
        try {
            if (releaseNumber != null) {
                osName = osName + releaseNumber.trim().replaceAll("\\.", "");
            }
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "os 정보를 얻는데 실패 했습니다.").toString());
        }
        return osName;
    }

    @NonNull
    public static String getResolution(Context context) {
        try {
            DisplayMetrics metrics = context.getResources().getDisplayMetrics();
            return (int) (metrics.widthPixels / metrics.density) + "*" + (int) (metrics.heightPixels / metrics.density);
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "해상도 정보를 얻지 못 했습니다.").toString());
        }
        return "0*0";
    }

    @NonNull
    public static String getSdkVersion() {
        return BuildConfig.VERSION_NAME;
    }

    public static String getVersion() {
        String fullVersion = Build.VERSION.RELEASE;
        String[] splitVersion = fullVersion.split("\\.");

        String parseVersion;

        if (splitVersion.length > 1) {
            parseVersion = splitVersion[0] + "." + splitVersion[1];
        } else {
            parseVersion = splitVersion[0];
        }

        return "Android " + parseVersion;
    }

    private static long installTimeFromPackageManager(@NonNull PackageManager packageManager,
                                                      @NonNull String packageName) {
        long timestamp;
        try {
            PackageInfo info = packageManager.getPackageInfo(packageName, 0);
            timestamp = info.firstInstallTime;
        } catch (Exception e) {
            timestamp = -1;
        }
        return timestamp;
    }

    public static String reduceTextCount(String rawString) {
        if (!TextUtils.isEmpty(rawString) &&
                rawString.length() > ACEStaticConfig.ACECONSTANT_INTEGER.MAX_LENGTH_REDUCE_TEXT_COUNT) {
            String result = rawString.substring(0, 124) + "...";
            ACELog.v(TAG, String.format(Locale.getDefault(), "%s 을 줄입니다.", rawString));
            return result;
        } else {
            return rawString;
        }
    }

    @Nullable
    static String stringAppendURLFormat(@Nullable String source,
                                        @Nullable String appandStr) {
        String _result = null;
        if (TextUtils.isEmpty(source)) {
            if (!TextUtils.isEmpty(appandStr)) {
                _result = String.format(Locale.getDefault(), "?%s", appandStr);
            }

            return _result;
        }

        _result = source.replaceFirst("^\\?", "");
        if (TextUtils.isEmpty(appandStr)) {
            _result = String.format(Locale.getDefault(), "?%s", _result);
        } else {
            String _appandStr = appandStr.replaceFirst("^\\?", "");
            if (!TextUtils.isEmpty(_appandStr)) {
                _result = String.format(Locale.getDefault(), "?%s&%s", _result, _appandStr);
            }
            else {
                _result = String.format(Locale.getDefault(), "?%s", _result);
            }
        }

        return _result;
    }
    //endregion Utils

    //region Map
    @NonNull
    public static Map<String, String> getURLToMap(@NonNull String getURL) {
        ACELog.d(TAG,
                String.format(Locale.getDefault(), "getURL: %s", getURL));

        Uri url = Uri.parse(String.format(Locale.getDefault(), "https://%s?%s", DOMAIN_DUMMY, getURL));
        Map<String, String> map = new TreeMap<String, String>();
        for (String key : url.getQueryParameterNames()) {
            String value = url.getQueryParameter(key);
            if (StringUtils.isNull(value)) {
                value = "";
            }
            map.put(key, value);
        }

        return map;
    }

    @NonNull
    public static Map<String, String> getURLToMapOnlyLetteringAtStartIndex(@NonNull String getURL) {
        getURL = StringUtils.onlyLetteringAtStartIndex(getURL);
        return getURLToMap(getURL);
    }
    //endregion Map
}
