package com.acecounter.android.acetm.common.logger;

import android.content.Context;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.BuildConfig;
import com.acecounter.android.acetm.common.config.ACECommonStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.http.ACENetwork;
import com.acecounter.android.acetm.common.http.ACENetworkAdapter;
import com.acecounter.android.acetm.common.http.ACENetworkResult;
import com.acecounter.android.acetm.common.http.IACENetworkParams;
import com.acecounter.android.acetm.common.parameter.ACEParameterUtil;
import com.acecounter.android.acetm.common.policy.ACEPolicyParameters;
import com.acecounter.android.acetm.common.util.NetworkUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;

public final class ACEDebugLog {
    private static final String TAG = ACEDebugLog.class.getSimpleName();
    private static int LNC_SEND_COUNT = 0;
    private static final int LNC_MAX_AVAILABLE_COUNT = 10;

    @NonNull
    private final ExecutorService _executorService;

    private ACEDebugLog() {
        _executorService = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
    }

    private static class Singleton {
        private static final ACEDebugLog INSTANCE = new ACEDebugLog();
    }

    private static ACEDebugLog getInstance() {
        return ACEDebugLog.Singleton.INSTANCE;
    }

    static void sendLNC(final int priority,
                        @NonNull String tag,
                        @Nullable String msg) {
        sendLNC(priority,
                tag,
                msg,
                null);
    }

    static void sendLNC(final int priority,
                        @NonNull final String tag,
                        @Nullable final String msg,
                        @Nullable final Throwable tr) {
        if (TextUtils.isEmpty(ACEPolicyParameters.getInstance().getToastAppKey())) {
            if (!ACELog.isLoggable(ACELog.DEBUG)) {
                return;
            }
            Log.d(TAG, "[SDK] appkey is empty");

            return;
        }
        if (ACEDebugLog.LNC_SEND_COUNT > ACEDebugLog.LNC_MAX_AVAILABLE_COUNT) {
            if (!ACELog.isLoggable(ACELog.DEBUG)) {
                return;
            }
            Log.d(TAG, "[SDK] LNC 전송 최대값을 초과 했습니다.");

            return;
        }
        ACEDebugLog.LNC_SEND_COUNT++;

        final String _logType = convertLogType(priority);

        Context _context = ACECommonStaticConfig.getContext();
        if (_context != null) {
            if (NetworkUtils.isNetworkConnected(_context)) {
                Runnable _runnable = new Runnable() {
                    public void run() {
                        ACENetworkAdapter networkAdapter = new ACENetworkAdapter();
                        networkAdapter.setGetURL(new ACENetwork.OnGetURL() {
                            @NonNull
                            @Override
                            public String getURL() {
                                Uri.Builder builder = new Uri.Builder()
                                        .scheme(ACEStaticConfig.PROTOCOL_TYPE.HTTPS)
                                        .encodedAuthority(ACEStaticConfig.ACECONSTANT.DOMAIN_LNC);
                                return builder.build().toString();
                            }
                        });
                        networkAdapter.setGetRequestMethod(new ACENetwork.OnGetRequestMethod() {
                            @NonNull
                            @Override
                            public String getRequestMethod() {
                                return ACEStaticConfig.HTTP_METHOD.POST;
                            }
                        });
                        networkAdapter.setGetRequestHeaderMap(new ACENetwork.OnGetRequestHeaderMap() {
                            @NonNull
                            @Override
                            public ConcurrentHashMap<String, String> getRequestHeaderMap() {
                                ConcurrentHashMap<String, String> _map = new ConcurrentHashMap<>();
                                _map.put(ACEStaticConfig.LOG.REQUEST_CONTENT_TYPE,
                                        ACEStaticConfig.LOG.REQUEST_CONTENT_TYPE_APPLICATION_JSON);
                                return _map;
                            }
                        });
                        networkAdapter.setGetRequestBody(new ACENetwork.OnGetRequestBody() {
                            @Nullable
                            @Override
                            public JSONObject getRequestBody() {
                                try {
                                    JSONObject _jsonBody = new JSONObject();
                                    if (!TextUtils.isEmpty(msg)) {
                                        _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.BODY, msg);
                                    }
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.HOST, ACECommonStaticConfig.getKey());
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.LOG_SOURCE, tag);
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.LOG_TYPE, _logType);
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.LOG_VERSION, ACEStaticConfig.ACECONSTANT.LNC_LOG_VERSION);
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.PROJECT_NAME, ACEPolicyParameters.getInstance().getToastAppKey());
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.PROJECT_VERSION, BuildConfig.VERSION_NAME);

                                    // 커스텀 필드 [S]
                                    _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.PLATFORM, ACEStaticConfig.ACECONSTANT.ANDROID);
                                    Context _context = ACECommonStaticConfig.getContext();
                                    if (_context != null) {
                                        _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.APP_NAME, ACEParameterUtil.getApplicationName(_context));
                                        _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.APP_VERSION, ACEParameterUtil.getApplicationVersion(_context));
                                        _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.DOMAIN, _context.getPackageName());
                                    }
                                    if (tr != null) {
                                        _jsonBody.put(ACEStaticConfig.ACEDebugLog_JSON_KEY.STACKTRACE, Log.getStackTraceString(tr));
                                    }
                                    // 커스텀 필드 [E]

                                    if (ACELog.getLogLevel() < ACELog.INFO) {
                                        android.util.Log.println(priority, tag, _jsonBody.toString(2));
                                    }
                                    return _jsonBody;
                                }
                                catch (JSONException e) {
                                    return null;
                                }
                            }
                        });
                        networkAdapter.request(new IACENetworkParams() {
                            @Override
                            public void completed(@NonNull ACENetworkResult response) {
                                ACEDebugLog.getInstance().completed(response);
                            }

                            @Override
                            public void failed(@NonNull Throwable throwable) {
                                ACEDebugLog.getInstance().failed(throwable);
                            }
                        });
                    }
                };

                try {
                    if (!ACEDebugLog.getInstance()._executorService.isShutdown() &&
                            !ACEDebugLog.getInstance()._executorService.isTerminated()) {
                        ACEDebugLog.getInstance()._executorService.submit(_runnable);
                    }
                    else {
                        if (ACEDebugLog.getInstance()._executorService.isShutdown()) {
                            ACEDebugLog.getInstance().failed(new Exception("executorService is shutdowned."));
                        }
                        else if (ACEDebugLog.getInstance()._executorService.isTerminated()) {
                            ACEDebugLog.getInstance().failed(new Exception("executorService is terminated."));
                        }
                    }
                } catch (RejectedExecutionException e) {
                    ACEDebugLog.getInstance().failed(e);
                } catch (NullPointerException e) {
                    ACEDebugLog.getInstance().failed(e);
                }
            }
            else if (ACELog.getLogLevel() < ACELog.INFO) {
                android.util.Log.println(priority, tag, "{ \"network\": \"offline\" }");
                android.util.Log.println(priority, tag, msg);
            }
        }
    }

    void completed(@Nullable final ACENetworkResult response) {
        if (ACELog.isLoggable(ACELog.DEBUG) &&
                response != null) {
            Log.d(TAG, "[SDK] " + response.toString());
        }
    }

    void failed(@Nullable final Throwable throwable) {
        if (ACELog.isLoggable(ACELog.DEBUG) &&
                throwable != null) {
            Log.d(TAG, "[SDK] LNC send failed", throwable);
        }
    }

    @NonNull
    private static String convertLogType(int priority) {
        switch (priority) {
            case ACELog.VERBOSE:
                return ACEStaticConfig.ACEDebugLog_Type.VERBOSE;
            case ACELog.DEBUG:
                return ACEStaticConfig.ACEDebugLog_Type.DEBUG;
            case ACELog.INFO:
                return ACEStaticConfig.ACEDebugLog_Type.INFO;
            case ACELog.WARN:
                return ACEStaticConfig.ACEDebugLog_Type.WARN;
            case ACELog.ERROR:
                return ACEStaticConfig.ACEDebugLog_Type.ERROR;
            case ACELog.ASSERT:
                return ACEStaticConfig.ACEDebugLog_Type.ASSERT;
            default:
                return ACEStaticConfig.ACEDebugLog_Type.VERBOSE;
        }
    }

    public static void v(String tag, String msg, Throwable tr) {
        sendLNC(ACELog.VERBOSE, tag, msg, tr);
    }

    public static void v(String tag, String msg) {
        v(tag, msg, null);
    }

    public static void wtf(String tag, String msg, Throwable tr) {
        sendLNC(ACELog.ASSERT, tag, msg, tr);
    }

    public static void wtf(String tag, String msg) {
        wtf(tag, msg, null);
    }
}
