package com.acecounter.android.acetm.common.http;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.common.config.ACECommonStaticConfig;
import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.logger.ACELog;

import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Http request abstract Class
 */
public abstract class ACENetwork {
    private final String TAG = ACENetwork.class.getSimpleName();

    @Nullable
    protected OnGetConnectTimeout _onGetConnectTimeoutListener;
    @Nullable
    protected OnGetReadTimeout _onGetReadTimeoutListener;
    @Nullable
    protected OnGetRequestBody _onGetRequestBodyListener;
    @Nullable
    protected OnGetRequestHeaderMap _onGetRequestHeaderMapListener;
    @Nullable
    protected OnGetRequestMethod _onGetRequestMethodListener;
    @Nullable
    protected OnGetURL _onGetURLListener;

    protected ACENetwork() {
        this._onGetConnectTimeoutListener = null;
        this._onGetReadTimeoutListener = null;
        this._onGetRequestBodyListener = null;
        this._onGetRequestHeaderMapListener = null;
        this._onGetRequestMethodListener = null;
        this._onGetURLListener = null;
    }

    //region public http
    public void request(@NonNull final IACENetworkParams argument) {
        HttpURLConnection connection = null;
        InputStream inputStream = null;
        OutputStream outputStream = null;
        try {
            // AOS 에서 오프라인시 로그 전송안하는 코드는 비활성화
//            Context _context = ACECommonStaticConfig.getContext();
//            if (_context != null) {
//                if (!NetworkUtils.isNetworkConnected(_context)) {
//                    argument.failed(new Throwable("not connected to the internet."));
//                    return;
//                }
//            }

            // request url [S]
            String _url = null;
            if (this._onGetURLListener != null) {
                _url = this._onGetURLListener.getURL();
            }
            else {
                _url = this._getURL();
            }

            if (TextUtils.isEmpty(_url)) {
                argument.failed(new Throwable("_url is null."));
                return;
            }
            // request url [E]

            // request headers [S]
            connection = (HttpURLConnection) new URL(_url).openConnection();
            ConcurrentHashMap<String, String> _mapHeader;
            if (this._onGetRequestHeaderMapListener != null) {
                _mapHeader = this._onGetRequestHeaderMapListener.getRequestHeaderMap();
            }
            else {
                _mapHeader = this._getRequestHeaderMap();
            }

            if (_mapHeader != null) {
                for (String key : _mapHeader.keySet()) {
                    ACELog.v(TAG,
                            String.format(
                                    Locale.getDefault(),
                                    "key: %s, value: %s", key, _mapHeader.get(key)));
                    connection.setRequestProperty(
                            key,
                            _mapHeader.get(key));
                }
            }
            // request headers [E]

            ACELog.i(TAG,
                    String.format(
                            Locale.getDefault(),
                            "request url: %s", connection.getURL()));

            // request ConnectTimeout [S]
            int _connectTimeout;
            if (this._onGetConnectTimeoutListener != null) {
                _connectTimeout = this._onGetConnectTimeoutListener.getConnectTimeout();
            }
            else {
                _connectTimeout = this._getConnectTimeout();
            }
            connection.setConnectTimeout(_connectTimeout);
            // request ConnectTimeout [E]
            // request ReadTimeout [S]
            int _readTimeout;
            if (this._onGetReadTimeoutListener != null) {
                _readTimeout = this._onGetReadTimeoutListener.getReadTimeout();
            }
            else {
                _readTimeout = this._getReadTimeout();
            }
            connection.setReadTimeout(_readTimeout);
            // request ReadTimeout [E]
            connection.setDoInput(true);
            // request method [S]
            String _method;
            if (this._onGetRequestMethodListener != null) {
                _method = this._onGetRequestMethodListener.getRequestMethod();
            }
            else {
                _method = this._getRequestMethod();
            }
            connection.setRequestMethod(_method);
            // request method [E]
            connection.connect();

            // request body [S]
            JSONObject _jsonForBody;
            if (this._onGetRequestBodyListener != null) {
                _jsonForBody = this._onGetRequestBodyListener.getRequestBody();
            }
            else {
                _jsonForBody = this._getRequestBody();
            }

            if (_jsonForBody != null) {
                String _body = _jsonForBody.toString();
                if (!TextUtils.isEmpty(_body)) {
                    outputStream = connection.getOutputStream();
                    if (outputStream != null) {
                        BufferedWriter bufferedWriter =
                                new BufferedWriter(
                                        new OutputStreamWriter(
                                                outputStream,
                                                ACECommonStaticConfig.CHARSET_UTF8));
                        bufferedWriter.write(_body);
                        bufferedWriter.flush();
                        bufferedWriter.close();
                        outputStream.close();
                    }
                }
            }
            // request body [E]

            final int responseCode = connection.getResponseCode();
            final String responseMessage = connection.getResponseMessage();
            final Map<String, List<String>> responseHeaders = connection.getHeaderFields();
            String _responseBody = null;
            if (responseCode == HttpURLConnection.HTTP_OK) {
                // Retrieve the response body as an InputStream.
                inputStream = connection.getInputStream();
                if (inputStream != null) {
                    BufferedReader bufferedReader =
                            new BufferedReader(
                                    new InputStreamReader(inputStream));

                    StringBuilder responseBodyBuilder = new StringBuilder();
                    String inputLine;
                    while ((inputLine = bufferedReader.readLine()) != null) {
                        responseBodyBuilder.append(inputLine);
                    }
                    bufferedReader.close();
                    _responseBody = responseBodyBuilder.toString();
                }

                argument.completed(new ACENetworkResult(
                        responseCode,
                        responseMessage,
                        _responseBody,
                        responseHeaders));
            }
            else {
                argument.failed(new Throwable(String.format(Locale.getDefault(), "responseCode is not ok(200): %d", responseCode)));
            }
        } catch (MalformedURLException e) {
            argument.failed(e);
        } catch (SocketTimeoutException e) {
            argument.failed(e);
        } catch (IOException e) {
            // InterruptedException 포함
            argument.failed(e);
        } catch (ClassCastException e) {
            argument.failed(e);
        } catch (NullPointerException e) {
            argument.failed(e);
        } catch (NumberFormatException e) {
            argument.failed(e);
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }
    //endregion public http

    //region abstract http
    protected int _getConnectTimeout() {
        return 1000;
    }

    protected int _getReadTimeout() {
        return 1000;
    }

    @Nullable
    protected JSONObject _getRequestBody() {
        return null;
    }

    @Nullable
    protected ConcurrentHashMap<String, String> _getRequestHeaderMap() {
        return null;
    }

    protected @ACEStaticConfig.HTTP_METHOD String _getRequestMethod() {
        return ACEStaticConfig.HTTP_METHOD.GET;
    }

    @Nullable
    protected String _getURL() {
        return null;
    }
    //endregion abstract http

    //region setter
    public void setGetConnectTimeout(@NonNull OnGetConnectTimeout listener) {
        _onGetConnectTimeoutListener = listener;
    }

    public void setGetReadTimeout(@NonNull OnGetReadTimeout listener) {
        _onGetReadTimeoutListener = listener;
    }

    public void setGetRequestBody(@NonNull OnGetRequestBody listener) {
        _onGetRequestBodyListener = listener;
    }

    public void setGetRequestHeaderMap(@NonNull OnGetRequestHeaderMap listener) {
        _onGetRequestHeaderMapListener = listener;
    }

    public void setGetRequestMethod(@NonNull OnGetRequestMethod listener) {
        _onGetRequestMethodListener = listener;
    }

    public void setGetURL(@NonNull OnGetURL listener) {
        _onGetURLListener = listener;
    }
    //endregion setter

    //region setter interface
    public interface OnGetConnectTimeout {
        int getConnectTimeout();
    }

    public interface OnGetReadTimeout {
        int getReadTimeout();
    }

    public interface OnGetRequestBody {
        @Nullable
        JSONObject getRequestBody();
    }

    public interface OnGetRequestHeaderMap {
        @Nullable
        ConcurrentHashMap<String, String> getRequestHeaderMap();
    }

    public interface OnGetRequestMethod {
        @NonNull
        String getRequestMethod();
    }

    public interface OnGetURL {
        @NonNull
        String getURL();
    }
    //endregion setter interface
}
