package com.acecounter.android.acetm.common.file;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Build;
import android.os.Environment;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresPermission;

import com.acecounter.android.acetm.common.config.ACEStaticConfig.ACECONSTANT_INTEGER;
import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.util.JsonUtil;
import com.acecounter.android.acetm.common.util.PermissionUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Locale;

import static com.acecounter.android.acetm.common.config.ACEStaticConfig.ACECONSTANT;

public final class ACEFileUtil {
    private static final String TAG = ACEFileUtil.class.getSimpleName();

    private static File getInternalStorageFile(@NonNull Context context) {
        File filesDir = context.getFilesDir();
        return new File(filesDir.getAbsolutePath() + "/" + ACECONSTANT.FAILED_SAVE_FILE_NAME);
    }

    private static File getExternalStorageFile() {
        String externalStorageDir = Environment.getExternalStorageDirectory().getAbsolutePath();
        File path = new File(externalStorageDir + ACECONSTANT.EXTERNAL_STORAGE_PATH);
        if (!path.mkdirs()) {
            ACELog.e(TAG, "디렉토리 생성 실패: " + path.getAbsolutePath());
        }
        return new File(path.getAbsolutePath() + "/" + ACECONSTANT.FAILED_SAVE_FILE_NAME);
    }

    private static String readFailedLogFileFromInternalStorage(@NonNull Context context) {
        File file = getInternalStorageFile(context);
        if (!file.exists()) {
            return null;
        }
        return readFile(file);
    }

    @RequiresPermission(Manifest.permission.READ_EXTERNAL_STORAGE)
    private static String readFailedLogFileFromExternalStorage() {
        File file = getExternalStorageFile();
        if (!file.exists()) {
            return null;
        }
        return readFile(file);
    }

    @Nullable
    private static String readFile(File file) {
        FileReader fileReader = null;
        BufferedReader bufferedReader = null;
        try {
            fileReader = new FileReader(file);
            bufferedReader = new BufferedReader(fileReader);

            StringBuilder sb = new StringBuilder();
            char cbuf[] = new char[1024];
            int len;
            while ((len = bufferedReader.read(cbuf, 0, cbuf.length)) != -1) {
                sb.append(cbuf, 0, len);
            }
            return sb.toString();

        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (bufferedReader != null) {
                try {
                    bufferedReader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (fileReader != null) {
                try {
                    fileReader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    @Nullable
    @SuppressLint("MissingPermission")
    public static synchronized JSONArray readFailedLogFile(@NonNull Context context) {
        String result = readFailedLogFileFromInternalStorage(context);
        if (TextUtils.isEmpty(result)) {
            if (PermissionUtil.hasPermission(context, Manifest.permission.READ_EXTERNAL_STORAGE)) {
                result = readFailedLogFileFromExternalStorage();
            }
        }

        if (!TextUtils.isEmpty(result)) {
            try {
                return new JSONArray(result);
            }
            catch (JSONException e) {
                e.printStackTrace();
                return null;
            }
        }
        else {
            return null;
        }
    }

    @SuppressLint("MissingPermission")
    private static boolean writeFailedLogFile(@NonNull Context context,
                                              @NonNull JSONArray jsonArray) {
        boolean result = false;
        if (PermissionUtil.hasPermission(context, Manifest.permission.WRITE_EXTERNAL_STORAGE)) {
            result = writeFailedLogFileToExternalStorage(jsonArray.toString());
        }
        return result || writeFailedLogFileToInternalStorage(context, jsonArray.toString());
    }

    @SuppressWarnings("UnusedReturnValue")
    @SuppressLint("MissingPermission")
    public static synchronized boolean appendFailedLogFile(final @NonNull Context context,
                                                           final @NonNull JSONArray jsonArray) {
        JSONArray _willSaveJsonArray = new JSONArray();
        final JSONArray _loadJsonArray = readFailedLogFile(context);
        if (_loadJsonArray == null) {
            ACELog.d(TAG, "_loadJsonArray is null.");
        }
        else {
            ACELog.d(TAG,
                    String.format(
                            Locale.getDefault(),
                            "_loadJsonArray.length(): %d", _loadJsonArray.length()));
        }

        if (_loadJsonArray != null && _loadJsonArray.length() > 0) {
            for (int i = 0; i < _loadJsonArray.length(); ++i) {
                final JSONObject obj = _loadJsonArray.optJSONObject(i);
                if (obj != null) {
                    _willSaveJsonArray.put(obj);
                }
            }
        }
        else {
            ACELog.i(TAG, "저장된 실패 로그가 없습니다.");
        }

        if (_willSaveJsonArray.length() > 0) {
            int willWriteFailedLogCount = _willSaveJsonArray.length() + jsonArray.length();
            if (willWriteFailedLogCount > ACECONSTANT_INTEGER.QUEUE_MAX_FAILED_LOG_COUNT) {
                int sliceFailedLogCount =
                        Math.abs(
                                (_willSaveJsonArray.length() -
                                        Math.abs(ACECONSTANT_INTEGER.QUEUE_MAX_FAILED_LOG_COUNT - jsonArray.length())));

                ACEDebugLog.wtf(TAG, String.format(Locale.getDefault(), "sliceFailedLogCount: %d", sliceFailedLogCount));
                try {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                        for (int i = 0; i < sliceFailedLogCount; ++i) {
                            _willSaveJsonArray.remove(0);
                        }
                    } else {
                        _willSaveJsonArray = JsonUtil.removeIndexCount(
                                0,
                                sliceFailedLogCount,
                                _willSaveJsonArray);
                    }
                } catch (IndexOutOfBoundsException e) {
                    ACEDebugLog.wtf(TAG, new ACEException(e, "오래된 실패 로그 버리는 과정에 예외 발생").toString());
                } catch (UnsupportedOperationException e) {
                    ACEDebugLog.wtf(TAG, new ACEException(e, "오래된 실패 로그 버리는 과정에 예외 발생").toString());
                } finally {
                    if (_willSaveJsonArray == null) {
                        _willSaveJsonArray = new JSONArray();
                    }
                }
            }
        }

        ACELog.d(TAG,
                String.format(
                        Locale.getDefault(),
                        "_willSaveJsonArray.length(): %d, jsonArray.length(): %d", _willSaveJsonArray.length(), jsonArray.length()));

        for (int i = 0; i < jsonArray.length(); ++i) {
            final JSONObject obj = jsonArray.optJSONObject(i);
            if (obj != null) {
                _willSaveJsonArray.put(obj);
            }
        }

        ACELog.d(TAG,
                String.format(
                        Locale.getDefault(),
                        "(%d) 개의 실패로그들을 저장합니다.", _willSaveJsonArray.length()));

        return writeFailedLogFile(context, _willSaveJsonArray);
    }

    @SuppressWarnings("UnusedReturnValue")
    @SuppressLint("MissingPermission")
    public static synchronized boolean overwriteFailedLogFile(final @NonNull Context context,
                                                              final @NonNull JSONArray jsonArray) {
        return writeFailedLogFile(context, jsonArray);
    }

    private static boolean writeFailedLogFileToInternalStorage(@NonNull Context context,
                                                               @NonNull String content) {
        File file = getInternalStorageFile(context);
        return writeFile(file, content);
    }

    @RequiresPermission(Manifest.permission.WRITE_EXTERNAL_STORAGE)
    private static boolean writeFailedLogFileToExternalStorage(@NonNull String content) {
        File file = getExternalStorageFile();
        return writeFile(file, content);
    }

    private static boolean writeFile(@NonNull File file,
                                     @NonNull String content) {
        FileWriter fileWriter = null;
        BufferedWriter bufferedWriter = null;
        try {
            fileWriter = new FileWriter(file);
            bufferedWriter = new BufferedWriter(fileWriter);
            bufferedWriter.write(content);
            return true;
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (bufferedWriter != null) {
                try {
                    bufferedWriter.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
            if (fileWriter != null) {
                try {
                    fileWriter.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return false;
    }
}
