package com.acecounter.android.acetm;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.common.config.ACEStaticConfig;
import com.acecounter.android.acetm.common.logger.ACEDebugLog;
import com.acecounter.android.acetm.common.logger.ACEException;
import com.acecounter.android.acetm.common.parameter.ACEParameterUtil;
import com.acecounter.android.acetm.common.util.TypeUtil;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public final class AceProduct {
    private static final String TAG = AceProduct.class.getSimpleName();

    @Nullable
    private String name;
    @Nullable
    private String code;
    @NonNull
    private String price;
    private int quantity;
    private ArrayList<AceOption> options;

    /*
     * AceProduct 객체를 생성하기 위한 생성자
     *
     * @param name 제품의 이름
     * @param code 제품의 코드
     * @param price 제품의 가격
     * @param quantity 제품의 수량
     */
    public AceProduct(@Nullable String name,
                      @Nullable String code,
                      double price,
                      int quantity) {
        setName(name);
        setCode(code);
        setPrice(price);
        setQuantity(quantity);
        options = new ArrayList<>();
    }

    @Nullable
    public String getCode() {
        return this.code;
    }

    public void setCode(@Nullable String code) {
        if (!TextUtils.isEmpty(code)) {
            this.code = code.replaceAll("\\^|@", "");
        }
        else {
            this.code = "";
        }
    }

    @Nullable
    public String getName() {
        return this.name;
    }

    public void setName(@Nullable String name) {
        if (!TextUtils.isEmpty(name)) {
            this.name = name.replaceAll("\\^|@", "");
        }
        else {
            this.name = "";
        }
    }

    public String getPrice() {
        return this.price;
    }

    public void setPrice(double price) {
        this.price = TypeUtil.doubleToString(price);
    }

    public int getQuantity() {
        return this.quantity;
    }

    public void setQuantity(int quantity) {
        this.quantity = quantity;
    }

    public void addOption(AceOption option) {
        options.add(option);
    }

    @NonNull
    public List<String> toQueryString() {
        ArrayList<String> queryList = new ArrayList<>();

        StringBuilder builder = new StringBuilder();
        try {
            if (options.size() == 0) {
                builder.append(this.code).append("@")
                        .append(ACEParameterUtil.reduceTextCount(this.name)).append("@")
                        .append(this.price).append("@")
                        .append(this.quantity).append("@@^");
            } else {
                for (AceOption option : options) {
                    if (builder.length() > ACEStaticConfig.ACECONSTANT_INTEGER.ACEPRODUCT_QUERY_MAX_LENGTH) {
                        builder.deleteCharAt(builder.length() - 1);
                        queryList.add(builder.toString());
                        builder = new StringBuilder();
                    }
                    builder.append(this.code).append('@')
                            .append(ACEParameterUtil.reduceTextCount(this.name)).append('@')
                            .append(this.price).append('@')
                            .append(option.toQueryString()).append('^');
                }
            }
            builder.deleteCharAt(builder.length() - 1);
            queryList.add(builder.toString());
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "AceProduct.toQueryString 실패 했습니다.").toString());
        }

        return queryList;
    }

    @NonNull
    public static List<String> toProductListQueryString(@Nullable List<AceProduct> products) {
        ArrayList<String> queryList = new ArrayList<>();

        StringBuilder builder = new StringBuilder();
        try {
            if (products == null || products.size() == 0) {
                return queryList;
            }
            for (AceProduct product : products) {
                if (builder.length() > ACEStaticConfig.ACECONSTANT_INTEGER.ACEPRODUCT_QUERY_MAX_LENGTH) {
                    builder.deleteCharAt(builder.length() - 1);
                    queryList.add(builder.toString());
                    builder = new StringBuilder();
                }

                if (product.options.size() == 0) {
                    builder.append(product.code)
                            .append("@")
                            .append(ACEParameterUtil.reduceTextCount(product.name))
                            .append("@").append(product.price)
                            .append("@").append(product.quantity)
                            .append("@@^");
                } else {
                    for (AceOption option : product.options) {
                        if (builder.length() > ACEStaticConfig.ACECONSTANT_INTEGER.ACEPRODUCT_QUERY_MAX_LENGTH) {
                            builder.deleteCharAt(builder.length() - 1);
                            queryList.add(builder.toString());
                            builder = new StringBuilder();
                        }
                        builder.append(product.code)
                                .append('@')
                                .append(ACEParameterUtil.reduceTextCount(product.name))
                                .append('@')
                                .append(product.price)
                                .append('@')
                                .append(option.toQueryString())
                                .append('^');
                    }
                }
            }
            builder.deleteCharAt(builder.length() - 1);
            queryList.add(builder.toString());
        } catch (Exception e) {
            ACEDebugLog.wtf(TAG, new ACEException(e, "toProductListQueryString 실패 했습니다.").toString());
        }
        return queryList;
    }

    @NonNull
    private JSONObject toJsonToString() {
        JSONObject _json = new JSONObject();

        try {
            _json.put("name", name);
            _json.put("code", code);
            _json.put("price", price);
            _json.put("quantity", quantity);
            JSONArray _option = new JSONArray();
            for (AceOption item : options) {
                _option.put(item.toJsonToString());
            }
            _json.put("options", _option);
        }
        catch (JSONException e) {
            e.printStackTrace();
        }

        return _json;
    }

    @NonNull
    public String toProductString() {
        return toJsonToString().toString();
    }

    public static final class AceOption {
        @Nullable
        private String code;
        @Nullable
        private String name;
        private int quantity;

        /*
         * AceOption 객체를 생성하기 위한 생성자
         *
         * @param code 옵션의 코드
         * @param name 옵션의 이름
         * @param quantity 옵션의 수량
         */
        public AceOption(@Nullable String code,
                         @Nullable String name,
                         int quantity) {
            setName(name);
            setCode(code);
            setQuantity(quantity);
        }

        @Nullable
        public String getCode() {
            return this.code;
        }

        public void setCode(@Nullable String code) {
            if (!TextUtils.isEmpty(code)) {
                this.code = code.replaceAll("\\^|@", "");
            }
            else {
                this.code = "";
            }
        }

        @Nullable
        public String getName() {
            return this.name;
        }

        public void setName(@Nullable String name) {
            if (!TextUtils.isEmpty(name)) {
                this.name = name.replaceAll("\\^|@", "");
            }
            else {
                this.name = "";
            }
        }

        public void setQuantity(int quantity) {
            this.quantity = quantity;
        }

        public int getQuantity() {
            return this.quantity;
        }

        String toQueryString() {
            return this.quantity + "@" + this.code + "@" + ACEParameterUtil.reduceTextCount(this.name);
        }

        @NonNull
        JSONObject toJsonToString() {
            JSONObject _json = new JSONObject();

            try {
                _json.put("name", name);
                _json.put("code", code);
                _json.put("quantity", quantity);
            }
            catch (JSONException e) {
                e.printStackTrace();
            }

            return _json;
        }

        @NonNull
        public String toOptionString() {
            return toJsonToString().toString();
        }
    }
}