package com.acecounter.android.acetm;

import android.app.Application;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.acecounter.android.acetm.common.config.ACEPublicStaticConfig;
import com.acecounter.android.acetm.common.logger.ACELog;
import com.acecounter.android.acetm.common.util.StringUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;

public final class AceConfiguration {
    private static final String TAG = AceConfiguration.class.getSimpleName();

    private Application _application;
    private boolean _debug;
    private boolean _enablePrivacyPolicy;
    private String _encryptKey;
    private boolean _gather;
    private String _key;

    private HashSet<String> _ignoreCaseActivityClassNameSet;
    private @ACEPublicStaticConfig.ACEPlatform int _platform = ACEPublicStaticConfig.ACEPlatform.ACPlus;

    private AceConfiguration(@NonNull AceConfigurationBuilder builder) {
        if (builder instanceof AceConfigurationBuilderForPlus) {
            AceConfigurationBuilderForPlus builderForPlus = (AceConfigurationBuilderForPlus)builder;
            init(builder._application,
                    builderForPlus.getKey(),
                    false,
                    builderForPlus._encryptKey,
                    builderForPlus.getDebug(),
                    builderForPlus.getGather(),
                    builderForPlus._platform);
        }
        else if (builder instanceof AceConfigurationBuilderForOne) {
            AceConfigurationBuilderForOne builderForOne = (AceConfigurationBuilderForOne)builder;
            init(builder._application,
                    builderForOne.getKey(),
                    builderForOne._enablePrivacyPolicy,
                    null,
                    builderForOne.getDebug(),
                    builderForOne.getGather(),
                    builderForOne._platform);
        }

        if (builder._ignoreCaseActivityClassNameSet.size() > 0) {
            _ignoreCaseActivityClassNameSet.addAll(builder._ignoreCaseActivityClassNameSet);
        }
    }

    private void init(
            @NonNull Application application,
            @Nullable String key,
            boolean enablePrivacyPolicy,
            @Nullable String encryptKey,
            boolean debug,
            boolean gather,
            @ACEPublicStaticConfig.ACEPlatform int platform) {
        _application = application;
        _key = key;
        _enablePrivacyPolicy = enablePrivacyPolicy;
        if (!StringUtils.isNull(encryptKey)) {
            _encryptKey = encryptKey;
        }
        _debug = debug;
        _gather = gather;
        _ignoreCaseActivityClassNameSet = new HashSet<>();
        _platform = platform;

        if (TextUtils.isEmpty(_key)) {
            ACELog.w(TAG, "not found AceCounter key.");
        }
    }

    @NonNull
    public Application getApplication() {
        return _application;
    }

    @Nullable
    public String getEncryptKey() {
        return _encryptKey;
    }

    @NonNull
    public HashSet<String> getIgnoreCase() {
        return _ignoreCaseActivityClassNameSet;
    }

    public @ACEPublicStaticConfig.ACEPlatform int getPlatform() {
        return this._platform;
    }

    public boolean getDebug() {
        return _debug;
    }

    public boolean getEnablePrivacyPolicy() {
        return _enablePrivacyPolicy;
    }

    public boolean getGather() {
        return _gather;
    }

    @NonNull
    public String getKey() {
        return _key;
    }

    //region JSON, toString
    @NonNull
    JSONObject getParamsToJSONobject() throws JSONException {
        JSONObject _json = new JSONObject();

        if (getPlatform() == ACEPublicStaticConfig.ACEPlatform.ACPlus) {
            _json.put("encryptKey", getEncryptKey());
        }
        else if (getPlatform() == ACEPublicStaticConfig.ACEPlatform.ACOne) {
            _json.put("enablePrivacyPolicy", getEnablePrivacyPolicy());
        }
        _json.put("key", getKey());
        _json.put("debug", getDebug());
        _json.put("gather", getGather());
        _json.put("activities", _ignoreCaseActivityClassNameSet);

        return _json;
    }

    @NonNull
    @Override
    public String toString() {
        try {
            return getParamsToJSONobject().toString(2);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return super.toString();
    }
    //endregion JSON, toString

    public static class AceConfigurationBuilderForPlus extends AceConfigurationBuilder {
        @Nullable
        private String _encryptKey;

        public AceConfigurationBuilderForPlus(@NonNull Application application) {
            this._application = application;
            this._ignoreCaseActivityClassNameSet = new HashSet<>();
            this._platform = ACEPublicStaticConfig.ACEPlatform.ACPlus;
            this._encryptKey = null;
        }

        public AceConfigurationBuilderForPlus addActivity(Class<?>... activities) {
            for (Class c : activities) {
                String resultActivityName = c.getSimpleName();
                if (!TextUtils.isEmpty(resultActivityName)) {
                    this._ignoreCaseActivityClassNameSet.add(c.getSimpleName());
                }
            }
            return this;
        }

        public AceConfigurationBuilderForPlus setDebug(boolean value) {
            super.setDebugAtParent(value);
            return this;
        }

        @Nullable
        public String getEncryptKey() {
            return this._encryptKey;
        }

        public AceConfigurationBuilderForPlus setEncryptKey(@Nullable String value) {
            if (!TextUtils.isEmpty(value)) {
                this._encryptKey = value;
            }

            return this;
        }

        public AceConfigurationBuilderForPlus setGather(boolean value) {
            super.setGatherAtParent(value);
            return this;
        }

        public AceConfigurationBuilderForPlus setKey(@Nullable String value) {
            super.setKeyAtParent(value);
            return this;
        }

        // region override methods
        public AceConfiguration build() {
            return new AceConfiguration(this);
        }
        // endregion
    }

    public static class AceConfigurationBuilderForOne extends AceConfigurationBuilder {
        private boolean _enablePrivacyPolicy;

        public AceConfigurationBuilderForOne(@NonNull Application application) {
            this._application = application;
            this._ignoreCaseActivityClassNameSet = new HashSet<>();
            this._platform = ACEPublicStaticConfig.ACEPlatform.ACOne;
            this._enablePrivacyPolicy = false;
        }

        public AceConfigurationBuilderForOne setDebug(boolean value) {
            super.setDebugAtParent(value);
            return this;
        }

        public boolean getEnablePrivacyPolicy() {
            return this._enablePrivacyPolicy;
        }

        public AceConfigurationBuilderForOne setEnablePrivacyPolicy(boolean value) {
            this._enablePrivacyPolicy = value;
            return this;
        }

        public AceConfigurationBuilderForOne setGather(boolean value) {
            super.setGatherAtParent(value);
            return this;
        }

        public AceConfigurationBuilderForOne setKey(@NonNull String value) {
            super.setKeyAtParent(value);
            return this;
        }

        // region override methods
        public AceConfiguration build() {
            return new AceConfiguration(this);
        }
        // endregion
    }
}
