package co.ronash.pushe.analytics.messages.upstream

import co.ronash.pushe.messaging.TypedUpstreamMessage
import co.ronash.pushe.analytics.session.SessionActivity
import co.ronash.pushe.analytics.session.SessionFragment
import co.ronash.pushe.messages.MessageType
import com.squareup.moshi.Json
import com.squareup.moshi.JsonClass

@JsonClass(generateAdapter = true)
class SessionInfoMessage (
    @Json(name = "id") val sessionId: String,
    @Json(name = "session_flow") val sessionFlow: List<SessionActivityMessageWrapper>,
    @Json(name = "av_code") val appVersionCode: Long? = null
) : TypedUpstreamMessage<SessionInfoMessage>(
    MessageType.Analytics.Upstream.SESSION_INFO,
    { SessionInfoMessageJsonAdapter(it) })

/**
 * The SessionFlow units, [SessionActivity] & [SessionFragment], have a dynamic startTime field that
 * is updated on every start of a layout and should not be in the sessionInfoMessage.
 * So They can not be used as sessionInfoMessage units and there is a need for unit wrappers
 * [SessionActivityMessageWrapper] and [SessionFragmentMessageWrapper].
 *
 */
@JsonClass(generateAdapter = true)
class SessionActivityMessageWrapper (
    @Json(name = "name") val name: String,
    @Json(name = "start_time") var startTime: Long,
    @Json(name = "duration") var duration: Long,
    @Json(name = "fragment_flows") val fragmentFlows: MutableMap<String, MutableList<SessionFragmentMessageWrapper>> = mutableMapOf(),
    @Json(name = "src_notif") var sourceNotifMessageId: String? = null
)

@JsonClass(generateAdapter = true)
class SessionFragmentMessageWrapper(
    @Json(name = "name") val name: String,
    @Json(name = "start_time") var startTime: Long,
    @Json(name = "duration") var duration: Long,
    @Json(name = "fragment_flows") val fragmentFlows: MutableMap<String, MutableList<SessionFragmentMessageWrapper>> = mutableMapOf()
)

/**
 * A singleton object with the only functionality of building SessionFlow message wrappers and
 * sessionInfoMessage object from sessionFlow units [SessionActivity] & [SessionFragment]
 */
object SessionInfoMessageBuilder {
    fun build(sessionId: String, sessionFlow: List<SessionActivity>, appVersionCode: Long?): SessionInfoMessage {
        val messageData: MutableList<SessionActivityMessageWrapper> = mutableListOf()
        var sessionActivityMessageWrapper: SessionActivityMessageWrapper
        for (sessionActivity in sessionFlow) {
            sessionActivityMessageWrapper = SessionActivityMessageWrapper(
                sessionActivity.name,
                sessionActivity.originalStartTime,
                sessionActivity.duration,
                mutableMapOf(),
                sessionActivity.sourceNotifMessageId
            )
            val fragmentFlows = sessionActivity.fragmentFlows
            if (fragmentFlows.isNotEmpty()){
                for (fragmentFlow in fragmentFlows){
                    sessionActivityMessageWrapper.fragmentFlows[fragmentFlow.key] =
                            getSessionFragmentWrappers(fragmentFlow.value)
                }
            }
            messageData.add(sessionActivityMessageWrapper)
        }
        return SessionInfoMessage(sessionId, messageData, appVersionCode)
    }

    private fun getSessionFragmentWrappers(sessionFragments: MutableList<SessionFragment>): MutableList<SessionFragmentMessageWrapper> {
        val fragmentWrappers: MutableList<SessionFragmentMessageWrapper> = mutableListOf()
        var sessionFragmentWrapper: SessionFragmentMessageWrapper
        for (sessionFragment in sessionFragments) {
            sessionFragmentWrapper = SessionFragmentMessageWrapper(
                sessionFragment.name,
                sessionFragment.originalStartTime,
                sessionFragment.duration,
                mutableMapOf()
            )
            val fragmentFlows = sessionFragment.fragmentFlows
            if (fragmentFlows.isNotEmpty()){
                for (fragmentFlow in fragmentFlows){
                    sessionFragmentWrapper.fragmentFlows[fragmentFlow.key] =
                            getSessionFragmentWrappers(fragmentFlow.value)
                }
            }
            fragmentWrappers.add(sessionFragmentWrapper)
        }
        return fragmentWrappers
    }
}
