/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.apm.servlet;

import co.elastic.apm.configuration.CoreConfiguration;
import co.elastic.apm.impl.ElasticApmTracer;
import co.elastic.apm.impl.context.Context;
import co.elastic.apm.impl.context.Request;
import co.elastic.apm.impl.context.Response;
import co.elastic.apm.impl.context.Url;
import co.elastic.apm.impl.transaction.Transaction;
import co.elastic.apm.matcher.WildcardMatcher;
import co.elastic.apm.shaded.slf4j.Logger;
import co.elastic.apm.shaded.slf4j.LoggerFactory;
import co.elastic.apm.util.PotentiallyMultiValuedMap;
import co.elastic.apm.web.ClientIpUtils;
import co.elastic.apm.web.ResultUtil;
import co.elastic.apm.web.WebConfiguration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import javax.annotation.Nullable;

public class ServletTransactionHelper {
    private final Logger logger = LoggerFactory.getLogger(ServletTransactionHelper.class);
    private final Set<String> METHODS_WITH_BODY = new HashSet<String>(Arrays.asList("POST", "PUT", "PATCH", "DELETE"));
    private final ElasticApmTracer tracer;
    private final CoreConfiguration coreConfiguration;
    private final WebConfiguration webConfiguration;

    ServletTransactionHelper(ElasticApmTracer tracer) {
        this.tracer = tracer;
        this.coreConfiguration = tracer.getConfig(CoreConfiguration.class);
        this.webConfiguration = tracer.getConfig(WebConfiguration.class);
    }

    @Nullable
    public Transaction onBefore(String servletPath, String pathInfo, String requestURI, @Nullable String userAgentHeader, @Nullable String traceContextHeader) {
        if (this.coreConfiguration.isActive() && this.tracer.currentTransaction() == null && !this.isExcluded(servletPath, pathInfo, requestURI, userAgentHeader)) {
            return this.tracer.startTransaction(traceContextHeader);
        }
        return null;
    }

    public void fillRequestContext(Transaction transaction, @Nullable String userName, String protocol, String method, boolean secure, String scheme, String serverName, int serverPort, String requestURI, String queryString, String remoteAddr, StringBuffer requestURL) {
        if (transaction.getName().length() == 0) {
            transaction.withName(method);
        }
        Context context = transaction.getContext();
        Request request = transaction.getContext().getRequest();
        this.fillRequest(request, protocol, method, secure, scheme, serverName, serverPort, requestURI, queryString, remoteAddr, requestURL);
        if (context.getUser().getUsername() == null) {
            context.getUser().withUsername(userName);
        }
    }

    public void onAfter(Transaction transaction, @Nullable Exception exception, boolean committed, int status, String method, Map<String, String[]> parameterMap) {
        try {
            this.fillRequestParameters(transaction, method, parameterMap);
            this.fillResponse(transaction.getContext().getResponse(), committed, status);
            transaction.withResult(ResultUtil.getResultByHttpStatus(status));
            transaction.withType("request");
            if (exception != null) {
                this.tracer.captureException(exception);
            }
        }
        catch (RuntimeException e) {
            this.logger.warn("Exception while capturing Elastic APM transaction", e);
        }
        transaction.end();
    }

    private void fillRequestParameters(Transaction transaction, String method, Map<String, String[]> parameterMap) {
        Request request = transaction.getContext().getRequest();
        if (this.hasBody(request.getHeaders(), method)) {
            if (this.webConfiguration.getCaptureBody() != WebConfiguration.EventType.OFF) {
                this.captureBody(request, parameterMap);
            } else {
                request.redactBody();
            }
        }
    }

    private boolean isExcluded(String servletPath, String pathInfo, String requestURI, @Nullable String userAgentHeader) {
        boolean excludeAgent;
        boolean excludeUrl = WildcardMatcher.anyMatch(this.webConfiguration.getIgnoreUrls(), servletPath, pathInfo);
        if (excludeUrl) {
            this.logger.debug("Not tracing this request as the URL {} is ignored by one of the matchers", (Object)requestURI, (Object)this.webConfiguration.getIgnoreUrls());
        }
        boolean bl = excludeAgent = userAgentHeader != null && WildcardMatcher.anyMatch(this.webConfiguration.getIgnoreUserAgents(), userAgentHeader);
        if (excludeAgent) {
            this.logger.debug("Not tracing this request as the User-Agent {} is ignored by one of the matchers", (Object)userAgentHeader, (Object)this.webConfiguration.getIgnoreUserAgents());
        }
        return excludeUrl || excludeAgent;
    }

    private void fillResponse(Response response, boolean committed, int status) {
        response.withFinished(true);
        response.withHeadersSent(committed);
        response.withStatusCode(status);
    }

    private void fillRequest(Request request, String protocol, String method, boolean secure, String scheme, String serverName, int serverPort, String requestURI, String queryString, String remoteAddr, StringBuffer requestURL) {
        request.withHttpVersion(this.getHttpVersion(protocol));
        request.withMethod(method);
        request.getSocket().withEncrypted(secure).withRemoteAddress(ClientIpUtils.getRealIp(request.getHeaders(), remoteAddr));
        request.getUrl().withProtocol(scheme).withHostname(serverName).withPort(serverPort).withPathname(requestURI).withSearch(queryString);
        this.fillFullUrl(request.getUrl(), queryString, requestURL);
    }

    private boolean hasBody(PotentiallyMultiValuedMap headers, String method) {
        return this.METHODS_WITH_BODY.contains(method) && headers.containsIgnoreCase("Content-Type");
    }

    private void captureBody(Request request, Map<String, String[]> params) {
        String contentTypeHeader = request.getHeaders().getFirst("Content-Type");
        if (contentTypeHeader != null && contentTypeHeader.startsWith("application/x-www-form-urlencoded")) {
            for (Map.Entry<String, String[]> param : params.entrySet()) {
                request.addFormUrlEncodedParameters(param.getKey(), param.getValue());
            }
        } else {
            request.redactBody();
        }
    }

    private void fillFullUrl(Url url, @Nullable String queryString, StringBuffer requestURL) {
        StringBuilder fullUrl = url.getFull();
        if (queryString != null) {
            fullUrl.ensureCapacity(requestURL.length() + 1 + queryString.length());
            fullUrl.append(requestURL).append('?').append(queryString);
        } else {
            fullUrl.ensureCapacity(requestURL.length());
            fullUrl.append(requestURL);
        }
    }

    private String getHttpVersion(String protocol) {
        switch (protocol) {
            case "HTTP/1.0": {
                return "1.0";
            }
            case "HTTP/1.1": {
                return "1.1";
            }
            case "HTTP/2.0": {
                return "2.0";
            }
        }
        return protocol.replace("HTTP/", "");
    }
}

