/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.apm.agent.profiler;

import co.elastic.apm.agent.configuration.CoreConfiguration;
import co.elastic.apm.agent.configuration.converter.TimeDuration;
import co.elastic.apm.agent.context.AbstractLifecycleListener;
import co.elastic.apm.agent.impl.ElasticApmTracer;
import co.elastic.apm.agent.impl.transaction.StackFrame;
import co.elastic.apm.agent.impl.transaction.TraceContext;
import co.elastic.apm.agent.matcher.WildcardMatcher;
import co.elastic.apm.agent.objectpool.Allocator;
import co.elastic.apm.agent.objectpool.ObjectPool;
import co.elastic.apm.agent.objectpool.impl.ListBasedObjectPool;
import co.elastic.apm.agent.profiler.CallTree;
import co.elastic.apm.agent.profiler.NanoClock;
import co.elastic.apm.agent.profiler.ProfilingConfiguration;
import co.elastic.apm.agent.profiler.ThreadMatcher;
import co.elastic.apm.agent.profiler.asyncprofiler.AsyncProfiler;
import co.elastic.apm.agent.profiler.asyncprofiler.JfrParser;
import co.elastic.apm.agent.profiler.collections.Long2ObjectHashMap;
import co.elastic.apm.agent.sdk.logging.Logger;
import co.elastic.apm.agent.sdk.logging.LoggerFactory;
import co.elastic.apm.agent.util.ExecutorUtils;
import com.lmax.disruptor.EventFactory;
import com.lmax.disruptor.EventPoller;
import com.lmax.disruptor.EventTranslatorTwoArg;
import com.lmax.disruptor.RingBuffer;
import com.lmax.disruptor.Sequence;
import com.lmax.disruptor.SequenceBarrier;
import com.lmax.disruptor.WaitStrategy;
import java.io.File;
import java.io.IOException;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.FileChannel;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import javax.annotation.Nullable;

public class SamplingProfiler
extends AbstractLifecycleListener
implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(SamplingProfiler.class);
    private static final int ACTIVATION_EVENTS_IN_FILE = 1000000;
    private static final int MAX_STACK_DEPTH = 256;
    private static final int PRE_ALLOCATE_ACTIVATION_EVENTS_FILE_MB = 10;
    private static final int MAX_ACTIVATION_EVENTS_FILE_SIZE = 106000000;
    private static final int ACTIVATION_EVENTS_BUFFER_SIZE = 434176;
    private final EventTranslatorTwoArg<ActivationEvent, TraceContext, TraceContext> ACTIVATION_EVENT_TRANSLATOR = new EventTranslatorTwoArg<ActivationEvent, TraceContext, TraceContext>(){

        @Override
        public void translateTo(ActivationEvent event, long sequence, TraceContext active, TraceContext previouslyActive) {
            event.activation(active, Thread.currentThread().getId(), previouslyActive, SamplingProfiler.this.nanoClock.nanoTime());
        }
    };
    private final EventTranslatorTwoArg<ActivationEvent, TraceContext, TraceContext> DEACTIVATION_EVENT_TRANSLATOR = new EventTranslatorTwoArg<ActivationEvent, TraceContext, TraceContext>(){

        @Override
        public void translateTo(ActivationEvent event, long sequence, TraceContext active, TraceContext previouslyActive) {
            event.deactivation(active, Thread.currentThread().getId(), previouslyActive, SamplingProfiler.this.nanoClock.nanoTime());
        }
    };
    static final int RING_BUFFER_SIZE = 4096;
    private final ProfilingConfiguration config;
    private final CoreConfiguration coreConfig;
    private final ScheduledExecutorService scheduler;
    private final Long2ObjectHashMap<CallTree.Root> profiledThreads = new Long2ObjectHashMap();
    private final RingBuffer<ActivationEvent> eventBuffer;
    private volatile boolean profilingSessionOngoing = false;
    private final Sequence sequence;
    private final ElasticApmTracer tracer;
    private final NanoClock nanoClock;
    private final ObjectPool<CallTree.Root> rootPool;
    private final ThreadMatcher threadMatcher = new ThreadMatcher();
    private final EventPoller<ActivationEvent> poller;
    @Nullable
    private File jfrFile;
    private boolean canDeleteJfrFile;
    private final WriteActivationEventToFileHandler writeActivationEventToFileHandler = new WriteActivationEventToFileHandler();
    @Nullable
    private JfrParser jfrParser;
    private volatile int profilingSessions;
    private final ByteBuffer activationEventsBuffer;
    @Nullable
    private File activationEventsFile;
    private boolean canDeleteActivationEventsFile;
    @Nullable
    private FileChannel activationEventsFileChannel;
    private final ObjectPool<CallTree> callTreePool;
    private final TraceContext contextForLogging;
    private boolean previouslyEnabled = false;

    public SamplingProfiler(ElasticApmTracer tracer, NanoClock nanoClock) {
        this(tracer, nanoClock, null, null);
    }

    public SamplingProfiler(final ElasticApmTracer tracer, NanoClock nanoClock, @Nullable File activationEventsFile, @Nullable File jfrFile) {
        this.tracer = tracer;
        this.config = tracer.getConfig(ProfilingConfiguration.class);
        this.coreConfig = tracer.getConfig(CoreConfiguration.class);
        this.scheduler = ExecutorUtils.createSingleThreadSchedulingDaemonPool("sampling-profiler");
        this.nanoClock = nanoClock;
        this.eventBuffer = this.createRingBuffer();
        this.sequence = new Sequence();
        this.eventBuffer.addGatingSequences(this.sequence);
        this.poller = this.eventBuffer.newPoller(new Sequence[0]);
        this.contextForLogging = TraceContext.with64BitId(tracer);
        this.callTreePool = ListBasedObjectPool.ofRecyclable(2048, new Allocator<CallTree>(){

            @Override
            public CallTree createInstance() {
                return new CallTree();
            }
        });
        this.rootPool = ListBasedObjectPool.ofRecyclable(512, new Allocator<CallTree.Root>(){

            @Override
            public CallTree.Root createInstance() {
                return new CallTree.Root(tracer);
            }
        });
        this.jfrFile = jfrFile;
        this.activationEventsBuffer = ByteBuffer.allocateDirect(434176);
        this.activationEventsFile = activationEventsFile;
    }

    private synchronized void createFilesIfRequired() throws IOException {
        if (this.jfrFile == null || !this.jfrFile.exists()) {
            this.jfrFile = File.createTempFile("apm-traces-", ".jfr");
            this.jfrFile.deleteOnExit();
            this.canDeleteJfrFile = true;
        }
        if (this.activationEventsFile == null || !this.activationEventsFile.exists()) {
            this.activationEventsFile = File.createTempFile("apm-activation-events-", ".bin");
            this.activationEventsFile.deleteOnExit();
            this.canDeleteActivationEventsFile = true;
        }
        if (this.activationEventsFileChannel == null || !this.activationEventsFileChannel.isOpen()) {
            this.activationEventsFileChannel = FileChannel.open(this.activationEventsFile.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE);
        }
        if (this.activationEventsFileChannel.size() == 0L) {
            SamplingProfiler.preAllocate(this.activationEventsFileChannel, 10);
        }
    }

    public void skipToEndOfActivationEventsFile() throws IOException {
        this.activationEventsFileChannel.position(this.activationEventsFileChannel.size());
    }

    private static void preAllocate(FileChannel channel, int mb) throws IOException {
        long initialPos = channel.position();
        ByteBuffer oneKb = ByteBuffer.allocate(1024);
        for (int i = 0; i < mb * 1024; ++i) {
            channel.write(oneKb);
            ((Buffer)oneKb).clear();
        }
        channel.position(initialPos);
    }

    private RingBuffer<ActivationEvent> createRingBuffer() {
        return RingBuffer.createMultiProducer(new EventFactory<ActivationEvent>(){

            @Override
            public ActivationEvent newInstance() {
                return new ActivationEvent();
            }
        }, 4096, new NoWaitStrategy());
    }

    public boolean onActivation(TraceContext activeSpan, @Nullable TraceContext previouslyActive) {
        if (this.profilingSessionOngoing) {
            boolean success;
            if (previouslyActive == null) {
                AsyncProfiler.getInstance(this.config.getProfilerLibDirectory(), this.config.getAsyncProfilerSafeMode()).enableProfilingCurrentThread();
            }
            if (!(success = this.eventBuffer.tryPublishEvent(this.ACTIVATION_EVENT_TRANSLATOR, activeSpan, previouslyActive)) && logger.isDebugEnabled()) {
                logger.debug("Could not add activation event to ring buffer as no slots are available");
            }
            return success;
        }
        return false;
    }

    public boolean onDeactivation(TraceContext activeSpan, @Nullable TraceContext previouslyActive) {
        if (this.profilingSessionOngoing) {
            boolean success;
            if (previouslyActive == null) {
                AsyncProfiler.getInstance(this.config.getProfilerLibDirectory(), this.config.getAsyncProfilerSafeMode()).disableProfilingCurrentThread();
            }
            if (!(success = this.eventBuffer.tryPublishEvent(this.DEACTIVATION_EVENT_TRANSLATOR, activeSpan, previouslyActive)) && logger.isDebugEnabled()) {
                logger.debug("Could not add deactivation event to ring buffer as no slots are available");
            }
            return success;
        }
        return false;
    }

    @Override
    public void run() {
        boolean enabled = this.config.isProfilingEnabled() && this.tracer.isRunning();
        boolean hasBeenDisabled = this.previouslyEnabled && !enabled;
        this.previouslyEnabled = enabled;
        if (!enabled) {
            if (this.jfrParser != null) {
                this.jfrParser = null;
            }
            if (!this.scheduler.isShutdown()) {
                this.scheduler.schedule(this, this.config.getProfilingInterval().getMillis(), TimeUnit.MILLISECONDS);
            }
            if (hasBeenDisabled) {
                try {
                    this.clear();
                }
                catch (Throwable throwable) {
                    logger.error("Error while trying to clear profiler constructs", throwable);
                }
            }
            return;
        }
        try {
            this.createFilesIfRequired();
        }
        catch (IOException e) {
            logger.error("unable to initialize profiling files", e);
            return;
        }
        TimeDuration sampleRate = this.config.getSamplingInterval();
        TimeDuration profilingDuration = this.config.getProfilingDuration();
        boolean postProcessingEnabled = this.config.isPostProcessingEnabled();
        this.setProfilingSessionOngoing(postProcessingEnabled);
        if (postProcessingEnabled) {
            logger.debug("Start full profiling session (async-profiler and agent processing)");
        } else {
            logger.debug("Start async-profiler profiling session");
        }
        try {
            this.profile(sampleRate, profilingDuration);
        }
        catch (Throwable t) {
            this.setProfilingSessionOngoing(false);
            logger.error("Stopping profiler", t);
            return;
        }
        logger.debug("End profiling session");
        boolean interrupted = Thread.currentThread().isInterrupted();
        boolean continueProfilingSession = this.config.isNonStopProfiling() && !interrupted && this.config.isProfilingEnabled() && postProcessingEnabled;
        this.setProfilingSessionOngoing(continueProfilingSession);
        if (!interrupted && !this.scheduler.isShutdown()) {
            long delay = this.config.getProfilingInterval().getMillis() - profilingDuration.getMillis();
            this.scheduler.schedule(this, delay, TimeUnit.MILLISECONDS);
        }
    }

    private void profile(TimeDuration sampleRate, TimeDuration profilingDuration) throws Exception {
        AsyncProfiler asyncProfiler = AsyncProfiler.getInstance(this.config.getProfilerLibDirectory(), this.config.getAsyncProfilerSafeMode());
        try {
            String startCommand = "start,jfr,event=wall,cstack=n,interval=" + sampleRate.getMillis() + "ms,filter,file=" + this.jfrFile + ",safemode=" + this.config.getAsyncProfilerSafeMode();
            String startMessage = asyncProfiler.execute(startCommand);
            logger.debug(startMessage);
            if (!this.profiledThreads.isEmpty()) {
                this.restoreFilterState(asyncProfiler);
            }
            ++this.profilingSessions;
            this.consumeActivationEventsFromRingBufferAndWriteToFile(profilingDuration);
            String stopMessage = asyncProfiler.execute("stop");
            logger.debug(stopMessage);
            this.processTraces();
        }
        catch (InterruptedException | ClosedByInterruptException e) {
            try {
                asyncProfiler.stop();
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
            Thread.currentThread().interrupt();
        }
    }

    private void restoreFilterState(AsyncProfiler asyncProfiler) {
        this.threadMatcher.forEachThread(new ThreadMatcher.NonCapturingPredicate<Thread, Long2ObjectHashMap.KeySet>(){

            @Override
            public boolean test(Thread thread, Long2ObjectHashMap.KeySet profiledThreads) {
                return profiledThreads.contains(thread.getId());
            }
        }, this.profiledThreads.keySet(), new ThreadMatcher.NonCapturingConsumer<Thread, AsyncProfiler>(){

            @Override
            public void accept(Thread thread, AsyncProfiler asyncProfiler) {
                asyncProfiler.enableProfilingThread(thread);
            }
        }, asyncProfiler);
    }

    private void consumeActivationEventsFromRingBufferAndWriteToFile(TimeDuration profilingDuration) throws Exception {
        this.resetActivationEventBuffer();
        long threshold = System.currentTimeMillis() + profilingDuration.getMillis();
        long initialSleep = 100000L;
        long maxSleep = 10000000L;
        long sleep = initialSleep;
        while (System.currentTimeMillis() < threshold && !Thread.currentThread().isInterrupted()) {
            if (this.activationEventsFileChannel.position() < 106000000L) {
                EventPoller.PollState poll = this.consumeActivationEventsFromRingBufferAndWriteToFile();
                if (poll == EventPoller.PollState.PROCESSING) {
                    sleep = initialSleep;
                    continue;
                }
                if (sleep < maxSleep) {
                    sleep *= 2L;
                }
                LockSupport.parkNanos(sleep);
                continue;
            }
            logger.warn("The activation events file is full. Try lowering the profiling_duration.");
            Thread.sleep(Math.max(0L, threshold - System.currentTimeMillis()));
        }
    }

    EventPoller.PollState consumeActivationEventsFromRingBufferAndWriteToFile() throws Exception {
        this.createFilesIfRequired();
        return this.poller.poll(this.writeActivationEventToFileHandler);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void processTraces() throws IOException {
        if (this.jfrParser == null) {
            this.jfrParser = new JfrParser();
        }
        if (Thread.currentThread().isInterrupted()) {
            return;
        }
        this.createFilesIfRequired();
        long eof = this.startProcessingActivationEventsFile();
        if (eof == 0L && this.activationEventsBuffer.limit() == 0 && this.profiledThreads.isEmpty()) {
            logger.debug("No activation events during this period. Skip processing stack traces.");
            return;
        }
        long start = System.nanoTime();
        List<WildcardMatcher> excludedClasses = this.config.getExcludedClasses();
        List<WildcardMatcher> includedClasses = this.config.getIncludedClasses();
        if (this.config.isBackupDiagnosticFiles()) {
            this.backupDiagnosticFiles(eof);
        }
        try {
            this.jfrParser.parse(this.jfrFile, excludedClasses, includedClasses);
            List<StackTraceEvent> stackTraceEvents = this.getSortedStackTraceEvents(this.jfrParser);
            if (logger.isDebugEnabled()) {
                logger.debug("Processing {} stack traces", (Object)stackTraceEvents.size());
            }
            ArrayList<StackFrame> stackFrames = new ArrayList<StackFrame>();
            ElasticApmTracer tracer = this.tracer;
            ActivationEvent event = new ActivationEvent();
            long inferredSpansMinDuration = this.getInferredSpansMinDurationNs();
            for (StackTraceEvent stackTrace : stackTraceEvents) {
                this.processActivationEventsUpTo(stackTrace.nanoTime, event, eof);
                CallTree.Root root = this.profiledThreads.get(stackTrace.threadId);
                if (root != null) {
                    this.jfrParser.resolveStackTrace(stackTrace.stackTraceId, true, stackFrames, 256);
                    if (stackFrames.size() == 256) {
                        logger.debug("Max stack depth reached. Set profiling_included_classes or profiling_excluded_classes.");
                    }
                    if (!stackFrames.isEmpty()) {
                        try {
                            root.addStackTrace(tracer, stackFrames, stackTrace.nanoTime, this.callTreePool, inferredSpansMinDuration);
                        }
                        catch (Exception e) {
                            logger.warn("Removing call tree for thread {} because of exception while adding a stack trace: {} {}", stackTrace.threadId, e.getClass(), e.getMessage());
                            logger.debug(e.getMessage(), e);
                            this.profiledThreads.remove(stackTrace.threadId);
                        }
                    }
                }
                stackFrames.clear();
            }
            this.processActivationEventsUpTo(System.nanoTime(), event, eof);
        }
        finally {
            if (logger.isDebugEnabled()) {
                logger.debug("Processing traces took {}\u00b5s", (Object)((System.nanoTime() - start) / 1000L));
            }
            this.jfrParser.resetState();
            this.resetActivationEventBuffer();
        }
    }

    private void backupDiagnosticFiles(long eof) throws IOException {
        String now = String.format("%tFT%<tT.%<tL", new Date());
        Path profilerDir = Paths.get(System.getProperty("java.io.tmpdir"), "profiler");
        profilerDir.toFile().mkdir();
        try (FileChannel activationsFile = FileChannel.open(profilerDir.resolve(now + "-activations.dat"), StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);){
            if (eof > 0L) {
                this.activationEventsFileChannel.transferTo(0L, eof, activationsFile);
            } else {
                int position = this.activationEventsBuffer.position();
                activationsFile.write(this.activationEventsBuffer);
                this.activationEventsBuffer.position(position);
            }
        }
        Files.copy(this.jfrFile.toPath(), profilerDir.resolve(now + "-traces.jfr"), new CopyOption[0]);
    }

    private long getInferredSpansMinDurationNs() {
        return Math.max(this.config.getInferredSpansMinDuration().getMillis(), this.coreConfig.getSpanMinDuration().getMillis()) * 1000000L;
    }

    private List<StackTraceEvent> getSortedStackTraceEvents(JfrParser jfrParser) throws IOException {
        final ArrayList<StackTraceEvent> stackTraceEvents = new ArrayList<StackTraceEvent>();
        jfrParser.consumeStackTraces(new JfrParser.StackTraceConsumer(){

            @Override
            public void onCallTree(long threadId, long stackTraceId, long nanoTime) {
                stackTraceEvents.add(new StackTraceEvent(nanoTime, stackTraceId, threadId));
            }
        });
        Collections.sort(stackTraceEvents);
        return stackTraceEvents;
    }

    void processActivationEventsUpTo(long timestamp, long eof) throws IOException {
        this.processActivationEventsUpTo(timestamp, new ActivationEvent(), eof);
    }

    public void processActivationEventsUpTo(long timestamp, ActivationEvent event, long eof) throws IOException {
        FileChannel activationEventsFileChannel = this.activationEventsFileChannel;
        ByteBuffer buf = this.activationEventsBuffer;
        long previousTimestamp = 0L;
        while (buf.hasRemaining() || activationEventsFileChannel.position() < eof) {
            long eventTimestamp;
            if (!buf.hasRemaining()) {
                this.readActivationEventsToBuffer(activationEventsFileChannel, eof, buf);
            }
            if ((eventTimestamp = SamplingProfiler.peekLong(buf)) < previousTimestamp && logger.isDebugEnabled()) {
                logger.debug("Timestamp of current activation event ({}) is lower than the one from the previous event ({})", (Object)eventTimestamp, (Object)previousTimestamp);
            }
            previousTimestamp = eventTimestamp;
            if (eventTimestamp <= timestamp) {
                event.deserialize(buf);
                try {
                    event.handle(this);
                }
                catch (Exception e) {
                    logger.warn("Removing call tree for thread {} because of exception while handling activation event: {} {}", event.threadId, e.getClass(), e.getMessage());
                    logger.debug(e.getMessage(), e);
                    this.profiledThreads.remove(event.threadId);
                }
                continue;
            }
            return;
        }
    }

    private void readActivationEventsToBuffer(FileChannel activationEventsFileChannel, long eof, ByteBuffer byteBuffer) throws IOException {
        ByteBuffer buf = byteBuffer;
        ((Buffer)buf).clear();
        long remaining = eof - activationEventsFileChannel.position();
        activationEventsFileChannel.read(byteBuffer);
        ((Buffer)buf).flip();
        if (remaining < (long)buf.capacity()) {
            ((Buffer)buf).limit((int)remaining);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static long peekLong(ByteBuffer buf) {
        int pos = buf.position();
        try {
            long l = buf.getLong();
            return l;
        }
        finally {
            ((Buffer)buf).position(pos);
        }
    }

    public void resetActivationEventBuffer() throws IOException {
        ((Buffer)this.activationEventsBuffer).clear();
        if (this.activationEventsFileChannel != null && this.activationEventsFileChannel.isOpen()) {
            this.activationEventsFileChannel.position(0L);
        }
    }

    private void flushActivationEvents() throws IOException {
        if (this.activationEventsBuffer.position() > 0) {
            ((Buffer)this.activationEventsBuffer).flip();
            this.activationEventsFileChannel.write(this.activationEventsBuffer);
            ((Buffer)this.activationEventsBuffer).clear();
        }
    }

    long startProcessingActivationEventsFile() throws IOException {
        ByteBuffer activationEventsBuffer = this.activationEventsBuffer;
        if (this.activationEventsFileChannel.position() > 0L) {
            this.flushActivationEvents();
            ((Buffer)activationEventsBuffer).limit(0);
        } else {
            ((Buffer)activationEventsBuffer).flip();
        }
        long eof = this.activationEventsFileChannel.position();
        this.activationEventsFileChannel.position(0L);
        return eof;
    }

    void copyFromFiles(Path activationEvents, Path traces) throws IOException {
        this.createFilesIfRequired();
        FileChannel otherActivationsChannel = FileChannel.open(activationEvents, StandardOpenOption.READ);
        this.activationEventsFileChannel.transferFrom(otherActivationsChannel, 0L, otherActivationsChannel.size());
        this.activationEventsFileChannel.position(otherActivationsChannel.size());
        FileChannel otherTracesChannel = FileChannel.open(traces, StandardOpenOption.READ);
        FileChannel.open(this.jfrFile.toPath(), StandardOpenOption.WRITE).transferFrom(otherTracesChannel, 0L, otherTracesChannel.size());
    }

    @Override
    public void start(ElasticApmTracer tracer) {
        this.scheduler.submit(this);
    }

    @Override
    public void stop() throws Exception {
        ExecutorUtils.shutdownAndWaitTermination(this.scheduler);
        if (this.activationEventsFileChannel != null) {
            this.activationEventsFileChannel.close();
        }
        if (this.jfrFile != null && this.canDeleteJfrFile) {
            this.jfrFile.delete();
        }
        if (this.activationEventsFile != null && this.canDeleteActivationEventsFile) {
            this.activationEventsFile.delete();
        }
    }

    void setProfilingSessionOngoing(boolean profilingSessionOngoing) {
        this.profilingSessionOngoing = profilingSessionOngoing;
        if (!profilingSessionOngoing) {
            this.clearProfiledThreads();
        } else if (!this.profiledThreads.isEmpty() && logger.isDebugEnabled()) {
            logger.debug("Retaining {} call tree roots", (Object)this.profiledThreads.size());
        }
    }

    public void clearProfiledThreads() {
        for (CallTree.Root root : this.profiledThreads.values()) {
            root.recycle(this.callTreePool, this.rootPool);
        }
        this.profiledThreads.clear();
    }

    CallTree.Root getRoot() {
        return this.profiledThreads.get(Thread.currentThread().getId());
    }

    void clear() throws IOException {
        try {
            this.poller.poll(new EventPoller.Handler<ActivationEvent>(){

                @Override
                public boolean onEvent(ActivationEvent event, long sequence, boolean endOfBatch) {
                    SamplingProfiler.this.sequence.set(sequence);
                    return true;
                }
            });
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.resetActivationEventBuffer();
        this.profiledThreads.clear();
        this.callTreePool.clear();
        this.rootPool.clear();
    }

    int getProfilingSessions() {
        return this.profilingSessions;
    }

    private class WriteActivationEventToFileHandler
    implements EventPoller.Handler<ActivationEvent> {
        private WriteActivationEventToFileHandler() {
        }

        @Override
        public boolean onEvent(ActivationEvent event, long sequence, boolean endOfBatch) throws IOException {
            if (endOfBatch) {
                SamplingProfiler.this.sequence.set(sequence);
            }
            if (SamplingProfiler.this.activationEventsFileChannel.size() < 106000000L) {
                event.serialize(SamplingProfiler.this.activationEventsBuffer);
                if (!SamplingProfiler.this.activationEventsBuffer.hasRemaining()) {
                    SamplingProfiler.this.flushActivationEvents();
                }
                return true;
            }
            return false;
        }
    }

    private static class NoWaitStrategy
    implements WaitStrategy {
        private NoWaitStrategy() {
        }

        @Override
        public long waitFor(long sequence, Sequence cursor, Sequence dependentSequence, SequenceBarrier barrier) {
            return dependentSequence.get();
        }

        @Override
        public void signalAllWhenBlocking() {
        }
    }

    private static class ActivationEvent {
        public static final int SERIALIZED_SIZE = 106;
        private static final Map<String, Short> serviceNameMap = new HashMap<String, Short>();
        private static final Map<Short, String> serviceNameBackMap = new HashMap<Short, String>();
        private static final Map<String, Short> serviceVersionMap = new HashMap<String, Short>();
        private static final Map<Short, String> serviceVersionBackMap = new HashMap<Short, String>();
        private long timestamp;
        @Nullable
        private String serviceName;
        @Nullable
        private String serviceVersion;
        private byte[] traceContextBuffer = new byte[42];
        private byte[] previousContextBuffer = new byte[42];
        private boolean rootContext;
        private long threadId;
        private boolean activation;

        private ActivationEvent() {
        }

        public void activation(TraceContext context, long threadId, @Nullable TraceContext previousContext, long nanoTime) {
            this.set(context, threadId, true, previousContext != null ? previousContext : null, nanoTime);
        }

        public void deactivation(TraceContext context, long threadId, @Nullable TraceContext previousContext, long nanoTime) {
            this.set(context, threadId, false, previousContext != null ? previousContext : null, nanoTime);
        }

        private void set(TraceContext traceContext, long threadId, boolean activation, @Nullable TraceContext previousContext, long nanoTime) {
            traceContext.serialize(this.traceContextBuffer);
            this.threadId = threadId;
            this.activation = activation;
            this.serviceName = traceContext.getServiceName();
            this.serviceVersion = traceContext.getServiceVersion();
            if (previousContext != null) {
                previousContext.serialize(this.previousContextBuffer);
                this.rootContext = false;
            } else {
                this.rootContext = true;
            }
            this.timestamp = nanoTime;
        }

        public void handle(SamplingProfiler samplingProfiler) {
            if (logger.isDebugEnabled()) {
                logger.debug("Handling event timestamp={} root={} threadId={} activation={}", this.timestamp, this.rootContext, this.threadId, this.activation);
            }
            if (this.activation) {
                this.handleActivationEvent(samplingProfiler);
            } else {
                this.handleDeactivationEvent(samplingProfiler);
            }
        }

        private void handleActivationEvent(SamplingProfiler samplingProfiler) {
            if (this.rootContext) {
                this.startProfiling(samplingProfiler);
            } else {
                CallTree.Root root = (CallTree.Root)samplingProfiler.profiledThreads.get(this.threadId);
                if (root != null) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Handling activation for thread {}", (Object)this.threadId);
                    }
                    root.onActivation(this.traceContextBuffer, this.timestamp);
                } else if (logger.isDebugEnabled()) {
                    logger.debug("Illegal state when handling activation event for thread {}: no root found for this thread", (Object)this.threadId);
                }
            }
        }

        private void startProfiling(SamplingProfiler samplingProfiler) {
            CallTree.Root orphaned;
            CallTree.Root root = CallTree.createRoot(samplingProfiler.rootPool, this.traceContextBuffer, this.serviceName, this.serviceVersion, this.timestamp);
            if (logger.isDebugEnabled()) {
                logger.debug("Create call tree ({}) for thread {}", (Object)this.deserialize(samplingProfiler, this.traceContextBuffer), (Object)this.threadId);
            }
            if ((orphaned = samplingProfiler.profiledThreads.put(this.threadId, root)) != null) {
                if (logger.isDebugEnabled()) {
                    logger.warn("Illegal state when stopping profiling for thread {}: orphaned root", (Object)this.threadId);
                }
                orphaned.recycle(samplingProfiler.callTreePool, samplingProfiler.rootPool);
            }
        }

        private TraceContext deserialize(SamplingProfiler samplingProfiler, byte[] traceContextBuffer) {
            samplingProfiler.contextForLogging.deserialize(traceContextBuffer, null, null);
            return samplingProfiler.contextForLogging;
        }

        private void handleDeactivationEvent(SamplingProfiler samplingProfiler) {
            if (this.rootContext) {
                this.stopProfiling(samplingProfiler);
            } else {
                CallTree.Root root = (CallTree.Root)samplingProfiler.profiledThreads.get(this.threadId);
                if (root != null) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("Handling deactivation for thread {}", (Object)this.threadId);
                    }
                    root.onDeactivation(this.traceContextBuffer, this.previousContextBuffer, this.timestamp);
                } else if (logger.isDebugEnabled()) {
                    logger.debug("Illegal state when handling deactivation event for thread {}: no root found for this thread", (Object)this.threadId);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void stopProfiling(SamplingProfiler samplingProfiler) {
            CallTree.Root callTree = (CallTree.Root)samplingProfiler.profiledThreads.get(this.threadId);
            if (callTree != null && callTree.getRootContext().traceIdAndIdEquals(this.traceContextBuffer)) {
                if (logger.isDebugEnabled()) {
                    logger.debug("End call tree ({}) for thread {}", (Object)this.deserialize(samplingProfiler, this.traceContextBuffer), (Object)this.threadId);
                }
                samplingProfiler.profiledThreads.remove(this.threadId);
                try {
                    callTree.end(samplingProfiler.callTreePool, samplingProfiler.getInferredSpansMinDurationNs());
                    int createdSpans = callTree.spanify();
                    if (logger.isDebugEnabled()) {
                        if (createdSpans > 0) {
                            logger.debug("Created spans ({}) for thread {}", (Object)createdSpans, (Object)this.threadId);
                        } else {
                            logger.debug("Created no spans for thread {} (count={})", (Object)this.threadId, (Object)callTree.getCount());
                        }
                    }
                }
                finally {
                    callTree.recycle(samplingProfiler.callTreePool, samplingProfiler.rootPool);
                }
            }
        }

        public void serialize(ByteBuffer buf) {
            buf.putLong(this.timestamp);
            buf.putShort(this.getServiceNameIndex());
            buf.putShort(this.getServiceVersionIndex());
            buf.put(this.traceContextBuffer);
            buf.put(this.previousContextBuffer);
            buf.put(this.rootContext ? (byte)1 : 0);
            buf.putLong(this.threadId);
            buf.put(this.activation ? (byte)1 : 0);
        }

        public void deserialize(ByteBuffer buf) {
            this.timestamp = buf.getLong();
            this.serviceName = serviceNameBackMap.get(buf.getShort());
            this.serviceVersion = serviceVersionBackMap.get(buf.getShort());
            buf.get(this.traceContextBuffer);
            buf.get(this.previousContextBuffer);
            this.rootContext = buf.get() == 1;
            this.threadId = buf.getLong();
            this.activation = buf.get() == 1;
        }

        private short getServiceNameIndex() {
            Short index = serviceNameMap.get(this.serviceName);
            if (index == null) {
                index = (short)serviceNameMap.size();
                serviceNameMap.put(this.serviceName, index);
                serviceNameBackMap.put(index, this.serviceName);
            }
            return index;
        }

        private short getServiceVersionIndex() {
            Short index = serviceVersionMap.get(this.serviceVersion);
            if (index == null) {
                index = (short)serviceVersionMap.size();
                serviceVersionMap.put(this.serviceVersion, index);
                serviceVersionBackMap.put(index, this.serviceVersion);
            }
            return index;
        }
    }

    public static class StackTraceEvent
    implements Comparable<StackTraceEvent> {
        private final long nanoTime;
        private final long stackTraceId;
        private final long threadId;

        private StackTraceEvent(long nanoTime, long stackTraceId, long threadId) {
            this.nanoTime = nanoTime;
            this.stackTraceId = stackTraceId;
            this.threadId = threadId;
        }

        public long getThreadId() {
            return this.threadId;
        }

        public long getNanoTime() {
            return this.nanoTime;
        }

        public long getStackTraceId() {
            return this.stackTraceId;
        }

        @Override
        public int compareTo(StackTraceEvent o) {
            return Long.compare(this.nanoTime, o.nanoTime);
        }
    }
}

