/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.apm.agent.profiler.asyncprofiler;

import co.elastic.apm.agent.impl.transaction.StackFrame;
import co.elastic.apm.agent.matcher.WildcardMatcher;
import co.elastic.apm.agent.objectpool.Recyclable;
import co.elastic.apm.agent.profiler.asyncprofiler.BufferedFile;
import co.elastic.apm.agent.profiler.collections.Int2IntHashMap;
import co.elastic.apm.agent.profiler.collections.Int2ObjectHashMap;
import co.elastic.apm.agent.profiler.collections.Long2LongHashMap;
import co.elastic.apm.agent.profiler.collections.Long2ObjectHashMap;
import co.elastic.apm.agent.sdk.logging.Logger;
import co.elastic.apm.agent.sdk.logging.LoggerFactory;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import javax.annotation.Nullable;

public class JfrParser
implements Recyclable {
    private static final Logger logger = LoggerFactory.getLogger(JfrParser.class);
    private static final byte[] MAGIC_BYTES = new byte[]{70, 76, 82, 0};
    private static final Set<String> JAVA_FRAME_TYPES = new HashSet<String>(Arrays.asList("Interpreted", "JIT compiled", "Inlined"));
    private static final int BIG_FILE_BUFFER_SIZE = 0x500000;
    private static final int SMALL_FILE_BUFFER_SIZE = 4096;
    private static final String SYMBOL_EXCLUDED = "3x cluded";
    private static final String SYMBOL_NULL = "n u11";
    private static final StackFrame FRAME_EXCLUDED = new StackFrame("excluded", "excluded");
    private static final StackFrame FRAME_NULL = new StackFrame("null", "null");
    private final BufferedFile bufferedFile;
    private final Int2IntHashMap classIdToClassNameSymbolId = new Int2IntHashMap(-1);
    private final Int2IntHashMap symbolIdToPos = new Int2IntHashMap(-1);
    private final Int2ObjectHashMap<String> symbolIdToString = new Int2ObjectHashMap();
    private final Int2IntHashMap stackTraceIdToFilePositions = new Int2IntHashMap(-1);
    private final Long2LongHashMap nativeTidToJavaTid = new Long2LongHashMap(-1L);
    private final Long2ObjectHashMap<StackFrame> frameIdToFrame = new Long2ObjectHashMap();
    private final Long2LongHashMap frameIdToMethodSymbol = new Long2LongHashMap(-1L);
    private final Long2LongHashMap frameIdToClassId = new Long2LongHashMap(-1L);
    private final StringBuilder symbolBuilder = new StringBuilder();
    private long eventsOffset;
    private long metadataOffset;
    @Nullable
    private boolean[] isJavaFrameType;
    @Nullable
    private List<WildcardMatcher> excludedClasses;
    @Nullable
    private List<WildcardMatcher> includedClasses;

    public JfrParser() {
        this(ByteBuffer.allocateDirect(0x500000), ByteBuffer.allocateDirect(4096));
    }

    JfrParser(ByteBuffer bigBuffer, ByteBuffer smallBuffer) {
        this.bufferedFile = new BufferedFile(bigBuffer, smallBuffer);
    }

    public void parse(File file, List<WildcardMatcher> excludedClasses, List<WildcardMatcher> includedClasses) throws IOException {
        this.excludedClasses = excludedClasses;
        this.includedClasses = includedClasses;
        this.bufferedFile.setFile(file);
        long fileSize = this.bufferedFile.size();
        if (fileSize < 16L) {
            throw new IllegalStateException("Unexpected sampling profiler error, everything else should work as expected. Please report to us with as many details, including OS and JVM details.");
        }
        logger.debug("Parsing {} ({} bytes)", (Object)file, (Object)fileSize);
        this.bufferedFile.ensureRemaining(16, 16);
        for (byte magicByte : MAGIC_BYTES) {
            if (this.bufferedFile.get() == magicByte) continue;
            throw new IllegalArgumentException("Not a JFR file");
        }
        short major = this.bufferedFile.getShort();
        short minor = this.bufferedFile.getShort();
        if (major != 0 || minor != 9) {
            throw new IllegalArgumentException(String.format("Can only parse version 0.9. Was %d.%d", major, minor));
        }
        this.metadataOffset = this.bufferedFile.getLong();
        this.eventsOffset = this.bufferedFile.position();
        long checkpointOffset = this.parseMetadata(this.metadataOffset);
        this.parseCheckpoint(checkpointOffset);
    }

    private long parseMetadata(long metadataOffset) throws IOException {
        this.bufferedFile.position(metadataOffset);
        this.bufferedFile.ensureRemaining(8, 8);
        int size = this.bufferedFile.getInt();
        this.expectEventType(0);
        this.bufferedFile.skip(size - 16);
        this.bufferedFile.ensureRemaining(8, 8);
        return this.bufferedFile.getLong();
    }

    private void expectEventType(int expectedEventType) throws IOException {
        int eventType = this.bufferedFile.getInt();
        if (eventType != expectedEventType) {
            throw new IOException("Expected " + expectedEventType + " but got " + eventType);
        }
    }

    private void parseCheckpoint(long checkpointOffset) throws IOException {
        this.bufferedFile.position(checkpointOffset);
        int size = this.bufferedFile.getInt();
        this.expectEventType(1);
        this.bufferedFile.getLong();
        this.bufferedFile.getLong();
        while (this.bufferedFile.position() < this.metadataOffset) {
            this.parseContent();
        }
    }

    private void parseContent() throws IOException {
        BufferedFile bufferedFile = this.bufferedFile;
        int contentTypeId = bufferedFile.getInt();
        logger.debug("Parsing content type {}", (Object)contentTypeId);
        int count = bufferedFile.getInt();
        switch (contentTypeId) {
            case 7: {
                for (int i = 0; i < count; ++i) {
                    int threadId = bufferedFile.getInt();
                    String string = this.readUtf8String().toString();
                }
                break;
            }
            case 8: {
                for (int i = 0; i < count; ++i) {
                    bufferedFile.ensureRemaining(16);
                    long javaThreadId = bufferedFile.getUnsafeLong();
                    int nativeThreadId = bufferedFile.getUnsafeInt();
                    int threadGroup = bufferedFile.getUnsafeInt();
                    this.nativeTidToJavaTid.put(nativeThreadId, javaThreadId);
                }
                break;
            }
            case 31: {
                break;
            }
            case 9: {
                for (int i = 0; i < count; ++i) {
                    bufferedFile.ensureRemaining(13);
                    int pos = (int)bufferedFile.position();
                    int stackTraceKey = (int)bufferedFile.getUnsafeLong();
                    this.stackTraceIdToFilePositions.put(stackTraceKey, pos);
                    bufferedFile.getUnsafe();
                    int numFrames = bufferedFile.getUnsafeInt();
                    int sizeOfFrame = 13;
                    bufferedFile.skip(numFrames * sizeOfFrame);
                }
                break;
            }
            case 10: {
                for (int i = 0; i < count; ++i) {
                    bufferedFile.ensureRemaining(26);
                    int classId = (int)bufferedFile.getUnsafeLong();
                    bufferedFile.getUnsafeLong();
                    int classNameSymbolId = (int)bufferedFile.getUnsafeLong();
                    this.classIdToClassNameSymbolId.put(classId, classNameSymbolId);
                    bufferedFile.getUnsafeShort();
                }
                break;
            }
            case 32: {
                for (int i = 1; i <= count; ++i) {
                    bufferedFile.ensureRemaining(35);
                    long id = bufferedFile.getUnsafeLong();
                    int classId = (int)bufferedFile.getUnsafeLong();
                    int methodNameSymbolId = (int)bufferedFile.getUnsafeLong();
                    this.frameIdToFrame.put(id, FRAME_NULL);
                    this.frameIdToClassId.put(id, classId);
                    this.frameIdToMethodSymbol.put(id, methodNameSymbolId);
                    bufferedFile.getUnsafeLong();
                    bufferedFile.getUnsafeShort();
                    bufferedFile.getUnsafe();
                }
                break;
            }
            case 33: {
                for (int i = 0; i < count; ++i) {
                    int symbolId = (int)bufferedFile.getLong();
                    int pos = (int)bufferedFile.position();
                    this.symbolIdToPos.put(symbolId, pos);
                    this.symbolIdToString.put(symbolId, SYMBOL_NULL);
                    this.skipString();
                }
                break;
            }
            case 34: {
                for (int i = 1; i <= count; ++i) {
                    bufferedFile.getShort();
                    this.skipString();
                }
                break;
            }
            case 47: {
                this.isJavaFrameType = new boolean[count + 1];
                for (int i = 1; i <= count; ++i) {
                    short id = bufferedFile.get();
                    if (i != id) {
                        throw new IllegalStateException("Expecting ids to be incrementing");
                    }
                    this.isJavaFrameType[id] = JAVA_FRAME_TYPES.contains(this.readUtf8String().toString());
                }
                break;
            }
            default: {
                throw new IOException("Unknown content type " + contentTypeId);
            }
        }
    }

    private void skipString() throws IOException {
        int stringLength = this.bufferedFile.getUnsignedShort();
        this.bufferedFile.skip(stringLength);
    }

    public void consumeStackTraces(StackTraceConsumer callback) throws IOException {
        if (!this.bufferedFile.isSet()) {
            throw new IllegalStateException("consumeStackTraces was called before parse");
        }
        this.bufferedFile.position(this.eventsOffset);
        while (this.bufferedFile.position() < this.metadataOffset) {
            this.bufferedFile.ensureRemaining(30);
            int size = this.bufferedFile.getUnsafeInt();
            int eventType = this.bufferedFile.getUnsafeInt();
            if (eventType == 10) {
                return;
            }
            if (eventType != 20) {
                throw new IOException("Expected 20 but got " + eventType);
            }
            long nanoTime = this.bufferedFile.getUnsafeLong();
            int tid = this.bufferedFile.getUnsafeInt();
            long stackTraceId = this.bufferedFile.getUnsafeLong();
            short threadState = this.bufferedFile.getUnsafeShort();
            long javaThreadId = this.nativeTidToJavaTid.get(tid);
            if (javaThreadId == -1L) continue;
            callback.onCallTree(javaThreadId, stackTraceId, nanoTime);
        }
    }

    public void resolveStackTrace(long stackTraceId, boolean onlyJavaFrames, List<StackFrame> stackFrames, int maxStackDepth) throws IOException {
        if (!this.bufferedFile.isSet()) {
            throw new IllegalStateException("getStackTrace was called before parse");
        }
        long position = this.bufferedFile.position();
        this.bufferedFile.position(this.stackTraceIdToFilePositions.get((int)stackTraceId));
        this.bufferedFile.ensureRemaining(13);
        long stackTraceIdFromFile = this.bufferedFile.getUnsafeLong();
        assert (stackTraceId == stackTraceIdFromFile);
        this.bufferedFile.getUnsafe();
        int numFrames = this.bufferedFile.getUnsafeInt();
        for (int i = 0; i < numFrames; ++i) {
            this.bufferedFile.ensureRemaining(13);
            long frameId = this.bufferedFile.getUnsafeLong();
            this.bufferedFile.getUnsafeInt();
            byte frameType = this.bufferedFile.getUnsafe();
            this.addFrameIfIncluded(stackFrames, onlyJavaFrames, frameId, frameType);
            if (stackFrames.size() <= maxStackDepth) continue;
            stackFrames.remove(0);
        }
        this.bufferedFile.position(position);
    }

    private void addFrameIfIncluded(List<StackFrame> stackFrames, boolean onlyJavaFrames, long frameId, byte frameType) throws IOException {
        StackFrame stackFrame;
        if ((!onlyJavaFrames || this.isJavaFrameType(frameType)) && (stackFrame = this.resolveStackFrame(frameId)) != FRAME_EXCLUDED) {
            stackFrames.add(stackFrame);
        }
    }

    private boolean isJavaFrameType(byte frameType) {
        return this.isJavaFrameType[frameType];
    }

    private String resolveSymbol(int id, boolean classSymbol) throws IOException {
        String symbol = this.symbolIdToString.get(id);
        if (symbol != SYMBOL_NULL) {
            return symbol;
        }
        StringBuilder symbolBuilder = this.resolveSymbolBuilder(this.symbolIdToPos.get(id), classSymbol);
        symbol = classSymbol && !this.isClassIncluded(symbolBuilder) ? SYMBOL_EXCLUDED : symbolBuilder.toString();
        this.symbolIdToString.put(id, symbol);
        return symbol;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private StringBuilder resolveSymbolBuilder(int pos, boolean replaceSlashWithDot) throws IOException {
        long currentPos = this.bufferedFile.position();
        this.bufferedFile.position(pos);
        try {
            StringBuilder stringBuilder = this.readUtf8String(replaceSlashWithDot);
            return stringBuilder;
        }
        finally {
            this.bufferedFile.position(currentPos);
        }
    }

    private boolean isClassIncluded(CharSequence className) {
        return WildcardMatcher.isAnyMatch(this.includedClasses, className) && WildcardMatcher.isNoneMatch(this.excludedClasses, className);
    }

    private StackFrame resolveStackFrame(long frameId) throws IOException {
        StackFrame stackFrame = this.frameIdToFrame.get(frameId);
        if (stackFrame != FRAME_NULL) {
            return stackFrame;
        }
        String className = this.resolveSymbol(this.classIdToClassNameSymbolId.get((int)this.frameIdToClassId.get(frameId)), true);
        if (className == SYMBOL_EXCLUDED) {
            stackFrame = FRAME_EXCLUDED;
        } else {
            String method = this.resolveSymbol((int)this.frameIdToMethodSymbol.get(frameId), false);
            stackFrame = new StackFrame(className, Objects.requireNonNull(method));
        }
        this.frameIdToFrame.put(frameId, stackFrame);
        return stackFrame;
    }

    private StringBuilder readUtf8String() throws IOException {
        return this.readUtf8String(false);
    }

    private StringBuilder readUtf8String(boolean replaceSlashWithDot) throws IOException {
        int size = this.bufferedFile.getUnsignedShort();
        this.bufferedFile.ensureRemaining(size);
        StringBuilder symbolBuilder = this.symbolBuilder;
        symbolBuilder.setLength(0);
        for (int i = 0; i < size; ++i) {
            char c = (char)this.bufferedFile.getUnsafe();
            if (replaceSlashWithDot && c == '/') {
                symbolBuilder.append('.');
                continue;
            }
            symbolBuilder.append(c);
        }
        return symbolBuilder;
    }

    @Override
    public void resetState() {
        this.bufferedFile.resetState();
        this.eventsOffset = 0L;
        this.metadataOffset = 0L;
        this.isJavaFrameType = null;
        this.classIdToClassNameSymbolId.clear();
        this.stackTraceIdToFilePositions.clear();
        this.frameIdToFrame.clear();
        this.frameIdToMethodSymbol.clear();
        this.frameIdToClassId.clear();
        this.symbolBuilder.setLength(0);
        this.excludedClasses = null;
        this.includedClasses = null;
        this.symbolIdToPos.clear();
        this.symbolIdToString.clear();
    }

    private static interface ContentTypeId {
        public static final int CONTENT_THREAD = 7;
        public static final int CONTENT_JAVA_THREAD = 8;
        public static final int CONTENT_STACKTRACE = 9;
        public static final int CONTENT_CLASS = 10;
        public static final int CONTENT_THREAD_GROUP = 31;
        public static final int CONTENT_METHOD = 32;
        public static final int CONTENT_SYMBOL = 33;
        public static final int CONTENT_STATE = 34;
        public static final int CONTENT_FRAME_TYPE = 47;
    }

    private static interface EventTypeId {
        public static final int EVENT_METADATA = 0;
        public static final int EVENT_CHECKPOINT = 1;
        public static final int EVENT_RECORDING = 10;
        public static final int EVENT_EXECUTION_SAMPLE = 20;
    }

    public static interface StackTraceConsumer {
        public void onCallTree(long var1, long var3, long var5) throws IOException;
    }
}

