/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.apm.agent.jmx;

import co.elastic.apm.agent.context.AbstractLifecycleListener;
import co.elastic.apm.agent.impl.ElasticApmTracer;
import co.elastic.apm.agent.jmx.JmxConfiguration;
import co.elastic.apm.agent.jmx.JmxMetric;
import co.elastic.apm.agent.metrics.DoubleSupplier;
import co.elastic.apm.agent.metrics.Labels;
import co.elastic.apm.agent.metrics.MetricRegistry;
import co.elastic.apm.agent.sdk.logging.Logger;
import co.elastic.apm.agent.sdk.logging.LoggerFactory;
import co.elastic.apm.agent.util.GlobalLocks;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.MBeanServerDelegate;
import javax.management.MBeanServerFactory;
import javax.management.MBeanServerNotification;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;
import javax.management.relation.MBeanServerNotificationFilter;
import org.stagemonitor.configuration.ConfigurationOption;

public class JmxMetricTracker
extends AbstractLifecycleListener {
    private static final String JMX_PREFIX = "jvm.jmx.";
    private static final Logger logger = LoggerFactory.getLogger(JmxMetricTracker.class);
    @Nullable
    private volatile Thread logManagerPropertyPoller;
    @Nullable
    private volatile MBeanServer server;
    private final JmxConfiguration jmxConfiguration;
    private final MetricRegistry metricRegistry;
    @Nullable
    private volatile NotificationListener listener;

    public JmxMetricTracker(ElasticApmTracer tracer) {
        this.jmxConfiguration = tracer.getConfig(JmxConfiguration.class);
        this.metricRegistry = tracer.getMetricRegistry();
    }

    @Override
    public void start(ElasticApmTracer tracer) {
        ConfigurationOption.ChangeListener<List<JmxMetric>> initChangeListener = new ConfigurationOption.ChangeListener<List<JmxMetric>>(){

            @Override
            public void onChange(ConfigurationOption<?> configurationOption, List<JmxMetric> oldValue, List<JmxMetric> newValue) {
                if (oldValue.isEmpty() && !newValue.isEmpty()) {
                    JmxMetricTracker.this.tryInit();
                }
            }
        };
        this.jmxConfiguration.getCaptureJmxMetrics().addChangeListener(initChangeListener);
        if (!this.jmxConfiguration.getCaptureJmxMetrics().get().isEmpty()) {
            this.tryInit();
            this.jmxConfiguration.getCaptureJmxMetrics().removeChangeListener(initChangeListener);
        } else {
            logger.debug("Deferring initialization of JMX metric tracking until capture_jmx_metrics is set.");
        }
    }

    private synchronized void tryInit() {
        if (this.server != null || this.logManagerPropertyPoller != null) {
            return;
        }
        if (this.setCustomPlatformMBeanServer()) {
            ArrayList<MBeanServer> servers = MBeanServerFactory.findMBeanServer(null);
            if (!servers.isEmpty()) {
                this.init((MBeanServer)servers.get(0));
            } else {
                this.deferInit();
            }
        } else {
            this.init(this.getPlatformMBeanServerThreadSafely());
        }
    }

    private MBeanServer getPlatformMBeanServerThreadSafely() {
        GlobalLocks.JUL_INIT_LOCK.lock();
        try {
            MBeanServer mBeanServer = ManagementFactory.getPlatformMBeanServer();
            return mBeanServer;
        }
        finally {
            GlobalLocks.JUL_INIT_LOCK.unlock();
        }
    }

    private void deferInit() {
        logger.debug("Deferring initialization of JMX metric tracking until platform mbean server is initialized");
        Thread thread = new Thread(new Runnable(){
            private final long timeout = System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5L);

            @Override
            public void run() {
                while (!Thread.currentThread().isInterrupted() || this.timeout <= System.currentTimeMillis()) {
                    ArrayList<MBeanServer> servers = MBeanServerFactory.findMBeanServer(null);
                    if (!servers.isEmpty()) {
                        JmxMetricTracker.this.init((MBeanServer)servers.get(0));
                        return;
                    }
                    try {
                        Thread.sleep(500L);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        });
        thread.setName("elastic-apm-jmx-init");
        thread.setDaemon(true);
        thread.start();
        this.logManagerPropertyPoller = thread;
    }

    private boolean setCustomPlatformMBeanServer() {
        return ClassLoader.getSystemClassLoader().getResource("org/jboss/modules/Main.class") != null || System.getProperty("weblogic.Name") != null || System.getProperty("weblogic.home") != null;
    }

    synchronized void init(final MBeanServer platformMBeanServer) {
        if (this.server != null) {
            return;
        }
        logger.debug("Init JMX metric tracking with server {}", (Object)platformMBeanServer);
        this.server = platformMBeanServer;
        this.registerMBeanNotificationListener(platformMBeanServer);
        this.jmxConfiguration.getCaptureJmxMetrics().addChangeListener(new ConfigurationOption.ChangeListener<List<JmxMetric>>(){

            @Override
            public void onChange(ConfigurationOption<?> configurationOption, List<JmxMetric> oldValue, List<JmxMetric> newValue) {
                List oldRegistrations = JmxMetricTracker.this.compileJmxMetricRegistrations(oldValue, platformMBeanServer);
                List newRegistrations = JmxMetricTracker.this.compileJmxMetricRegistrations(newValue, platformMBeanServer);
                for (JmxMetricRegistration addedRegistration : JmxMetricTracker.removeAll(oldRegistrations, newRegistrations)) {
                    addedRegistration.register(platformMBeanServer, JmxMetricTracker.this.metricRegistry);
                }
                for (JmxMetricRegistration deletedRegistration : JmxMetricTracker.removeAll(newRegistrations, oldRegistrations)) {
                    deletedRegistration.unregister(JmxMetricTracker.this.metricRegistry);
                }
            }
        });
        this.register(this.jmxConfiguration.getCaptureJmxMetrics().get(), platformMBeanServer);
    }

    private void registerMBeanNotificationListener(final MBeanServer server) {
        MBeanServerNotificationFilter filter = new MBeanServerNotificationFilter();
        filter.enableAllObjectNames();
        filter.enableType("JMX.mbean.registered");
        this.listener = new NotificationListener(){

            @Override
            public void handleNotification(Notification notification, Object handback) {
                try {
                    if (notification instanceof MBeanServerNotification) {
                        this.onMBeanAdded(((MBeanServerNotification)notification).getMBeanName());
                    }
                }
                catch (Exception e) {
                    logger.error(e.getMessage(), e);
                }
            }

            private void onMBeanAdded(ObjectName mBeanName) {
                logger.trace("Receiving MBean registration notification for {}", (Object)mBeanName);
                for (JmxMetric jmxMetric : JmxMetricTracker.this.jmxConfiguration.getCaptureJmxMetrics().get()) {
                    ObjectName metricName = jmxMetric.getObjectName();
                    if (!metricName.apply(mBeanName) && !this.matchesJbossStatisticsPool(mBeanName, metricName, server)) continue;
                    logger.debug("MBean added at runtime: {}", (Object)jmxMetric.getObjectName());
                    JmxMetricTracker.this.register(Collections.singletonList(jmxMetric), server);
                }
            }

            private boolean matchesJbossStatisticsPool(ObjectName beanName, ObjectName metricName, MBeanServer server2) {
                String asDomain = "jboss.as";
                String exprDomain = "jboss.as.expr";
                if (!asDomain.equals(metricName.getDomain())) {
                    return false;
                }
                if (!exprDomain.equals(beanName.getDomain()) && !asDomain.equals(beanName.getDomain())) {
                    return false;
                }
                try {
                    ObjectName newName;
                    boolean matches;
                    Hashtable<String, String> metricProperties = metricName.getKeyPropertyList();
                    Hashtable<String, String> beanProperties = beanName.getKeyPropertyList();
                    if ("pool".equals(metricProperties.get("statistics")) && !beanProperties.containsKey("statistics")) {
                        beanProperties.put("statistics", "pool");
                    }
                    boolean bl = matches = metricName.apply(newName = new ObjectName(asDomain, beanProperties)) && server2.queryMBeans(newName, null).size() > 0;
                    if (matches) {
                        logger.debug("JBoss fallback detection applied for {} MBean metric registration", (Object)newName);
                    }
                    return matches;
                }
                catch (MalformedObjectNameException e) {
                    return false;
                }
            }
        };
        try {
            server.addNotificationListener(MBeanServerDelegate.DELEGATE_NAME, this.listener, (NotificationFilter)filter, null);
        }
        catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    private static <T> List<T> removeAll(List<T> removeFromThis, List<T> toRemove) {
        ArrayList<T> result = new ArrayList<T>(toRemove);
        result.removeAll(removeFromThis);
        return result;
    }

    private void register(List<JmxMetric> jmxMetrics, MBeanServer server) {
        for (JmxMetricRegistration registration : this.compileJmxMetricRegistrations(jmxMetrics, server)) {
            registration.register(server, this.metricRegistry);
        }
    }

    private List<JmxMetricRegistration> compileJmxMetricRegistrations(List<JmxMetric> jmxMetrics, MBeanServer server) {
        ArrayList<JmxMetricRegistration> registrations = new ArrayList<JmxMetricRegistration>();
        for (JmxMetric jmxMetric : jmxMetrics) {
            try {
                this.addJmxMetricRegistration(jmxMetric, registrations, server);
            }
            catch (Exception e) {
                logger.error("Failed to register JMX metric {}", (Object)jmxMetric.toString(), (Object)e);
            }
        }
        return registrations;
    }

    private void addJmxMetricRegistration(JmxMetric jmxMetric, List<JmxMetricRegistration> registrations, MBeanServer server) throws JMException {
        Set<ObjectInstance> mbeans = server.queryMBeans(jmxMetric.getObjectName(), null);
        if (!mbeans.isEmpty()) {
            logger.debug("Found mbeans for object name {}", (Object)jmxMetric.getObjectName());
        } else {
            logger.debug("Found no mbeans for object name {}. Listening for mbeans added later.", (Object)jmxMetric.getObjectName());
        }
        for (ObjectInstance mbean : mbeans) {
            for (JmxMetric.Attribute attribute : jmxMetric.getAttributes()) {
                ObjectName objectName = mbean.getObjectName();
                try {
                    Object value = server.getAttribute(objectName, attribute.getJmxAttributeName());
                    if (value instanceof Number) {
                        logger.debug("Found number attribute {}={}", (Object)attribute.getJmxAttributeName(), value);
                        registrations.add(new JmxMetricRegistration(JMX_PREFIX + attribute.getMetricName(), Labels.Mutable.of(objectName.getKeyPropertyList()), attribute.getJmxAttributeName(), null, objectName));
                        continue;
                    }
                    if (value instanceof CompositeData) {
                        CompositeData compositeValue = (CompositeData)value;
                        for (String key : compositeValue.getCompositeType().keySet()) {
                            if (compositeValue.get(key) instanceof Number) {
                                logger.debug("Found composite number attribute {}.{}={}", attribute.getJmxAttributeName(), key, value);
                                registrations.add(new JmxMetricRegistration(JMX_PREFIX + attribute.getMetricName() + "." + key, Labels.Mutable.of(objectName.getKeyPropertyList()), attribute.getJmxAttributeName(), key, objectName));
                                continue;
                            }
                            logger.warn("Can't create metric '{}' because composite value '{}' is not a number: '{}'", jmxMetric, key, value);
                        }
                        continue;
                    }
                    logger.warn("Can't create metric '{}' because attribute '{}' is not a number: '{}'", jmxMetric, attribute.getJmxAttributeName(), value);
                }
                catch (AttributeNotFoundException e) {
                    logger.warn("Can't create metric '{}' because attribute '{}' could not be found", (Object)jmxMetric, (Object)attribute.getJmxAttributeName());
                }
            }
        }
    }

    @Override
    public void stop() throws Exception {
        Thread logManagerPropertyPoller;
        MBeanServer server = this.server;
        NotificationListener listener = this.listener;
        if (server != null && listener != null) {
            server.removeNotificationListener(MBeanServerDelegate.DELEGATE_NAME, listener);
        }
        if ((logManagerPropertyPoller = this.logManagerPropertyPoller) != null) {
            logManagerPropertyPoller.interrupt();
        }
    }

    static class JmxMetricRegistration {
        private static final Logger logger = LoggerFactory.getLogger(JmxMetricRegistration.class);
        private final String metricName;
        private final Labels.Immutable labels;
        private final String jmxAttribute;
        @Nullable
        private final String compositeDataKey;
        private final ObjectName objectName;

        private JmxMetricRegistration(String metricName, Labels labels, String jmxAttribute, @Nullable String compositeDataKey, ObjectName objectName) {
            this.metricName = metricName;
            this.labels = labels.immutableCopy();
            this.jmxAttribute = jmxAttribute;
            this.compositeDataKey = compositeDataKey;
            this.objectName = objectName;
        }

        void register(final MBeanServer server, final MetricRegistry metricRegistry) {
            logger.debug("Registering JMX metric {} {}.{} as metric_name: {} labels: {}", this.objectName, this.jmxAttribute, this.compositeDataKey, this.metricName, this.labels);
            metricRegistry.add(this.metricName, this.labels, new DoubleSupplier(){

                @Override
                public double get() {
                    try {
                        if (JmxMetricRegistration.this.compositeDataKey == null) {
                            return ((Number)server.getAttribute(JmxMetricRegistration.this.objectName, JmxMetricRegistration.this.jmxAttribute)).doubleValue();
                        }
                        return ((Number)((CompositeData)server.getAttribute(JmxMetricRegistration.this.objectName, JmxMetricRegistration.this.jmxAttribute)).get(JmxMetricRegistration.this.compositeDataKey)).doubleValue();
                    }
                    catch (AttributeNotFoundException | InstanceNotFoundException e) {
                        JmxMetricRegistration.this.unregister(metricRegistry);
                        return Double.NaN;
                    }
                    catch (Exception e) {
                        return Double.NaN;
                    }
                }
            });
        }

        void unregister(MetricRegistry metricRegistry) {
            logger.debug("Unregistering JMX metric {} {}.{} metric_name: {} labels: {}", this.objectName, this.jmxAttribute, this.compositeDataKey, this.metricName, this.labels);
            metricRegistry.removeGauge(this.metricName, this.labels);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            JmxMetricRegistration that = (JmxMetricRegistration)o;
            return this.metricName.equals(that.metricName) && this.labels.equals(that.labels);
        }

        public int hashCode() {
            return Objects.hash(this.metricName, this.labels);
        }
    }
}

