/*
 * Decompiled with CFR 0.152.
 */
package co.elastic.apm.agent.impl.circuitbreaker;

import co.elastic.apm.agent.context.AbstractLifecycleListener;
import co.elastic.apm.agent.impl.ElasticApmTracer;
import co.elastic.apm.agent.impl.circuitbreaker.CircuitBreakerConfiguration;
import co.elastic.apm.agent.impl.circuitbreaker.GCStressMonitor;
import co.elastic.apm.agent.impl.circuitbreaker.StressMonitor;
import co.elastic.apm.agent.impl.circuitbreaker.SystemCpuStressMonitor;
import co.elastic.apm.agent.sdk.logging.Logger;
import co.elastic.apm.agent.sdk.logging.LoggerFactory;
import co.elastic.apm.agent.util.ExecutorUtils;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class CircuitBreaker
extends AbstractLifecycleListener {
    private static final Logger logger = LoggerFactory.getLogger(CircuitBreaker.class);
    private final ThreadPoolExecutor threadPool;
    private final ElasticApmTracer tracer;
    private final CircuitBreakerConfiguration circuitBreakerConfiguration;
    private final long pollInterval;
    private boolean isCurrentlyUnderStress = false;
    private final List<StressMonitor> stressMonitors = new CopyOnWriteArrayList<StressMonitor>();

    public CircuitBreaker(ElasticApmTracer tracer) {
        this.tracer = tracer;
        this.circuitBreakerConfiguration = tracer.getConfig(CircuitBreakerConfiguration.class);
        this.pollInterval = this.circuitBreakerConfiguration.getStressMonitoringPollingIntervalMillis();
        this.threadPool = ExecutorUtils.createSingleThreadDaemonPool("circuit-breaker", 1);
    }

    @Override
    public void start(ElasticApmTracer tracer) {
        this.loadGCStressMonitor(tracer);
        this.loadSystemCpuStressMonitor(tracer);
        this.threadPool.execute(new Runnable(){

            @Override
            public void run() {
                CircuitBreaker.this.pollStressMonitors();
            }
        });
    }

    private void loadGCStressMonitor(ElasticApmTracer tracer) {
        try {
            this.stressMonitors.add(new GCStressMonitor(tracer));
        }
        catch (Throwable throwable) {
            logger.error("Failed to load the GC stress monitor. Circuit breaker will not be triggered based on GC events.", throwable);
        }
    }

    private void loadSystemCpuStressMonitor(ElasticApmTracer tracer) {
        try {
            this.stressMonitors.add(new SystemCpuStressMonitor(tracer));
        }
        catch (Throwable throwable) {
            logger.error("Failed to load the system CPU stress monitor. Circuit breaker will not be triggered based on system CPU events.", throwable);
        }
    }

    private void pollStressMonitors() {
        while (!Thread.currentThread().isInterrupted()) {
            try {
                if (this.circuitBreakerConfiguration.isCircuitBreakerEnabled()) {
                    if (this.isCurrentlyUnderStress) {
                        if (this.isStressRelieved()) {
                            logger.info("All registered stress monitors indicate that the stress has been relieved");
                            this.isCurrentlyUnderStress = false;
                            this.tracer.onStressRelieved();
                        }
                    } else if (this.isUnderStress()) {
                        this.isCurrentlyUnderStress = true;
                        this.tracer.onStressDetected();
                    }
                } else if (this.isCurrentlyUnderStress) {
                    this.isCurrentlyUnderStress = false;
                    this.tracer.onStressRelieved();
                }
            }
            catch (Throwable throwable) {
                logger.error("Error occurred during Circuit Breaker polling", throwable);
            }
            try {
                if (logger.isTraceEnabled()) {
                    logger.trace("Scheduling next stress monitor polling in {}s", (Object)this.pollInterval);
                }
                TimeUnit.MILLISECONDS.sleep(this.pollInterval);
            }
            catch (InterruptedException e) {
                logger.info("Stopping the Circuit Breaker thread.");
                Thread.currentThread().interrupt();
            }
        }
    }

    private boolean isUnderStress() {
        for (StressMonitor stressMonitor : this.stressMonitors) {
            try {
                if (!stressMonitor.isUnderStress()) continue;
                logger.info("Stress detected by {}: {}", (Object)stressMonitor.getClass().getName(), (Object)stressMonitor.getStressDetectionInfo());
                return true;
            }
            catch (Exception e) {
                if (!logger.isDebugEnabled()) continue;
                logger.debug("Failed to poll " + stressMonitor.getClass().getName(), e);
            }
        }
        return false;
    }

    private boolean isStressRelieved() {
        boolean stressRelieved = true;
        for (StressMonitor stressMonitor : this.stressMonitors) {
            try {
                stressRelieved &= stressMonitor.isStressRelieved();
            }
            catch (Exception e) {
                if (!logger.isDebugEnabled()) continue;
                logger.debug("Failed to poll " + stressMonitor.getClass().getName(), e);
            }
        }
        return stressRelieved;
    }

    void registerStressMonitor(StressMonitor monitor) {
        this.stressMonitors.add(monitor);
    }

    void unregisterStressMonitor(StressMonitor monitor) {
        this.stressMonitors.remove(monitor);
    }

    @Override
    public void stop() {
        this.threadPool.shutdownNow();
    }
}

