package co.csadev.kellocharts.model

import android.graphics.Color
import android.graphics.Typeface
import co.csadev.kellocharts.formatter.AxisValueFormatter

import java.util.ArrayList

import co.csadev.kellocharts.formatter.SimpleAxisValueFormatter
import co.csadev.kellocharts.util.ChartUtils

/**
 * Single axis model. By default axis is auto-generated. Use [.setAutoGenerated] to disable axis values
 * generation and set values manually using [.setValues]. If Axis is auto-generated [.setValues]
 * will be ignored but if you set some values Axis will switch to manual mode. Change how axis labels are displayed by
 * changing formatter [.setFormatter]. Axis can have a name
 * that should be displayed next to
 * labels(that depends on renderer implementation), you can change name using [.setName], by default axis
 * name is empty and therefore not displayed.
 */
class Axis(values: MutableList<AxisValue> = ArrayList(),
           var name: String? = null,
           var isAutoGenerated: Boolean = true,
           var hasLines: Boolean = false,
           var isInside: Boolean = false,
           var textColor: Int = Color.LTGRAY,
           var lineColor: Int = ChartUtils.DEFAULT_DARKEN_COLOR,
           var textSize: Int = DEFAULT_TEXT_SIZE_SP,
           maxLabelChars: Int = DEFAULT_MAX_AXIS_LABEL_CHARS,
           var typeface: Typeface? = null,
           var formatter: AxisValueFormatter = SimpleAxisValueFormatter(),
           var hasSeparationLine: Boolean = true,
           var hasTiltedLabels: Boolean = false,
           var isReversed: Boolean = false,
           var maxLabels: Int = -1) {

    var maxLabelChars: Int = maxLabelChars
        set(value) {
            field = Math.min(32, Math.max(0, value))
        }

    var values: MutableList<AxisValue> = values
        get() {
            //Copy and transfer to ensure that other value logic stays intact
            if (isReversed) {
                return field.indices.reversed().mapTo(ArrayList()) { field[it] }
            }
            return field
        }
        set(value) {
            field = value
            this.isAutoGenerated = false
        }

    fun copy() = Axis(values.map { it.copy() }.toMutableList(), name, isAutoGenerated, hasLines, isInside, textColor, lineColor, textSize, maxLabelChars, typeface, formatter, hasSeparationLine)

    companion object {
        val DEFAULT_TEXT_SIZE_SP = 12
        val DEFAULT_MAX_AXIS_LABEL_CHARS = 3

        /**
         * Generates Axis with values from start to stop inclusive.
         */
        fun generateAxisFromRange(start: Float, stop: Float, step: Float): Axis {

            val values = ArrayList<AxisValue>()
            var value = start
            while (value <= stop) {
                val axisValue = AxisValue(value)
                values.add(axisValue)
                value += step
            }

            return Axis(values = values)
        }

        /**
         * Generates Axis with values from given list.
         */
        fun generateAxisFromCollection(axisValues: List<Float>): Axis {
            val values = ArrayList<AxisValue>()
            var index = 0
            for (value in axisValues) {
                val axisValue = AxisValue(value)
                values.add(axisValue)
                ++index
            }

            return Axis(values = values)
        }

        /**
         * Generates Axis with values and labels from given lists, both lists must have the same size.
         */
        fun generateAxisFromCollection(axisValues: List<Float>, axisValuesLabels: List<String>): Axis {
            if (axisValues.size != axisValuesLabels.size) {
                throw IllegalArgumentException("Values and labels lists must have the same size!")
            }
            return Axis(values = axisValues.mapIndexedTo(ArrayList()) { index, value -> AxisValue(value, axisValuesLabels[index].toCharArray()) })
        }
    }
}
