/*
 * Decompiled with CFR 0.152.
 */
package co.cask.tephra.persist;

import co.cask.tephra.persist.AbstractTransactionStateStorage;
import co.cask.tephra.persist.LocalFileTransactionLog;
import co.cask.tephra.persist.TransactionLog;
import co.cask.tephra.persist.TransactionSnapshot;
import co.cask.tephra.snapshot.SnapshotCodecProvider;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.google.common.primitives.Longs;
import com.google.inject.Inject;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LocalFileTransactionStateStorage
extends AbstractTransactionStateStorage {
    private static final String TMP_SNAPSHOT_FILE_PREFIX = ".in-progress.";
    private static final String SNAPSHOT_FILE_PREFIX = "snapshot.";
    private static final String LOG_FILE_PREFIX = "txlog.";
    private static final Logger LOG = LoggerFactory.getLogger(LocalFileTransactionStateStorage.class);
    static final int BUFFER_SIZE = 16384;
    private static final FilenameFilter SNAPSHOT_FILE_FILTER = new FilenameFilter(){

        @Override
        public boolean accept(File file, String s) {
            return s.startsWith(LocalFileTransactionStateStorage.SNAPSHOT_FILE_PREFIX);
        }
    };
    private final String configuredSnapshotDir;
    private File snapshotDir;

    @Inject
    public LocalFileTransactionStateStorage(Configuration conf, SnapshotCodecProvider codecProvider) {
        super(codecProvider);
        this.configuredSnapshotDir = conf.get("data.tx.snapshot.local.dir");
    }

    protected void startUp() throws Exception {
        Preconditions.checkState((this.configuredSnapshotDir != null ? 1 : 0) != 0, (Object)"Snapshot directory is not configured.  Please set data.tx.snapshot.local.dir in configuration.");
        this.snapshotDir = new File(this.configuredSnapshotDir);
        if (!this.snapshotDir.exists()) {
            if (!this.snapshotDir.mkdirs()) {
                throw new IOException("Failed to create directory " + this.configuredSnapshotDir + " for transaction snapshot storage");
            }
        } else {
            Preconditions.checkState((boolean)this.snapshotDir.isDirectory(), (Object)("Configured snapshot directory " + this.configuredSnapshotDir + " is not a directory!"));
            Preconditions.checkState((boolean)this.snapshotDir.canWrite(), (Object)("Configured snapshot directory " + this.configuredSnapshotDir + " exists but is not writable!"));
        }
    }

    protected void shutDown() throws Exception {
    }

    @Override
    public String getLocation() {
        return this.snapshotDir.getAbsolutePath();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void writeSnapshot(TransactionSnapshot snapshot) throws IOException {
        File snapshotTmpFile = new File(this.snapshotDir, TMP_SNAPSHOT_FILE_PREFIX + snapshot.getTimestamp());
        LOG.info("Writing snapshot to temporary file {}", (Object)snapshotTmpFile);
        OutputStream out = (OutputStream)Files.newOutputStreamSupplier((File)snapshotTmpFile).getOutput();
        boolean threw = true;
        try {
            this.codecProvider.encode(out, snapshot);
            threw = false;
        }
        finally {
            Closeables.close((Closeable)out, (boolean)threw);
        }
        File finalFile = new File(this.snapshotDir, SNAPSHOT_FILE_PREFIX + snapshot.getTimestamp());
        if (!snapshotTmpFile.renameTo(finalFile)) {
            throw new IOException("Failed renaming temporary snapshot file " + snapshotTmpFile.getName() + " to " + finalFile.getName());
        }
        LOG.info("Completed snapshot to file {}", (Object)finalFile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public TransactionSnapshot getLatestSnapshot() throws IOException {
        InputStream is = this.getLatestSnapshotInputStream();
        if (is == null) {
            return null;
        }
        try {
            TransactionSnapshot transactionSnapshot = this.readSnapshotFile(is);
            return transactionSnapshot;
        }
        finally {
            is.close();
        }
    }

    private InputStream getLatestSnapshotInputStream() throws IOException {
        File[] snapshotFiles = this.snapshotDir.listFiles(SNAPSHOT_FILE_FILTER);
        TimestampedFilename mostRecent = null;
        for (File file : snapshotFiles) {
            TimestampedFilename tsFile = new TimestampedFilename(file);
            if (mostRecent != null && tsFile.compareTo(mostRecent) <= 0) continue;
            mostRecent = tsFile;
        }
        if (mostRecent == null) {
            LOG.info("No snapshot files found in {}", (Object)this.snapshotDir.getAbsolutePath());
            return null;
        }
        return new FileInputStream(mostRecent.getFile());
    }

    private TransactionSnapshot readSnapshotFile(InputStream is) throws IOException {
        return this.codecProvider.decode(is);
    }

    @Override
    public long deleteOldSnapshots(int numberToKeep) throws IOException {
        File[] snapshotFiles = this.snapshotDir.listFiles(SNAPSHOT_FILE_FILTER);
        if (snapshotFiles.length == 0) {
            return -1L;
        }
        TimestampedFilename[] snapshotFilenames = new TimestampedFilename[snapshotFiles.length];
        for (int i = 0; i < snapshotFiles.length; ++i) {
            snapshotFilenames[i] = new TimestampedFilename(snapshotFiles[i]);
        }
        Arrays.sort(snapshotFilenames, Collections.reverseOrder());
        if (snapshotFilenames.length <= numberToKeep) {
            return snapshotFilenames[snapshotFilenames.length - 1].getTimestamp();
        }
        int toRemoveCount = snapshotFilenames.length - numberToKeep;
        TimestampedFilename[] toRemove = new TimestampedFilename[toRemoveCount];
        System.arraycopy(snapshotFilenames, numberToKeep, toRemove, 0, toRemoveCount);
        int removedCnt = 0;
        for (int i = 0; i < toRemove.length; ++i) {
            File currentFile = toRemove[i].getFile();
            LOG.debug("Removing old snapshot file {}", (Object)currentFile.getAbsolutePath());
            if (!toRemove[i].getFile().delete()) {
                LOG.error("Failed deleting snapshot file {}", (Object)currentFile.getAbsolutePath());
                continue;
            }
            ++removedCnt;
        }
        long oldestTimestamp = snapshotFilenames[numberToKeep - 1].getTimestamp();
        LOG.info("Removed {} out of {} expected snapshot files older than {}", new Object[]{removedCnt, toRemoveCount, oldestTimestamp});
        return oldestTimestamp;
    }

    @Override
    public List<String> listSnapshots() throws IOException {
        File[] snapshots = this.snapshotDir.listFiles(SNAPSHOT_FILE_FILTER);
        return Lists.transform(Arrays.asList(snapshots), (Function)new Function<File, String>(){

            @Nullable
            public String apply(@Nullable File input) {
                return input.getName();
            }
        });
    }

    @Override
    public List<TransactionLog> getLogsSince(long timestamp) throws IOException {
        File[] logFiles = this.snapshotDir.listFiles(new LogFileFilter(timestamp, Long.MAX_VALUE));
        Object[] timestampedFiles = new TimestampedFilename[logFiles.length];
        for (int i = 0; i < logFiles.length; ++i) {
            timestampedFiles[i] = new TimestampedFilename(logFiles[i]);
        }
        Arrays.sort(timestampedFiles);
        return Lists.transform(Arrays.asList(timestampedFiles), (Function)new Function<TimestampedFilename, TransactionLog>(){

            @Nullable
            public TransactionLog apply(@Nullable TimestampedFilename input) {
                return new LocalFileTransactionLog(input.getFile(), input.getTimestamp());
            }
        });
    }

    @Override
    public TransactionLog createLog(long timestamp) throws IOException {
        File newLogFile = new File(this.snapshotDir, LOG_FILE_PREFIX + timestamp);
        LOG.info("Creating new transaction log at {}", (Object)newLogFile.getAbsolutePath());
        return new LocalFileTransactionLog(newLogFile, timestamp);
    }

    @Override
    public void deleteLogsOlderThan(long timestamp) throws IOException {
        File[] logFiles = this.snapshotDir.listFiles(new LogFileFilter(0L, timestamp));
        int removedCnt = 0;
        for (File file : logFiles) {
            LOG.debug("Removing old transaction log {}", (Object)file.getPath());
            if (file.delete()) {
                ++removedCnt;
                continue;
            }
            LOG.warn("Failed to remove log file {}", (Object)file.getAbsolutePath());
        }
        LOG.info("Removed {} transaction logs older than {}", (Object)removedCnt, (Object)timestamp);
    }

    @Override
    public List<String> listLogs() throws IOException {
        File[] logs = this.snapshotDir.listFiles(new LogFileFilter(0L, Long.MAX_VALUE));
        return Lists.transform(Arrays.asList(logs), (Function)new Function<File, String>(){

            @Nullable
            public String apply(@Nullable File input) {
                return input.getName();
            }
        });
    }

    private static class TimestampedFilename
    implements Comparable<TimestampedFilename> {
        private File file;
        private String prefix;
        private long timestamp;

        public TimestampedFilename(File file) {
            this.file = file;
            String[] parts = file.getName().split("\\.");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Filename " + file.getName() + " did not match the expected pattern prefix.timestamp");
            }
            this.prefix = parts[0];
            this.timestamp = Long.parseLong(parts[1]);
        }

        public File getFile() {
            return this.file;
        }

        public String getPrefix() {
            return this.prefix;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        @Override
        public int compareTo(TimestampedFilename other) {
            int res = this.prefix.compareTo(other.getPrefix());
            if (res == 0) {
                res = Longs.compare((long)this.timestamp, (long)other.getTimestamp());
            }
            return res;
        }
    }

    private static class LogFileFilter
    implements FilenameFilter {
        private final long startTime;
        private final long endTime;

        public LogFileFilter(long startTime, long endTime) {
            this.startTime = startTime;
            this.endTime = endTime;
        }

        @Override
        public boolean accept(File file, String s) {
            String[] parts;
            if (s.startsWith(LocalFileTransactionStateStorage.LOG_FILE_PREFIX) && (parts = s.split("\\.")).length == 2) {
                try {
                    long fileTime = Long.parseLong(parts[1]);
                    return fileTime >= this.startTime && fileTime < this.endTime;
                }
                catch (NumberFormatException ignored) {
                    LOG.warn("Filename {} did not match the expected pattern prefix.<timestamp>", (Object)s);
                }
            }
            return false;
        }
    }
}

