/*
 * Decompiled with CFR 0.152.
 */
package co.cask.tephra.persist;

import co.cask.tephra.persist.AbstractTransactionStateStorage;
import co.cask.tephra.persist.HDFSTransactionLog;
import co.cask.tephra.persist.TransactionEdit;
import co.cask.tephra.persist.TransactionLog;
import co.cask.tephra.persist.TransactionLogReader;
import co.cask.tephra.persist.TransactionSnapshot;
import co.cask.tephra.snapshot.SnapshotCodecProvider;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.primitives.Longs;
import com.google.inject.Inject;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import javax.annotation.Nullable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HDFSTransactionStateStorage
extends AbstractTransactionStateStorage {
    private static final Logger LOG = LoggerFactory.getLogger(HDFSTransactionStateStorage.class);
    private static final String SNAPSHOT_FILE_PREFIX = "snapshot.";
    private static final String TMP_SNAPSHOT_FILE_PREFIX = ".in-progress.snapshot.";
    private static final String LOG_FILE_PREFIX = "txlog.";
    private static final PathFilter SNAPSHOT_FILE_FILTER = new PathFilter(){

        public boolean accept(Path path) {
            return path.getName().startsWith(HDFSTransactionStateStorage.SNAPSHOT_FILE_PREFIX);
        }
    };
    private static final int BUFFER_SIZE = 16384;
    private FileSystem fs;
    private Configuration hConf;
    private String configuredSnapshotDir;
    private Path snapshotDir;

    @Inject
    public HDFSTransactionStateStorage(Configuration hConf, SnapshotCodecProvider codecProvider) {
        super(codecProvider);
        this.hConf = hConf;
        this.configuredSnapshotDir = hConf.get("data.tx.snapshot.dir");
    }

    protected void startUp() throws Exception {
        Preconditions.checkState((this.configuredSnapshotDir != null ? 1 : 0) != 0, (Object)"Snapshot directory is not configured.  Please set data.tx.snapshot.dir in configuration.");
        String hdfsUser = this.hConf.get("data.tx.hdfs.user");
        if (hdfsUser == null || UserGroupInformation.isSecurityEnabled()) {
            if (hdfsUser != null && LOG.isDebugEnabled()) {
                LOG.debug("Ignoring configuration {}={}, running on secure Hadoop", (Object)"data.tx.hdfs.user", (Object)hdfsUser);
            }
            this.fs = FileSystem.newInstance((URI)FileSystem.getDefaultUri((Configuration)this.hConf), (Configuration)this.hConf);
        } else {
            this.fs = FileSystem.newInstance((URI)FileSystem.getDefaultUri((Configuration)this.hConf), (Configuration)this.hConf, (String)hdfsUser);
        }
        this.snapshotDir = new Path(this.configuredSnapshotDir);
        LOG.info("Using snapshot dir " + this.snapshotDir);
        if (!this.fs.exists(this.snapshotDir)) {
            LOG.info("Creating snapshot dir at {}", (Object)this.snapshotDir);
            this.fs.mkdirs(this.snapshotDir);
        }
    }

    protected void shutDown() throws Exception {
        this.fs.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void writeSnapshot(TransactionSnapshot snapshot) throws IOException {
        Path snapshotTmpFile = new Path(this.snapshotDir, TMP_SNAPSHOT_FILE_PREFIX + snapshot.getTimestamp());
        LOG.info("Writing snapshot to temporary file {}", (Object)snapshotTmpFile);
        FSDataOutputStream out = this.fs.create(snapshotTmpFile, false, 16384);
        try {
            this.codecProvider.encode((OutputStream)out, snapshot);
        }
        finally {
            out.close();
        }
        Path finalFile = new Path(this.snapshotDir, SNAPSHOT_FILE_PREFIX + snapshot.getTimestamp());
        this.fs.rename(snapshotTmpFile, finalFile);
        LOG.info("Completed snapshot to file {}", (Object)finalFile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public TransactionSnapshot getLatestSnapshot() throws IOException {
        InputStream in = this.getLatestSnapshotInputStream();
        if (in == null) {
            return null;
        }
        try {
            TransactionSnapshot transactionSnapshot = this.readSnapshotInputStream(in);
            return transactionSnapshot;
        }
        finally {
            in.close();
        }
    }

    private InputStream getLatestSnapshotInputStream() throws IOException {
        Object[] snapshots = this.listSnapshotFiles();
        Arrays.sort(snapshots);
        if (snapshots.length > 0) {
            return this.fs.open(((TimestampedFilename)snapshots[snapshots.length - 1]).getPath(), 16384);
        }
        LOG.info("No snapshot files found in {}", (Object)this.snapshotDir);
        return null;
    }

    private TransactionSnapshot readSnapshotInputStream(InputStream in) throws IOException {
        return this.codecProvider.decode(in);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private TransactionSnapshot readSnapshotFile(Path filePath) throws IOException {
        FSDataInputStream in = this.fs.open(filePath, 16384);
        try {
            TransactionSnapshot transactionSnapshot = this.readSnapshotInputStream((InputStream)in);
            return transactionSnapshot;
        }
        finally {
            in.close();
        }
    }

    private TimestampedFilename[] listSnapshotFiles() throws IOException {
        FileStatus[] snapshotFileStatuses = this.fs.listStatus(this.snapshotDir, SNAPSHOT_FILE_FILTER);
        TimestampedFilename[] snapshotFiles = new TimestampedFilename[snapshotFileStatuses.length];
        for (int i = 0; i < snapshotFileStatuses.length; ++i) {
            snapshotFiles[i] = new TimestampedFilename(snapshotFileStatuses[i].getPath());
        }
        return snapshotFiles;
    }

    @Override
    public long deleteOldSnapshots(int numberToKeep) throws IOException {
        TimestampedFilename[] snapshots = this.listSnapshotFiles();
        if (snapshots.length == 0) {
            return -1L;
        }
        Arrays.sort(snapshots, Collections.reverseOrder());
        if (snapshots.length <= numberToKeep) {
            return snapshots[snapshots.length - 1].getTimestamp();
        }
        int toRemoveCount = snapshots.length - numberToKeep;
        TimestampedFilename[] toRemove = new TimestampedFilename[toRemoveCount];
        System.arraycopy(snapshots, numberToKeep, toRemove, 0, toRemoveCount);
        for (TimestampedFilename f : toRemove) {
            LOG.debug("Removing old snapshot file {}", (Object)f.getPath());
            this.fs.delete(f.getPath(), false);
        }
        long oldestTimestamp = snapshots[numberToKeep - 1].getTimestamp();
        LOG.info("Removed {} old snapshot files prior to {}", (Object)toRemoveCount, (Object)oldestTimestamp);
        return oldestTimestamp;
    }

    @Override
    public List<String> listSnapshots() throws IOException {
        FileStatus[] files = this.fs.listStatus(this.snapshotDir, SNAPSHOT_FILE_FILTER);
        return Lists.transform(Arrays.asList(files), (Function)new Function<FileStatus, String>(){

            @Nullable
            public String apply(@Nullable FileStatus input) {
                return input.getPath().getName();
            }
        });
    }

    @Override
    public List<TransactionLog> getLogsSince(long timestamp) throws IOException {
        FileStatus[] statuses = this.fs.listStatus(this.snapshotDir, (PathFilter)new LogFileFilter(timestamp, Long.MAX_VALUE));
        TimestampedFilename[] timestampedFiles = new TimestampedFilename[statuses.length];
        for (int i = 0; i < statuses.length; ++i) {
            timestampedFiles[i] = new TimestampedFilename(statuses[i].getPath());
        }
        return Lists.transform(Arrays.asList(timestampedFiles), (Function)new Function<TimestampedFilename, TransactionLog>(){

            @Nullable
            public TransactionLog apply(@Nullable TimestampedFilename input) {
                return HDFSTransactionStateStorage.this.openLog(input.getPath(), input.getTimestamp());
            }
        });
    }

    @Override
    public TransactionLog createLog(long timestamp) throws IOException {
        Path newLog = new Path(this.snapshotDir, LOG_FILE_PREFIX + timestamp);
        return this.openLog(newLog, timestamp);
    }

    private TransactionLog openLog(Path path, long timestamp) {
        return new HDFSTransactionLog(this.fs, this.hConf, path, timestamp);
    }

    @Override
    public void deleteLogsOlderThan(long timestamp) throws IOException {
        FileStatus[] statuses = this.fs.listStatus(this.snapshotDir, (PathFilter)new LogFileFilter(0L, timestamp));
        int removedCnt = 0;
        for (FileStatus status2 : statuses) {
            LOG.debug("Removing old transaction log {}", (Object)status2.getPath());
            if (this.fs.delete(status2.getPath(), false)) {
                ++removedCnt;
                continue;
            }
            LOG.error("Failed to delete transaction log file {}", (Object)status2.getPath());
        }
        LOG.info("Removed {} transaction logs older than {}", (Object)removedCnt, (Object)timestamp);
    }

    @Override
    public List<String> listLogs() throws IOException {
        FileStatus[] files = this.fs.listStatus(this.snapshotDir, (PathFilter)new LogFileFilter(0L, Long.MAX_VALUE));
        return Lists.transform(Arrays.asList(files), (Function)new Function<FileStatus, String>(){

            @Nullable
            public String apply(@Nullable FileStatus input) {
                return input.getPath().getName();
            }
        });
    }

    @Override
    public String getLocation() {
        return this.snapshotDir.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static void main(String[] args) {
        ArrayList filenames = Lists.newArrayList();
        CLIMode mode = null;
        for (String arg : args) {
            if ("-s".equals(arg)) {
                mode = CLIMode.SNAPSHOT;
                continue;
            }
            if ("-l".equals(arg)) {
                mode = CLIMode.TXLOG;
                continue;
            }
            if ("-h".equals(arg)) {
                HDFSTransactionStateStorage.printUsage(null);
                continue;
            }
            filenames.add(arg);
        }
        if (mode == null) {
            HDFSTransactionStateStorage.printUsage("ERROR: Either -s or -l is required to set mode.", 1);
        }
        Configuration config = HBaseConfiguration.create();
        HDFSTransactionStateStorage storage = new HDFSTransactionStateStorage(config, new SnapshotCodecProvider(config));
        storage.startAndWait();
        try {
            switch (mode) {
                case SNAPSHOT: {
                    String file;
                    try {
                        if (filenames.isEmpty()) {
                            TransactionSnapshot snapshot = storage.getLatestSnapshot();
                            HDFSTransactionStateStorage.printSnapshot(snapshot);
                        }
                        Iterator i$ = filenames.iterator();
                        while (i$.hasNext()) {
                            file = (String)i$.next();
                            Path path = new Path(file);
                            TransactionSnapshot snapshot = storage.readSnapshotFile(path);
                            HDFSTransactionStateStorage.printSnapshot(snapshot);
                            System.out.println();
                        }
                        return;
                    }
                    catch (IOException ioe) {
                        System.err.println("Error reading snapshot files: " + ioe.getMessage());
                        ioe.printStackTrace();
                        System.exit(1);
                        return;
                    }
                }
                case TXLOG: {
                    String file;
                    if (filenames.isEmpty()) {
                        HDFSTransactionStateStorage.printUsage("ERROR: At least one transaction log filename is required!", 1);
                    }
                    Iterator i$ = filenames.iterator();
                    while (i$.hasNext()) {
                        file = (String)i$.next();
                        TimestampedFilename timestampedFilename = new TimestampedFilename(new Path(file));
                        TransactionLog log = storage.openLog(timestampedFilename.getPath(), timestampedFilename.getTimestamp());
                        HDFSTransactionStateStorage.printLog(log);
                        System.out.println();
                    }
                    return;
                }
            }
            return;
        }
        finally {
            storage.stop();
        }
    }

    private static void printUsage(String message) {
        HDFSTransactionStateStorage.printUsage(message, 0);
    }

    private static void printUsage(String message, int exitCode) {
        if (message != null) {
            System.out.println(message);
        }
        System.out.println("Usage: java " + HDFSTransactionStateStorage.class.getName() + " (-s|-l) file1 [file2...]");
        System.out.println();
        System.out.println("\t-s\tRead files as transaction state snapshots (will default to latest if no file given)");
        System.out.println("\t-l\tRead files as transaction logs [filename is required]");
        System.out.println("\t-h\tPrint this message");
        System.exit(exitCode);
    }

    private static void printSnapshot(TransactionSnapshot snapshot) {
        Date snapshotDate = new Date(snapshot.getTimestamp());
        System.out.println("TransactionSnapshot at " + snapshotDate.toString());
        System.out.println("\t" + snapshot.toString());
    }

    private static void printLog(TransactionLog log) {
        try {
            TransactionEdit edit;
            System.out.println("TransactionLog " + log.getName());
            TransactionLogReader reader = log.getReader();
            long seq = 0L;
            while ((edit = reader.next()) != null) {
                System.out.println(String.format("    %d: %s", seq++, edit.toString()));
            }
        }
        catch (IOException ioe) {
            System.err.println("ERROR reading log " + log.getName() + ": " + ioe.getMessage());
            ioe.printStackTrace();
        }
    }

    private static enum CLIMode {
        SNAPSHOT,
        TXLOG;

    }

    private static class TimestampedFilename
    implements Comparable<TimestampedFilename> {
        private Path path;
        private String prefix;
        private long timestamp;

        public TimestampedFilename(Path path) {
            this.path = path;
            String[] parts = path.getName().split("\\.");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Filename " + path.getName() + " did not match the expected pattern prefix.timestamp");
            }
            this.prefix = parts[0];
            this.timestamp = Long.parseLong(parts[1]);
        }

        public Path getPath() {
            return this.path;
        }

        public String getPrefix() {
            return this.prefix;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        @Override
        public int compareTo(TimestampedFilename other) {
            int res = this.prefix.compareTo(other.getPrefix());
            if (res == 0) {
                res = Longs.compare((long)this.timestamp, (long)other.getTimestamp());
            }
            return res;
        }
    }

    private static class LogFileFilter
    implements PathFilter {
        private final long startTime;
        private final long endTime;

        public LogFileFilter(long startTime, long endTime) {
            this.startTime = startTime;
            this.endTime = endTime;
        }

        public boolean accept(Path path) {
            String[] parts;
            if (path.getName().startsWith(HDFSTransactionStateStorage.LOG_FILE_PREFIX) && (parts = path.getName().split("\\.")).length == 2) {
                try {
                    long fileTime = Long.parseLong(parts[1]);
                    return fileTime >= this.startTime && fileTime < this.endTime;
                }
                catch (NumberFormatException ignored) {
                    LOG.warn("Filename {} did not match the expected pattern prefix.<timestamp>", (Object)path.getName());
                }
            }
            return false;
        }
    }
}

