package cn.lollypop.android.thermometer.ble.ota;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.UUID;

import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattService;
import android.content.Context;
import android.os.Environment;
import com.basic.util.ClsUtil;
import com.basic.util.CommonUtil;
import com.orhanobut.logger.Logger;

import cn.lollypop.android.thermometer.ble.BleCallback;
import cn.lollypop.android.thermometer.ble.utils.DeviceInformationServiceUtil;

public class OtaManager {

  public static final UUID UUID_OTA_SERVICE =
      UUID.fromString(
          GattAttributes.SPOTA_SERVICE);

  public static final UUID UUID_OTA_MEM_DEV_UUID = UUID.fromString(
      GattAttributes.SPOTA_MEM_DEV_UUID
  );

  public static final UUID UUID_OTA_GPIO_MAP_UUID = UUID.fromString(
      GattAttributes.SPOTA_GPIO_MAP_UUID
  );

  public static final UUID UUID_OTA_MEM_INFO_UUID = UUID.fromString(
      GattAttributes.SPOTA_MEM_INFO_UUID
  );

  public static final UUID UUID_OTA_PATCH_LEN_UUID = UUID.fromString(
      GattAttributes
          .SPOTA_PATCH_LEN_UUID);

  public static final UUID UUID_OTA_PATCH_DATA_UUID = UUID.fromString(
      GattAttributes
          .SPOTA_PATCH_DATA_UUID);

  public static final UUID UUID_OTA_SERV_STATUS_UUID = UUID.fromString(
      GattAttributes
          .SPOTA_SERV_STATUS_UUID);

  private BluetoothGattCharacteristic otaMemDevCharacteristic;
  private BluetoothGattCharacteristic otaGpioMapCharacteristic;
  private BluetoothGattCharacteristic otaMemInfoCharacteristic;
  private BluetoothGattCharacteristic otaPatchLenCharacteristic;
  private BluetoothGattCharacteristic otaPatchDataCharacteristic;
  private BluetoothGattCharacteristic otaServStatusCharacteristic;

  private int step;
  private boolean isSendData;

  private PatchData patchData;
  private PatchData.Block block;
  private BluetoothGatt bluetoothGatt;
  private BleCallback bleCallback;
  private String serverVersion;

  private Context context;

  /**
   * 是否正在传输.
   *
   * @return boolean
   */
  public boolean isTransferring() {
    return step > 0;
  }

  /**
   * 初始化特征值.
   *
   * @param gattService OTA service
   */
  public void initCharacteristic(BluetoothGattService gattService) {
    if (UUID_OTA_SERVICE.equals(gattService.getUuid())) {
      for (BluetoothGattCharacteristic gattCharacteristic : gattService
          .getCharacteristics()) {
        if (UUID_OTA_GPIO_MAP_UUID.equals(gattCharacteristic.getUuid())) {
          otaGpioMapCharacteristic = gattCharacteristic;
        } else if (UUID_OTA_MEM_DEV_UUID.equals(gattCharacteristic.getUuid())) {
          otaMemDevCharacteristic = gattCharacteristic;
        } else if (UUID_OTA_MEM_INFO_UUID.equals(
            gattCharacteristic.getUuid())) {
          otaMemInfoCharacteristic = gattCharacteristic;
        } else if (UUID_OTA_PATCH_DATA_UUID.equals(
            gattCharacteristic.getUuid())) {
          otaPatchDataCharacteristic = gattCharacteristic;
        } else if (UUID_OTA_PATCH_LEN_UUID.equals(
            gattCharacteristic.getUuid())) {
          otaPatchLenCharacteristic = gattCharacteristic;
        } else if (UUID_OTA_SERV_STATUS_UUID.equals(
            gattCharacteristic.getUuid())) {
          otaServStatusCharacteristic = gattCharacteristic;
        }
      }
    }
  }

  public void start(Context context, String serverVersion,
                    BluetoothGatt bluetoothGatt, BleCallback bleCallback) {
    this.bluetoothGatt = bluetoothGatt;
    this.bleCallback = bleCallback;
    this.serverVersion = serverVersion;
    this.context = context;

    File file = getFile(context);
    if (file == null) {
      refreshView(BleCallback.BleStatus.OPERATE_FAIL, null);
      return;
    }

    try {
      InputStream is = new FileInputStream(file);
      Logger.i("ota img size : " + is.available());
      patchData = new PatchData(is);
      step = 0;
      if (patchData.getTotal() > 0) {
        setStartAddress();
      } else {
        otaError();
      }
    } catch (IOException e) {
      e.printStackTrace();
      refreshView(BleCallback.BleStatus.OPERATE_FAIL, null);
    }
  }

  public boolean ackWrite(BluetoothGattCharacteristic characteristic,
                          int status) {
    if (UUID_OTA_MEM_DEV_UUID.equals(characteristic.getUuid())) {
      if (!Arrays.equals(GattAttributes.ADDRESS_RESET,
          characteristic.getValue())) { //不是重启回调
        readServStatus();
      }
      return true;
    } else if (UUID_OTA_GPIO_MAP_UUID.equals(characteristic.getUuid())) {
      readServStatus();
      return true;
    } else if (UUID_OTA_MEM_INFO_UUID.equals(characteristic.getUuid())) {
      readServStatus();
      return true;
    } else if (UUID_OTA_PATCH_LEN_UUID.equals(characteristic.getUuid())) {
      readServStatus();
      return true;
    } else if (UUID_OTA_PATCH_DATA_UUID.equals(characteristic.getUuid())) {
      //一个block发送结束校验一次
      if (block.isEnd()) { //校验
        readServStatus();
      } else { //不校验，继续发送data
        sendData(block);
      }
      return true;
    }
    return false;
  }

  public boolean ackRead(BluetoothGattCharacteristic characteristic,
                         int status) {
    if (UUID_OTA_SERV_STATUS_UUID.equals(characteristic.getUuid())) { //读取状态
      byte[] value = characteristic.getValue();
      if (Arrays.equals(value, new byte[]{0x10}) ||
          Arrays.equals(value, new byte[]{0x01}) ||
          Arrays.equals(value, new byte[]{0x02}) ||
          Arrays.equals(value, new byte[]{0x03})) { //OK
        switch (step) {
          case 0:
            Logger.i("OTA set start address，next to set GPIO");
            step++;
            setGPIO();
            break;
          case 1:
            Logger.i("OTA set GPIO，next to send data");
            step++;
            sendNextBlock();
            break;
          case 2:
            if (!isSendData) {
              isSendData = true;
            }
            sendBlock();
            break;
          case 3:
            Logger.i("OTA set end address， next to reboot");
            setRest();
            break;
          default:
            break;
        }
      } else {
        String error = "OTA error!";
        if (value != null && value.length > 0) {
          error += " return error value :" + ClsUtil.Bytes2HexString(value);
        }
        Logger.i(error);
        otaError();
      }
      return true;
    }
    return false;
  }

  /**
   * 设置开始地址.
   */
  private void setStartAddress() {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    otaMemDevCharacteristic.setValue(
        GattAttributes.ADDRESS_START
    );
    bluetoothGatt.writeCharacteristic(otaMemDevCharacteristic);
  }

  /**
   * 设置结束地址.
   */
  private void setEndAddress() {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    otaMemDevCharacteristic.setValue(
        GattAttributes.ADDRESS_END
    );
    bluetoothGatt.writeCharacteristic(otaMemDevCharacteristic);
  }

  /**
   * 设置复位.
   */
  private void setRest() {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    otaMemDevCharacteristic.setValue(
        GattAttributes.ADDRESS_RESET
    );
    bluetoothGatt.writeCharacteristic(otaMemDevCharacteristic);

    step = 0;
    DeviceInformationServiceUtil.setFirmwareVersion(context, serverVersion);
    refreshView(BleCallback.BleStatus.END_OTA, null);
  }

  /**
   * 设置GPIO.
   */
  private void setGPIO() {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    otaGpioMapCharacteristic.setValue(
        GattAttributes.GPIO);
    bluetoothGatt.writeCharacteristic(otaGpioMapCharacteristic);
  }

  private void sendNextBlock() {
    isSendData = false;
    block = patchData.getNext();
    sendBlock();
    refreshView(
        BleCallback.BleStatus.PROGRESS_OTA,
        CommonUtil.convertFloatToInt(patchData.getProgress(), 100)
    );
  }

  /**
   * 发送数据块.
   */
  private void sendBlock() {
    if (!isSendData) { //发送len
      setFileLength(block.getLen());
    } else { //发送数据
      if (block.isEnd()) { //数据发送完
        if (patchData.isEnd()) { //所有数据块都发送完成
          step++;
          Logger.i("OTA send data successfully，next to set end address");
          setEndAddress();
        } else { //数据块未发送完，发送下一个数据块
          sendNextBlock();
        }
      } else { //继续发送数据
        sendData(block);
      }
    }
  }

  /**
   * 设置文件长度.
   */
  private void setFileLength(int len) {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    otaPatchLenCharacteristic.setValue(
        len, BluetoothGattCharacteristic.FORMAT_UINT16, 0);
    bluetoothGatt.writeCharacteristic(otaPatchLenCharacteristic);
  }

  /**
   * 读取状态.
   */
  private void readServStatus() {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    bluetoothGatt.readCharacteristic(otaServStatusCharacteristic);
  }

  /**
   * 发送文件.
   */
  private void sendData(PatchData.Block block) {
    if (bluetoothGatt == null) {
      otaError();
      return;
    }

    otaPatchDataCharacteristic.setValue(block.getNextBytes());
    bluetoothGatt.writeCharacteristic(otaPatchDataCharacteristic);
  }

  private void refreshView(BleCallback.BleStatus bleState, Object obj) {
    bleCallback.callback(bleState, obj);
  }

  private void otaError() {
    refreshView(BleCallback.BleStatus.OPERATE_FAIL, null);
  }

  public File getFile(Context context) {
    File path = context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS);
    if (path != null) {
      File[] files = path.listFiles();
      if (files != null && files.length > 0) {
        for (File file : files) {
          if (file.getName().equals(serverVersion)) { //找到固件
            return file;
          }
        }
      }
    }
    return null;
  }

  public static boolean checkExist(Context context, String version) {
    File path = context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS);
    if (path != null) {
      File[] files = path.listFiles();
      if (files != null && files.length > 0) {
        for (File file : files) {
          if (file.getName().equals(version)) { //已经存在固件
            return true;
          }
        }
      }
    }
    return false;
  }
}
