package cn.lollypop.android.thermometer.ble.action;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattService;
import android.content.Context;
import android.os.Handler;
import com.basic.util.Callback;
import com.orhanobut.logger.Logger;

import cn.lollypop.android.thermometer.ble.BleAutoConnect;
import cn.lollypop.android.thermometer.ble.BleCallback;
import cn.lollypop.android.thermometer.ble.Constants;
import cn.lollypop.android.thermometer.ble.action.request.BleRequest;
import cn.lollypop.android.thermometer.ble.model.AlarmTimeModel;

public class BleActionManager {
  private static final int RE_DO_ACTION_MAX = 3;

  private BleRequest curBleRequest;
  private int reDoActionCount;

  private Context context;
  private final List<BleRequest> bleRequestList = new ArrayList<>();
  private final BleGattInteraction bleGattInteraction =
      new BleGattInteraction();
  private final Map<UUID, BluetoothGattCharacteristic> characteristics =
      new HashMap<>();
  private final Handler timeoutHandler = new Handler();

  private IBleAction bleAction;
  private BleAutoConnect.IResponse response;
  private BluetoothGatt bluetoothGatt;

  public BleActionManager(Context service,
                          BleAutoConnect.IResponse response,
                          IBleAction bleAction) {
    this.context = service;
    this.response = response;
    this.bleAction = bleAction;
  }

  public void start(BluetoothGatt bluetoothGatt,
                    boolean switchOriginalTemperature,
                    Callback callback) {
    this.bluetoothGatt = bluetoothGatt;
    handlerGattServices(bluetoothGatt.getServices(), callback);
    clearRequest();
    bleRequestList.addAll(
        bleAction.initConnect(context, switchOriginalTemperature)
    );
    doAction();
    timeoutHandler.postDelayed(timeoutRunnable, 60000);
  }

  private Runnable timeoutRunnable = new Runnable() {
    @Override
    public void run() {
      clearRequest();
      response.refreshView(BleCallback.BleStatus.OPERATE_FAIL);
    }
  };

  private void handlerGattServices(List<BluetoothGattService> gattServices,
                                   Callback callback) {
    if (gattServices == null) {
      return;
    }

    characteristics.clear();
    for (BluetoothGattService gattService : gattServices) {
      if (Constants.UUID_HEALTH_THERMOMETER_SERVICE.equals(
          gattService.getUuid()) ||
          Constants.UUID_BATTERY_SERVICE.equals(gattService.getUuid()) ||
          Constants.UUID_CUSTOM_SERVICE.equals(gattService.getUuid()) ||
          Constants.UUID_CURRENT_TIME_SERVICE.equals(gattService.getUuid()) ||
          Constants.UUID_DEVICE_INFORMATION_SERVICE.equals(
              gattService.getUuid()) ||
          Constants.UUID_REFERENCE_TIME_UPDATE_SERVICE.equals(
              gattService.getUuid())) {

        for (BluetoothGattCharacteristic characteristic : gattService
            .getCharacteristics()) {
          characteristics.put(characteristic.getUuid(), characteristic);
        }
      }
      callback.doCallback(true, gattService);
    }
  }

  public void end() {
    clearRequest();
  }

  public void clearRequest() {
    bleRequestList.clear();
    curBleRequest = null;
    reDoActionCount = 0;
  }

  public void startMeasure(boolean switchOriginalTemperature) {
    bleRequestList.addAll(
        bleAction.startMeasure(switchOriginalTemperature)
    );
    doAction();
  }

  public void setAlarm(AlarmTimeModel alarmTimeModel) {
    bleRequestList.addAll(bleAction.setAlarm(alarmTimeModel));
    doAction();
  }

  public void getBatteryLevel() {
    bleRequestList.addAll(bleAction.getBatteryLevel());
    doAction();
  }

  public void doOta() {
    bleRequestList.addAll(bleAction.doOta());
    doAction();
  }

  public void makeDeviceSleep() {
    bleRequestList.addAll(bleAction.makeDeviceSleep());
    doAction();
  }

  public void enterDebug() {
    bleRequestList.addAll(bleAction.enterDebugMode());
    doAction();
  }

  public void quitDebug() {
    bleRequestList.addAll(bleAction.quitDebugMode());
    doAction();
  }

  public void triggerBeep() {
    bleRequestList.addAll(bleAction.triggerBeep());
    doAction();
  }

  public void doNextAction() {
    reDoActionCount = 0;
    curBleRequest = null;
    doAction();
  }

  public void reDoAction() {
    reDoActionCount++;
    doAction();
  }

  private void doAction() {
    if (reDoActionCount > 0) {
      if (reDoActionCount >= RE_DO_ACTION_MAX) { //超出重试次数
        clearRequest();
        response.refreshView(BleCallback.BleStatus.OPERATE_FAIL);
        Logger.w("ble connect failed, over reDoAction times");
        return;
      }
      if (curBleRequest != null) {
        Logger.w("ble reDoAction : " + curBleRequest.getCharacteristicUUID());
        doInteractive(curBleRequest);
        return;
      }
    }
    if (curBleRequest != null) {
      return;
    }

    doInteractive(getAction());
  }

  private BleRequest getAction() {
    if (bleRequestList.size() > 0) {
      return bleRequestList.remove(0);
    }

    timeoutHandler.removeCallbacks(timeoutRunnable);
    return null;
  }

  private void doInteractive(BleRequest bleRequest) {
    curBleRequest = bleRequest;
    if (curBleRequest == null) {
      return;
    }

    boolean result = bleGattInteraction.doInteractive(
        bluetoothGatt, curBleRequest, characteristics, response
    );
    if (result) {
      return;
    }

    doNextAction();
  }

  public BleRequest getCurBleRequest() {
    return curBleRequest;
  }
}