package cn.lollypop.android.thermometer.ble.action;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import android.bluetooth.BluetoothGattDescriptor;
import android.content.Context;
import com.basic.util.TimeUtil;

import cn.lollypop.android.thermometer.ble.Constants;
import cn.lollypop.android.thermometer.ble.action.request.AlarmReadRequest;
import cn.lollypop.android.thermometer.ble.action.request.AlarmWriteRequest;
import cn.lollypop.android.thermometer.ble.action.request.BatteryRequest;
import cn.lollypop.android.thermometer.ble.action.request.BleRequest;
import cn.lollypop.android.thermometer.ble.action.request.CurrentTimeReadRequest;
import cn.lollypop.android.thermometer.ble.action.request.CurrentTimeWriteRequest;
import cn.lollypop.android.thermometer.ble.action.request.CurrentVoltageRequest;
import cn.lollypop.android.thermometer.ble.action.request.DebugInfoRequest;
import cn.lollypop.android.thermometer.ble.action.request.EnterDebugModeRequest;
import cn.lollypop.android.thermometer.ble.action.request.HardwareRevisionRequest;
import cn.lollypop.android.thermometer.ble.action.request.IndoorTemperatureRequest;
import cn.lollypop.android.thermometer.ble.action.request.MakeDeviceSleepRequest;
import cn.lollypop.android.thermometer.ble.action.request.QuitDebugModeRequest;
import cn.lollypop.android.thermometer.ble.action.request.ReadDeviceTimeRequest;
import cn.lollypop.android.thermometer.ble.action.request.RefreshAlarmRequest;
import cn.lollypop.android.thermometer.ble.action.request.RefreshCurrentTimeRequest;
import cn.lollypop.android.thermometer.ble.action.request.RefreshCurrentVoltageRequest;
import cn.lollypop.android.thermometer.ble.action.request.RefreshDebugInfoRequest;
import cn.lollypop.android.thermometer.ble.action.request.RefreshIndoorTemperature;
import cn.lollypop.android.thermometer.ble.action.request.SNRevisionRequest;
import cn.lollypop.android.thermometer.ble.action.request.SampleIndicationRequest;
import cn.lollypop.android.thermometer.ble.action.request.SampleNotificationRequest;
import cn.lollypop.android.thermometer.ble.action.request.SoftwareRevisionRequest;
import cn.lollypop.android.thermometer.ble.action.request.TemperatureIndicationRequest;
import cn.lollypop.android.thermometer.ble.action.request.TemperatureIntervalRequest;
import cn.lollypop.android.thermometer.ble.action.request.TemperatureNotificationRequest;
import cn.lollypop.android.thermometer.ble.action.request.TriggerBeepRequest;
import cn.lollypop.android.thermometer.ble.model.AlarmTimeModel;
import cn.lollypop.android.thermometer.ble.utils.AlarmTimeServiceUtil;
import cn.lollypop.android.thermometer.ble.utils.DeviceInformationServiceUtil;

/**
 * Copyright (c) 2015, Bongmi
 * All rights reserved
 * Author: wangjunjie@bongmi.com
 */
public class BleActionImpl implements IBleAction {

  @Override
  public List<BleRequest> initConnect(Context context,
                                      boolean switchOriginalTemperature) {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new SoftwareRevisionRequest());
    if (DeviceInformationServiceUtil.canGetSN(context)) {
      bleRequestList.add(new HardwareRevisionRequest());
      bleRequestList.add(new SNRevisionRequest());
    }
    bleRequestList.addAll(addUpdateTime());
    if (DeviceInformationServiceUtil.canOpenAlarm(context)) {
      bleRequestList.addAll(addAlarmAction(AlarmTimeModel.getSingle(context)));
    }
    if (DeviceInformationServiceUtil.canGetDebugInfo(context)) {
      bleRequestList.add(new RefreshDebugInfoRequest());
      bleRequestList.add(new DebugInfoRequest());

      bleRequestList.add(new RefreshCurrentVoltageRequest());
      bleRequestList.add(new CurrentVoltageRequest());

      bleRequestList.add(new RefreshIndoorTemperature());
      bleRequestList.add(new IndoorTemperatureRequest());
    }
    bleRequestList.add(new BatteryRequest());
    bleRequestList.addAll(addStartMeasureAction(switchOriginalTemperature));
    return bleRequestList;
  }

  @Override
  public List<BleRequest> setAlarm(AlarmTimeModel alarmTimeModel) {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.addAll(addAlarmAction(alarmTimeModel));
    return bleRequestList;
  }

  @Override
  public List<BleRequest> startMeasure(boolean switchOriginalTemperature) {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.addAll(addStartMeasureAction(switchOriginalTemperature));
    return bleRequestList;
  }

  @Override
  public List<BleRequest> getBatteryLevel() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new BatteryRequest());
    return bleRequestList;
  }

  @Override
  public List<BleRequest> doOta() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.addAll(addEndSampleDataAction());
    bleRequestList.addAll(addEndMeasureAction());
    return bleRequestList;
  }

  @Override
  public List<BleRequest> makeDeviceSleep() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new MakeDeviceSleepRequest());
    return bleRequestList;
  }

  @Override
  public List<BleRequest> enterDebugMode() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new EnterDebugModeRequest());
    return bleRequestList;
  }

  @Override
  public List<BleRequest> quitDebugMode() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new QuitDebugModeRequest());
    return bleRequestList;
  }

  @Override
  public List<BleRequest> triggerBeep() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new TriggerBeepRequest());
    return bleRequestList;
  }

  protected List<BleRequest> addStartMeasureAction(boolean switchFlag) {
    List<BleRequest> bleRequestList = new ArrayList<>();
    if (switchFlag) {
      bleRequestList.addAll(addStartSampleDataAction());
    }
    //设置notification
    bleRequestList.add(new TemperatureNotificationRequest(true));
    //打开indication
    bleRequestList.add(new TemperatureIndicationRequest(
        BluetoothGattDescriptor.ENABLE_INDICATION_VALUE));
    //设置interval
    bleRequestList.add(new TemperatureIntervalRequest(Constants.INTERVAL, 0));
    return bleRequestList;
  }

  private List<BleRequest> addStartSampleDataAction() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    //设置notification
    bleRequestList.add(new SampleNotificationRequest(true));
    //打开indication
    bleRequestList.add(new SampleIndicationRequest(
        BluetoothGattDescriptor.ENABLE_INDICATION_VALUE));
    return bleRequestList;
  }

  private List<BleRequest> addEndMeasureAction() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    //关闭interval
    bleRequestList.add(new TemperatureIntervalRequest(0, 0));
    //关闭notification
    bleRequestList.add(new TemperatureNotificationRequest(false));
    //关闭indication
    bleRequestList.add(new TemperatureIndicationRequest(
        BluetoothGattDescriptor.DISABLE_NOTIFICATION_VALUE));
    return bleRequestList;
  }

  private List<BleRequest> addEndSampleDataAction() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    bleRequestList.add(new SampleNotificationRequest(false));
    bleRequestList.add(new SampleIndicationRequest(
        BluetoothGattDescriptor.DISABLE_NOTIFICATION_VALUE));
    return bleRequestList;
  }

  protected List<BleRequest> addAlarmAction(AlarmTimeModel alarmTimeModel) {
    List<BleRequest> bleRequestList = new ArrayList<>();
    byte[] value = AlarmTimeServiceUtil.generateSendData(alarmTimeModel);
    bleRequestList.add(new AlarmWriteRequest(value));
    //刷新闹钟
    bleRequestList.add(new RefreshAlarmRequest());
    //读取闹钟
    bleRequestList.add(new AlarmReadRequest(value));
    return bleRequestList;
  }

  protected List<BleRequest> addUpdateTime() {
    List<BleRequest> bleRequestList = new ArrayList<>();
    //read original device time
    bleRequestList.add(new RefreshCurrentTimeRequest());
    bleRequestList.add(new ReadDeviceTimeRequest());

    //写入时间
    int currentTime = TimeUtil.getTimestamp(
        Calendar.getInstance().getTimeInMillis()
    );
    bleRequestList.add(new CurrentTimeWriteRequest(currentTime));
    //刷新时间
    bleRequestList.add(new RefreshCurrentTimeRequest());
    //读取时间
    bleRequestList.add(new CurrentTimeReadRequest(currentTime));
    return bleRequestList;
  }
}