package cn.lollypop.android.thermometer.ble;

import java.util.ArrayList;
import java.util.List;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.os.Build;
import android.os.Handler;
import android.text.TextUtils;
import com.basic.thread.LollypopHandler;
import com.basic.thread.LollypopThread;
import com.basic.thread.LollypopThreadPool;
import com.orhanobut.logger.Logger;

/**
 * Copyright (c) 2015, Bongmi
 * All rights reserved
 * Author: wangjunjie@bongmi.com
 */
public class BleScan {

  private BleCallback scanCallback;
  private boolean isScanning;
  private BluetoothAdapter bluetoothAdapter;

  private static final int SCAN_TIMEOUT = 4000;
  private final Handler scanHandler = new Handler();
  private final LollypopHandler threadHandler;
  private ScanCallback bleScanCallback;
  private BluetoothDevice nearestDevice;
  private int lowestRSSI;
  private String deviceName;

  public BleScan(BluetoothAdapter bluetoothAdapter, String deviceName) {
    this.bluetoothAdapter = bluetoothAdapter;
    this.deviceName = deviceName;
    threadHandler = LollypopThreadPool.getInstance().get(
        LollypopThread.BLE_SCAN);
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      bleScanCallback = new ScanCallback() {
        @TargetApi(Build.VERSION_CODES.LOLLIPOP)
        @Override
        public void onScanResult(int callbackType, ScanResult result) {
          handlerResult(result.getDevice(), result.getRssi());
        }

        @TargetApi(Build.VERSION_CODES.LOLLIPOP)
        @Override
        public void onBatchScanResults(List<ScanResult> results) {
          for (ScanResult sr : results) {
            Logger.i("ScanResult - Results : " + sr.toString());
          }
        }

        @Override
        public void onScanFailed(int errorCode) {
          Logger.e("Scan Failed", "Error Code: " + errorCode);
        }
      };
    }
  }

  public void doScan(BleCallback scanCallback) {
    this.scanCallback = scanCallback;

    Logger.i("begin scan ble ...");
    if (isScanning) {
      Logger.w("it is scanning, directly return");
      return;
    }

    isScanning = true;

    lowestRSSI = -999;
    nearestDevice = null;
    scanHandler.postDelayed(scanRunnable, SCAN_TIMEOUT);

    threadHandler.post(new Runnable() {
      @Override
      public void run() {
        Logger.i("SDK_INT = " + Build.VERSION.SDK_INT);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
          ScanSettings settings = new ScanSettings.Builder()
              .setScanMode(ScanSettings.SCAN_MODE_LOW_LATENCY)
              .build();
          List<ScanFilter> filters = new ArrayList<>();
          bluetoothAdapter.getBluetoothLeScanner().startScan(
              filters, settings, bleScanCallback
          );
        } else {
          boolean b = bluetoothAdapter.startLeScan(leScanCallback);
          Logger.i("ble startScan ： " + b);
        }
      }
    });
  }

  private final Runnable scanRunnable = new Runnable() {
    @Override
    public void run() {
      Logger.i("ble scan timeout after 4 seconds");
      if (nearestDevice != null) { //已经有扫描到设备
        Logger.i("find device address : " + nearestDevice.getAddress());
        stopScan();
        scanCallback.callback(BleCallback.BleStatus.SCAN_SUC, nearestDevice);
      } else { //没扫描到设备，继续扫
        if (isScanning) {
          scanHandler.postDelayed(this, SCAN_TIMEOUT);
        } else {
          Logger.i("scan failed");
          scanHandler.removeCallbacks(this);
          scanCallback.callback(BleCallback.BleStatus.SCAN_FAIL, null);
        }
      }
    }
  };

  public void stopScan() {
    if (isScanning) {
      Logger.i("ble stop scan");
      isScanning = false;
      threadHandler.post(new Runnable() {
        @Override
        public void run() {
          if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            if (bluetoothAdapter == null ||
                bluetoothAdapter.getBluetoothLeScanner() == null) {
              return;
            }

            bluetoothAdapter.getBluetoothLeScanner().stopScan(bleScanCallback);
          } else {
            bluetoothAdapter.stopLeScan(leScanCallback);
          }
        }
      });
    }
  }

  // Device scan scanCallback.
  private final BluetoothAdapter.LeScanCallback leScanCallback =
      new BluetoothAdapter.LeScanCallback() {

        @Override
        public void onLeScan(BluetoothDevice device,
                             int rssi,
                             byte[] scanRecord) {
          handlerResult(device, rssi);
        }
      };

  private void handlerResult(BluetoothDevice device, int rssi) {
    if (!isScanning) { //已经停止扫描
      return;
    }

    Logger.i("ble find devices : " + device.getName()
        + ", " + device.getAddress() + ", " + device.getType() + ", " + rssi);

    if (BluetoothDevice.DEVICE_TYPE_LE == device.getType()
        && !TextUtils.isEmpty(device.getName())
        && device.getName().toUpperCase().contains(deviceName)) { //扫描到棒米设备

      if (rssi > lowestRSSI) { //距离更近
        lowestRSSI = rssi;
        nearestDevice = device;
      }
    }
  }
}