package com.basic.util;

import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.BitmapFactory.Options;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorMatrix;
import android.graphics.ColorMatrixColorFilter;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PixelFormat;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.media.ExifInterface;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

public class BitmapUtil {

  /**
   * 按比例压缩图片
   *
   * @param image
   * @param hh    高度 800f
   * @param ww    宽度 480
   * @return
   */
  public static Bitmap comp(Bitmap image, float hh, float ww, boolean autoCompress) {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    if (autoCompress) {
      long s = image.getRowBytes() * image.getHeight();
      float news = hh * ww;
      if (image.getConfig().equals(Config.ALPHA_8)) {
        news *= 1;
      } else if (image.getConfig().equals(Config.ARGB_4444)) {
        news *= 2;
      } else if (image.getConfig().equals(Config.ARGB_8888)) {
        news *= 4;
      } else if (image.getConfig().equals(Config.RGB_565)) {
        news *= 2;
      }
      //		long sizekb = s / 10240; //kb为单位
      if (s > news) { //进行压缩避免在生成图片（BitmapFactory.decodeStream）时溢出
        int rate = (int) (100 * news / s);
        if (rate < 20) {
          rate = 20;
        }
        Log.i("压缩图片", "压缩比：" + rate);
        image.compress(CompressFormat.JPEG, rate, baos); //把压缩后的数据存放到baos中
      } else {
        Log.i("压缩图片", "不压缩");
        image.compress(CompressFormat.JPEG, 100, baos); //100表示不压缩
      }
    } else {
      image.compress(CompressFormat.JPEG, 100, baos); //100表示不压缩
    }
    Options newOpts = new Options();
    newOpts.inJustDecodeBounds = true;
    ByteArrayInputStream isBm = new ByteArrayInputStream(baos.toByteArray());
    //开始读入图片，此时把options.inJustDecodeBounds 设回true了
    //		BitmapFactory.decodeFile(filePath, options);
    Bitmap bitmap = BitmapFactory.decodeStream(isBm, null, newOpts);
    int w = newOpts.outWidth;
    int h = newOpts.outHeight;
    //缩放比。由于是固定比例缩放，只用高或者宽其中一个数据进行计算即可
    int be = 1;//be=1表示不缩放
    if (w > ww) {//如果宽度大的话根据宽度固定大小缩放
      be = (int) (newOpts.outWidth / ww);
    } else if (h > hh) {//如果高度高的话根据高度固定大小缩放
      be = (int) (newOpts.outHeight / hh);
    }
    Log.i("压缩图片", "缩放比例：" + be);
    if (be <= 0)
      be = 1;
    newOpts.inSampleSize = be;//设置缩放比例
    newOpts.inJustDecodeBounds = false;
    //重新读入图片，注意此时已经把options.inJustDecodeBounds 设回false了
    isBm = new ByteArrayInputStream(baos.toByteArray());
    //		BitmapFactory.decodeFile(filePath, options);
    bitmap = BitmapFactory.decodeStream(isBm, null, newOpts);
    return bitmap;
  }

  /**
   * 质量压缩
   *
   * @param image
   * @param size  以kb为单位，压缩的大小
   * @return
   */
  public static Bitmap compressImage(Bitmap image, int size) {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    image.compress(CompressFormat.JPEG, 100, baos);//质量压缩方法，这里100表示不压缩，把压缩后的数据存放到baos中
    int options = 100;
    while (baos.toByteArray().length / 1024 > size) {  //循环判断如果压缩后图片是否大于500kb,大于继续压缩
      if (options > 10) {
        baos.reset();//重置baos即清空baos
        options -= 10;//每次都减少10
        image.compress(CompressFormat.JPEG, options, baos);//这里压缩options%，把压缩后的数据存放到baos中
      } else {
        break;
      }
    }
    ByteArrayInputStream isBm = new ByteArrayInputStream(baos.toByteArray());//把压缩后的数据baos存放到ByteArrayInputStream中
    try {
      Bitmap bitmap = BitmapFactory.decodeStream(isBm, null, null);//把ByteArrayInputStream数据生成图片
      return bitmap;
    } catch (OutOfMemoryError e) {
      baos.reset();
      image.compress(CompressFormat.JPEG, 10, baos);
      isBm = new ByteArrayInputStream(baos.toByteArray());
      try {
        Bitmap bitmap = BitmapFactory.decodeStream(isBm, null, null);//把ByteArrayInputStream数据生成图片
        return bitmap;
      } catch (OutOfMemoryError e2) {
        return image;
      }
    }
  }

  /**
   * 按像素大小缩放图片，并做相应的旋转
   *
   * @param path   图片路径
   * @param width
   * @param height
   * @return
   */
  public static Bitmap fitSizeImg(String path, int width, int height) {
    if (path == null || path.length() < 1) return null;
    Options opts = new Options();
    opts.inJustDecodeBounds = true;
    Bitmap resizeBmp = BitmapFactory.decodeFile(path, opts);
    int picWidth = opts.outWidth;
    int picHeight = opts.outHeight;
    opts.inSampleSize = 1;
    if (width < picWidth) {
      opts.inSampleSize = picWidth / width;
    } else if (height < picHeight) {
      opts.inSampleSize = picHeight / height;
    }
    opts.inJustDecodeBounds = false;
    resizeBmp = BitmapFactory.decodeFile(path, opts);
    //被旋转的角度
    int angle = readPictureDegree(path);
    //旋转回来
    resizeBmp = rotationBitmap(angle, resizeBmp);
    return resizeBmp;
  }

  /**
   * 读取图片属性：旋转的角度
   *
   * @param path 图片绝对路径
   * @return degree旋转的角度
   */
  public static int readPictureDegree(String path) {
    int degree = 0;
    try {
      ExifInterface exifInterface = new ExifInterface(path);
      int orientation = exifInterface.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
      switch (orientation) {
        case ExifInterface.ORIENTATION_ROTATE_90:
          degree = 90;
          break;
        case ExifInterface.ORIENTATION_ROTATE_180:
          degree = 180;
          break;
        case ExifInterface.ORIENTATION_ROTATE_270:
          degree = 270;
          break;
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
    return degree;
  }

  /**
   * 旋转图片
   *
   * @param angle  旋转角度
   * @param bitmap 要旋转的bitmap
   * @return Bitmap
   */
  public static Bitmap rotationBitmap(int angle, Bitmap bitmap) {
    //旋转图片 动作
    Matrix matrix = new Matrix();
    matrix.postRotate(angle);
    // 创建新的图片
    try {
      Bitmap resizedBitmap = Bitmap.createBitmap(bitmap, 0, 0,
          bitmap.getWidth(), bitmap.getHeight(), matrix, true);
      return resizedBitmap;
    } catch (OutOfMemoryError e) {
      return bitmap;
    }
  }

  /**
   * 获取截屏
   *
   * @param activity
   * @return
   */
  public static Bitmap takeScreenShot(Activity activity) {
    View view = activity.getWindow().getDecorView();
    view.setDrawingCacheEnabled(true);
    view.buildDrawingCache();
    Bitmap bitmap = view.getDrawingCache();
    Rect rect = new Rect();
    activity.getWindow().getDecorView().getWindowVisibleDisplayFrame(rect);
    int statusBarHeight = rect.top;

    int width = activity.getWindowManager().getDefaultDisplay().getWidth();
    int height = activity.getWindowManager().getDefaultDisplay().getHeight();

    Bitmap bitmap2 = Bitmap.createBitmap(bitmap, 0, statusBarHeight, width, height - statusBarHeight);
    view.destroyDrawingCache();
    return bitmap2;
  }

  /**
   * 保存图片
   *
   * @param bitmap
   * @param filename
   */
  public static void savePic(Bitmap bitmap, String filename, CompressFormat format, int quality) {
    FileOutputStream fileOutputStream = null;
    try {
      fileOutputStream = new FileOutputStream(filename);
      if (fileOutputStream != null) {
        bitmap.compress(format, quality, fileOutputStream);
        fileOutputStream.flush();
        fileOutputStream.close();
      }
    } catch (FileNotFoundException e) {
      Log.d("BitmapUtil", "Exception:FileNotFoundException");
      e.printStackTrace();
    } catch (IOException e) {
      Log.d("BitmapUtil", "IOException:IOException");
      e.printStackTrace();
    }
  }

  /**
   * 保存图片，默认是png，质量降低到90
   *
   * @param bitmap
   * @param filename
   */
  public static void savePic(Bitmap bitmap, String filename) {
    savePic(bitmap, filename, CompressFormat.PNG, 90);
  }

  /**
   * 通过Drawable转成bitmap
   */
  public static Bitmap drawableToBitmap(Drawable drawable) {

    Bitmap bitmap = Bitmap.createBitmap(
        drawable.getIntrinsicWidth(),
        drawable.getIntrinsicHeight(),
        drawable.getOpacity() != PixelFormat.OPAQUE ? Config.ARGB_8888
            : Config.RGB_565);

    Canvas canvas = new Canvas(bitmap);
    drawable.setBounds(0, 0, drawable.getIntrinsicWidth(), drawable.getIntrinsicHeight());
    drawable.draw(canvas);

    return bitmap;
  }

  /**
   * 加载本地图片
   *
   * @param url："/sdcard/tupian.jpg"
   * @return
   */
  public static Bitmap getLocalBitmap(String url) {
    if (TextUtils.isEmpty(url)) {
      return Bitmap.createBitmap(100, 100, Config.ARGB_8888);
    }
    try {
      Bitmap bitmap;

      Options optSize = new Options();
      optSize.inJustDecodeBounds = true;

      BitmapFactory.decodeFile(url, optSize);
      int imgSize = optSize.outWidth * optSize.outHeight;
      if (imgSize > 1500000) {
        Options thOption = new Options();
        thOption.inSampleSize = 4;
        bitmap = BitmapFactory.decodeFile(url, thOption);
      } else {
        bitmap = BitmapFactory.decodeFile(url);
      }
      return bitmap;
    } catch (OutOfMemoryError e) {
      e.printStackTrace();
      return Bitmap.createBitmap(100, 100, Config.ARGB_8888);
    }
  }

  /**
   * 缩放drawable
   *
   * @param drawable
   * @param w
   * @param h
   * @return
   */
  public static Drawable zoomDrawable(Drawable drawable, int w, int h) {
    int width = drawable.getIntrinsicWidth();
    int height = drawable.getIntrinsicHeight();
    Bitmap oldbmp = drawableToBitmap(drawable); // drawable 转换成 bitmap
    Matrix matrix = new Matrix();   // 创建操作图片用的 Matrix 对象
    float scaleWidth = ((float) w / width);   // 计算缩放比例
    float scaleHeight = ((float) h / height);
    matrix.postScale(scaleWidth, scaleHeight);         // 设置缩放比例
    Bitmap newbmp = Bitmap.createBitmap(oldbmp, 0, 0, width, height, matrix, true);       // 建立新的 bitmap ，其内容是对原 bitmap 的缩放后的图
    return new BitmapDrawable(newbmp);       // 把 bitmap 转换成 drawable 并返回
  }

  public static Bitmap zoomBitmap(Bitmap bmp, float w, float h) {
    int width = bmp.getWidth();
    int height = bmp.getHeight();
    Matrix matrix = new Matrix();   // 创建操作图片用的 Matrix 对象
    float scaleWidth = (w / width);   // 计算缩放比例
    float scaleHeight = (h / height);
    matrix.postScale(scaleWidth, scaleHeight);         // 设置缩放比例
    return Bitmap.createBitmap(bmp, 0, 0, width, height, matrix, true);       // 把 bitmap 转换成 drawable 并返回
  }

  /**
   * 将view做成bitmap
   *
   * @param v
   * @return
   */
  public static Bitmap getViewBitmap(View v) {
    v.clearFocus();
    v.setPressed(false);
    boolean willNotCache = v.willNotCacheDrawing();
    v.setWillNotCacheDrawing(false);
    // Reset the drawing cache background color to fully transparent
    // for the duration of this operation
    int color = v.getDrawingCacheBackgroundColor();
    v.setDrawingCacheBackgroundColor(0);
    if (color != 0) {
      v.destroyDrawingCache();
    }
    v.buildDrawingCache();
    Bitmap cacheBitmap = v.getDrawingCache();
    if (cacheBitmap == null) {
      return null;
    }
    Bitmap bitmap = Bitmap.createBitmap(cacheBitmap);
    v.destroyDrawingCache();
    v.setWillNotCacheDrawing(willNotCache);
    v.setDrawingCacheBackgroundColor(color);
    return bitmap;
  }

  // 将view转为bitmap
  public static Bitmap getBitmapFromView(View view) {
    // Define a bitmap with the same size as the view
    Bitmap returnedBitmap = Bitmap.createBitmap(view.getWidth(), view.getHeight(), Bitmap.Config.ARGB_8888);
    // Bind a canvas to it
    Canvas canvas = new Canvas(returnedBitmap);
    // Get the view's background
    Drawable bgDrawable = view.getBackground();
    if (bgDrawable != null)
      // has background drawable, then draw it on the canvas
      bgDrawable.draw(canvas);
    else
      // does not have background drawable, then draw white background on
      // the canvas
      canvas.drawColor(Color.WHITE);
    // draw the view on the canvas
    view.draw(canvas);
    // return the bitmap
    return returnedBitmap;
  }

  public static Bitmap createScaledBitmap(Bitmap image, float hh, float ww, int size, CompressFormat format) {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    int reduce = 100;
    while (reduce > 0) {
      try {
        //质量压缩方法，这里100表示不压缩，把压缩后的数据存放到baos中
        image.compress(format, reduce, baos);
        break;
      } catch (Exception e) {
        baos.reset();
        reduce -= 5;
      }
    }
    Options newOpts = new Options();
    newOpts.inJustDecodeBounds = true;
    ByteArrayInputStream isBm = new ByteArrayInputStream(baos.toByteArray());
    BitmapFactory.decodeStream(isBm, null, newOpts);
    int w = newOpts.outWidth;
    int h = newOpts.outHeight;
    //缩放比。由于是固定比例缩放，只用高或者宽其中一个数据进行计算即可
    int be = 1;//be=1表示不缩放
    if (w > ww) {//如果宽度大的话根据宽度固定大小缩放
      be = (int) (newOpts.outWidth / ww);
    } else if (h > hh) {//如果高度高的话根据高度固定大小缩放
      be = (int) (newOpts.outHeight / hh);
    }
    if (be <= 0) {
      be = 1;
    }
    Log.i("缩放图片", "缩放比例：" + be);

    Log.i("压缩图片", "当前kb：" + baos.toByteArray().length / 1024);
    int options = 100;
    while (baos.toByteArray().length / 1024 > size) {  //循环判断如果压缩后图片是否大于size的kb,大于则继续压缩
      if (options > 10) {
        baos.reset();//重置baos即清空baos
        options -= 5;//每次都减少5
        image.compress(format, options, baos);//这里压缩options%，把压缩后的数据存放到baos中
      } else {
        break;
      }
    }
    Log.i("压缩图片", "压缩后kb：" + baos.toByteArray().length / 1024);

    newOpts.inSampleSize = be;//设置缩放比例
    newOpts.inJustDecodeBounds = false;
    //重新读入图片，注意此时已经把options.inJustDecodeBounds 设回false了
    isBm = new ByteArrayInputStream(baos.toByteArray());
    return BitmapFactory.decodeStream(isBm, null, newOpts);
  }


  /**
   * 转换图片成圆形
   *
   * @param bitmap 传入Bitmap对象
   * @return
   */
  public static Bitmap toRoundBitmap(Bitmap bitmap) {
    if (bitmap == null) {
      return null;
    }
    int width = bitmap.getWidth();
    int height = bitmap.getHeight();
    float roundPx;
    float left, top, right, bottom, dst_left, dst_top, dst_right, dst_bottom;
    if (width <= height) {
      roundPx = width / 2;
      top = 0;
      bottom = width;
      left = 0;
      right = width;
      height = width;
      dst_left = 0;
      dst_top = 0;
      dst_right = width;
      dst_bottom = width;
    } else {
      roundPx = height / 2;
      float clip = (width - height) / 2;
      left = clip;
      right = width - clip;
      top = 0;
      bottom = height;
      width = height;
      dst_left = 0;
      dst_top = 0;
      dst_right = height;
      dst_bottom = height;
    }

    Bitmap output = null;
    try {
      output = Bitmap.createBitmap(width, height, Config.ARGB_8888);
    } catch (OutOfMemoryError e) {
      System.gc();
      System.runFinalization();
      output = Bitmap.createBitmap(width, height, Config.ARGB_8888);
    }
    Canvas canvas = new Canvas(output);

    final int color = 0xff424242;
    final Paint paint = new Paint();
    final Rect src = new Rect((int) left, (int) top, (int) right, (int) bottom);
    final Rect dst = new Rect((int) dst_left, (int) dst_top, (int) dst_right, (int) dst_bottom);
    final RectF rectF = new RectF(dst);

    paint.setAntiAlias(true);

    canvas.drawARGB(0, 0, 0, 0);
    paint.setColor(color);
    canvas.drawRoundRect(rectF, roundPx, roundPx, paint);

    paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_IN));
    canvas.drawBitmap(bitmap, src, dst, paint);
    return output;
  }

  public static Bitmap grey(Bitmap bitmap) {
    int width = bitmap.getWidth();
    int height = bitmap.getHeight();

    Bitmap faceIconGreyBitmap = Bitmap
        .createBitmap(width, height, Bitmap.Config.ARGB_8888);

    Canvas canvas = new Canvas(faceIconGreyBitmap);
    Paint paint = new Paint();
    ColorMatrix colorMatrix = new ColorMatrix();
    colorMatrix.setSaturation(0);
    ColorMatrixColorFilter colorMatrixFilter = new ColorMatrixColorFilter(
        colorMatrix);
    paint.setColorFilter(colorMatrixFilter);
    canvas.drawBitmap(bitmap, 0, 0, paint);
    return faceIconGreyBitmap;
  }
}
