package com.basic.controller;

import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import android.app.Activity;
import android.content.Context;
import android.content.SharedPreferences;
import com.basic.util.GsonUtil;
import com.basic.util.TimeUtil;
import com.google.gson.reflect.TypeToken;
import com.orhanobut.logger.Logger;

/**
 * Copyright (c) 2016, Bongmi
 * All rights reserved
 * Author: jianghao@bongmi.com
 */
public class SyncedDataManager {

  private static SyncedDataManager instance = new SyncedDataManager();

  private static final String SYNCED_MONTHS = "SYNCED_MONTHS";
  private static final String TEMPERATURE_STRING = "TEMPERATURE";
  private static final String BODY_STATUS_STRING = "BODY_STATUS";

  public synchronized static SyncedDataManager getInstance() {
    return instance;
  }

  public TimeSegment getRequestTime(Context context,
                                    int familyMemberId,
                                    SaveType saveType,
                                    Date startDate, Date endDate) {
    Calendar startCal = getCalendarDate(startDate);
    Calendar endCal = getCalendarDate(endDate);
    if (startCal.after(endCal)) {
      return null;
    }

    List<Calendar> hasSyncedMonths = getSyncedTime(
        context, familyMemberId, saveType);
    if (hasSyncedMonths == null || hasSyncedMonths.size() == 0) {
      return getTimeSegment(startCal, endCal);
    }

    Calendar lastMonth = hasSyncedMonths.get(hasSyncedMonths.size() - 1);
    Calendar firstMonth = hasSyncedMonths.get(0);

    if (!startCal.before(firstMonth) && !endCal.after(lastMonth)) {
      return null;
    }

    if (startCal.before(firstMonth) && endCal.after(lastMonth)) {
      return getTimeSegment(startCal, endCal);
    }

    if (startCal.before(firstMonth)) {
      return getTimeSegment(startCal, firstMonth);
    } else {
      return getTimeSegment(lastMonth, endCal);
    }
  }

  public TimeSegment getTimeSegment(Calendar startCal, Calendar endCal) {
    TimeSegment timeSegment = new TimeSegment();
    timeSegment.setStartTime(TimeUtil.getTimestamp(startCal.getTime()
        .getTime()));
    endCal.add(Calendar.MONTH, 1);
    int orgTime = TimeUtil.getTimestamp(endCal.getTime().getTime());
    timeSegment.setEndTime(orgTime - 1);
    return timeSegment;
  }

  public void setSyncedTime(Context context,
                            Date startDate, Date endDate,
                            int familyMemberId,
                            SaveType saveType) {
    Calendar startCal = getCalendarDate(startDate);
    Calendar endCal = getCalendarDate(endDate);
    if (startCal.after(endCal)) {
      return;
    }

    List<Calendar> hasSyncedMonths = getSyncedTime(
        context, familyMemberId, saveType);
    if (hasSyncedMonths == null) {
      hasSyncedMonths = new LinkedList<>();
      Logger.d("no synced data yet," + saveType);
    }

    while (!startCal.after(endCal)) {
      if (!hasSyncedMonths.contains(startCal)) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startCal.getTime());
        hasSyncedMonths.add(calendar);
      }
      startCal.add(Calendar.MONTH, 1);
    }

    String syncedMonths = GsonUtil.getGson().toJson(hasSyncedMonths);
    Logger.d("set synced months: " + syncedMonths);
    setSyncedMonths(context, syncedMonths, familyMemberId, saveType);
  }

  public List<Calendar> getSyncedTime(Context context,
                                      int familyMemberId,
                                      SaveType saveType) {
    String jsonString = getSyncedMonths(context, familyMemberId, saveType);
    Logger.d("get synced months: " + jsonString);

    List<Calendar> syncedMonths = GsonUtil.getGson().fromJson(jsonString,
        new TypeToken<List<Calendar>>() {}.getType());
    if (syncedMonths != null) {
      Collections.sort(syncedMonths);
    }
    return syncedMonths;
  }

  public Calendar getCalendarDate(Date date) {
    Calendar c = Calendar.getInstance();
    c.setTimeInMillis(TimeUtil.getDateBeginTimeInMillis(date.getTime()));
    c.set(Calendar.DAY_OF_MONTH, 1);
    return c;
  }

  public void setSyncedMonths(Context context,
                              String content,
                              int familyMemberId,
                              SaveType saveType) {
    String dataType = getDataType(saveType);
    SharedPreferences mySharedPreferences = context.getSharedPreferences(
        SYNCED_MONTHS + dataType + familyMemberId,
        Activity.MODE_PRIVATE
    );
    SharedPreferences.Editor editor = mySharedPreferences.edit();
    editor.putString(SYNCED_MONTHS, content);
    editor.apply();
  }

  public String getSyncedMonths(Context context,
                                int familyMemberId,
                                SaveType saveType) {
    String dataType = getDataType(saveType);
    SharedPreferences settings = context.getSharedPreferences(
        SYNCED_MONTHS + dataType + familyMemberId,
        Activity.MODE_PRIVATE
    );
    return settings.getString(SYNCED_MONTHS, "");
  }

  private String getDataType(SaveType saveType) {
    switch (saveType) {
      case BODY_STATUS:
        return BODY_STATUS_STRING;
      case TEMPERATURE:
        return TEMPERATURE_STRING;
      default:
        return "";
    }
  }

  public enum SaveType {
    TEMPERATURE(1),
    BODY_STATUS(2);

    private final int value;

    SaveType(int value) {
      this.value = value;
    }

    public int getValue() {
      return this.value;
    }
  }
}
