package cn.flyelf.cache.sample.controller;

import cn.flyelf.cache.annotation.CacheBean;
import cn.flyelf.cache.annotation.CacheConstant;
import cn.flyelf.cache.core.Cache;
import cn.flyelf.cache.core.model.CacheTestModel;
import cn.flyelf.cache.sample.util.ModelUtil;
import cn.flyelf.cache.spring.annotation.ConcurrencyApi;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Mono;

import java.util.concurrent.TimeUnit;

/**
 * 缓存接口例子
 *
 * @author wujr
 * 2019-12-24
 * @version 1.0
 * 变更历史
 * [wujr 2019-12-24 1.0 新增]
 */
@RestController
@RequestMapping("/cache/simple")
@Slf4j
@RequiredArgsConstructor
public class CacheSimpleController {
    @CacheBean(layer = {CacheConstant.LAYER_CAFFEINE})
    private Cache<String, CacheTestModel> cacheCaffeine;
    @CacheBean
    private Cache<String, CacheTestModel> cacheRedis;

    @GetMapping("/caffeine/put/{key}")
    @ConcurrencyApi(value = "#id", timeout = 1000000L)
    public Mono<CacheTestModel> putSimpleCaffeine(@PathVariable("key") String key,
                                                  @RequestParam(value = "duration", required = false) Long duration,
                                                  @RequestParam("id") long id,
                                                  @RequestParam("name") String name,
                                                  @RequestParam("index") Integer index){
        return putModel(key, duration, cacheCaffeine, id, name, index);
    }
    @GetMapping("/caffeine/get2/{key}")
    @ConcurrencyApi
    public CacheTestModel getSimpleCaffeine2(@PathVariable("key") String key){
        return cacheCaffeine.get(key).block();
    }
    @GetMapping("/caffeine/get/{key}")
    @ConcurrencyApi
    public Mono<CacheTestModel> getSimpleCaffeine(@PathVariable("key") String key){
        return cacheCaffeine.get(key);
    }

    private Mono<CacheTestModel> putModel(String key, Long duration,
                                          Cache<String, CacheTestModel> cache,
                                          long id,
                                          String name,
                                          Integer index){
        CacheTestModel model = ModelUtil.buildModel(id, name, index);
        Mono<Boolean> result = cache.put(key, model, duration, TimeUnit.SECONDS);
        return result.map(r -> model);
    }
    @GetMapping("/redis/put/{key}")
    public Mono<CacheTestModel> putRedis(@PathVariable("key") String key,
                                   @RequestParam("duration") Long duration,
                                   @RequestParam("id") long id,
                                   @RequestParam("name") String name,
                                   @RequestParam("index") Integer index){
        return putModel(key, duration, cacheRedis, id, name, index);
    }
    @GetMapping("/redis/get/{key}")
    public Mono<CacheTestModel> getRedis(@PathVariable("key") String key){
        return cacheRedis.get(key);
    }
    @GetMapping("/redis/block/{key}")
    public CacheTestModel blockRedis(@PathVariable("key") String key){
        Mono<CacheTestModel> result = cacheRedis.get(key);
        return result.block();
    }

    @GetMapping("/redis/load/{key}")
    public Mono<CacheTestModel> getRedisNullAndLoad(@PathVariable("key") String key,
                                                    @RequestParam(value = "id", required = false) Long id,
                                                    @RequestParam(value = "name", required = false) String name,
                                                    @RequestParam(value = "index", required = false) Integer index){
        CacheTestModel model = null;
        if (null != id){
            model = ModelUtil.buildModel(id, name, index);
        }
        CacheTestModel finalModel = model;
        return cacheRedis.get(key, k -> finalModel);
    }
}
