package cn.flyelf.cache.sample.controller;

import cn.flyelf.cache.annotation.CacheBean;
import cn.flyelf.cache.annotation.CacheConstant;
import cn.flyelf.cache.core.ListCache;
import cn.flyelf.cache.core.model.CacheTestModel;
import cn.flyelf.cache.sample.util.ModelUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.ArrayList;
import java.util.List;

/**
 * 缓存list的接口
 *
 * @author wujr
 * 2020/3/22
 * @version 1.0
 * 变更历史
 * [wujr 2020/3/22 1.0 新增]
 */
@RestController
@RequestMapping("/cache/list")
@Slf4j
@RequiredArgsConstructor
public class CacheListController {
    @CacheBean(layer = {CacheConstant.LAYER_CAFFEINE})
    private ListCache<String, CacheTestModel> cacheListCaffeine;
    @CacheBean
    private ListCache<String, CacheTestModel> cacheListRedis;

    @GetMapping("/caffeine/put/{key}")
    public Flux<CacheTestModel> putCaffeineList(@PathVariable("key") String key,
                                                @RequestParam("id") long[] id,
                                                @RequestParam("name") String[] name,
                                                @RequestParam("index") Integer[] index,
                                                @RequestParam(value = "duration", required = false) Long duration){
        return putList(cacheListCaffeine, key, id, name, index, duration);
    }
    private Flux<CacheTestModel> putList(ListCache<String, CacheTestModel> cache,
                                         String key, long[] id, String[] name, Integer[] index,
                                         Long duration){
        List<CacheTestModel> models = new ArrayList<>();
        return ModelUtil.putCollection(cache, models, key, id, name, index, duration);
    }
    @GetMapping("/caffeine/get/{key}")
    public Flux<CacheTestModel> getCaffeineList(@PathVariable("key") String key){
        Mono<List<CacheTestModel>> result = cacheListCaffeine.get(key);
        return result.flatMapMany(Flux::fromIterable);
    }
    @GetMapping("/caffeine/del/{key}")
    public Mono<Long> delCaffeineList(@PathVariable("key") String key){
        return cacheListCaffeine.delete(key);
    }
    @GetMapping("/caffeine/rpush/{key}")
    public Mono<Boolean> rpushCaffeineList(@PathVariable("key") String key,
                                        @RequestParam("id") long id,
                                        @RequestParam("name") String name,
                                        @RequestParam("index") Integer index){
        CacheTestModel model = ModelUtil.buildModel(id, name, index);
        return cacheListCaffeine.rpush(key, model);
    }
    @GetMapping("/caffeine/lpush/{key}")
    public Mono<Boolean> lpushCaffeineList(@PathVariable("key") String key,
                                        @RequestParam("id") long id,
                                        @RequestParam("name") String name,
                                        @RequestParam("index") Integer index){
        CacheTestModel model = ModelUtil.buildModel(id, name, index);
        return cacheListCaffeine.lpush(key, model);
    }
    @GetMapping("/caffeine/rpop/{key}")
    public Mono<CacheTestModel> rpopCaffeineList(@PathVariable("key") String key){
        return cacheListCaffeine.rpop(key);
    }
    @GetMapping("/caffeine/lpop/{key}")
    public Mono<CacheTestModel> lpopCaffeineList(@PathVariable("key") String key){
        return cacheListCaffeine.lpop(key);
    }


    @GetMapping("/redis/put/{key}")
    public Flux<CacheTestModel> putRedisList(@PathVariable("key") String key,
                                             @RequestParam("id") long[] id,
                                             @RequestParam("name") String[] name,
                                             @RequestParam("index") Integer[] index,
                                             @RequestParam(value = "duration", required = false) Long duration){
        return putList(cacheListRedis, key, id, name, index, duration);
    }
    @GetMapping("/redis/get/{key}")
    public Flux<CacheTestModel> getRedisList(@PathVariable("key") String key){
        Mono<List<CacheTestModel>> result = cacheListRedis.get(key);
        return result.flatMapMany(Flux::fromIterable);
    }
    @GetMapping("/redis/del/{key}")
    public Mono<Long> delRedisList(@PathVariable("key") String key){
        return cacheListRedis.delete(key);
    }
    @GetMapping("/redis/rpush/{key}")
    public Mono<Boolean> rpushRedisList(@PathVariable("key") String key,
                                        @RequestParam("id") long id,
                                        @RequestParam("name") String name,
                                        @RequestParam("index") Integer index){
        CacheTestModel model = ModelUtil.buildModel(id, name, index);
        return cacheListRedis.rpush(key, model);
    }
    @GetMapping("/redis/lpush/{key}")
    public Mono<Boolean> lpushRedisList(@PathVariable("key") String key,
                                        @RequestParam("id") long id,
                                        @RequestParam("name") String name,
                                        @RequestParam("index") Integer index){
        CacheTestModel model = ModelUtil.buildModel(id, name, index);
        return cacheListRedis.lpush(key, model);
    }
    @GetMapping("/redis/rpop/{key}")
    public Mono<CacheTestModel> rpopRedisList(@PathVariable("key") String key){
        return cacheListRedis.rpop(key);
    }
    @GetMapping("/redis/lpop/{key}")
    public Mono<CacheTestModel> lpopRedisList(@PathVariable("key") String key){
        return cacheListRedis.lpop(key);
    }
}
