package cn.flyelf.cache.sample.controller;

import cn.flyelf.cache.annotation.CacheBean;
import cn.flyelf.cache.annotation.CacheConstant;
import cn.flyelf.cache.core.HashCache;
import cn.flyelf.cache.core.model.CacheTestModel;
import cn.flyelf.cache.sample.util.ModelUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * hash数据类型的缓存接口
 *
 * @author wujr
 * 2020/3/22
 * @version 1.0
 * 变更历史
 * [wujr 2020/3/22 1.0 新增]
 */
@RestController
@RequestMapping("/cache/map")
@Slf4j
@RequiredArgsConstructor
public class CacheHashController {
    @CacheBean(layer = {CacheConstant.LAYER_CAFFEINE})
    private HashCache<String, Long, CacheTestModel> cacheMapCaffeine;
    @CacheBean
    private HashCache<String, Long, CacheTestModel> cacheMapRedis;

    private Mono<Boolean> addMap(HashCache<String, Long, CacheTestModel> cache,
                                 String key, long id, String name, Integer index){
        CacheTestModel model = ModelUtil.buildModel(id, name, index);
        return cache.add(key, id, model);
    }
    private Flux<CacheTestModel> putMap(HashCache<String, Long, CacheTestModel> cache,
                                        String key, long[] id, String[] name, Integer[] index,
                                        Long duration){
        Map<Long, CacheTestModel> models = new HashMap<>(id.length);
        for (int i = 0; i < id.length; ++ i){
            CacheTestModel model = ModelUtil.buildModel(i, id, name, index);
            models.put(model.getId(), model);
        }
        return cache.put(key, models, duration, TimeUnit.SECONDS).flatMapIterable(r -> models.values());
    }
    @GetMapping("/caffeine/put/{key}")
    public Flux<CacheTestModel> putCaffeineHash(@PathVariable("key") String key,
                                                @RequestParam("id") long[] id,
                                                @RequestParam("name") String[] name,
                                                @RequestParam("index") Integer[] index,
                                                @RequestParam(value = "duration", required = false) Long duration){
        return putMap(cacheMapCaffeine, key, id, name, index, duration);
    }
    @GetMapping("/caffeine/get/{key}")
    public Flux<CacheTestModel> getCaffeineHash(@PathVariable("key") String key){
        return cacheMapCaffeine.get(key).flatMapIterable(Map::values);
    }
    @GetMapping("/caffeine/add/{key}")
    public Mono<Boolean> addCaffeineHash(@PathVariable("key") String key,
                                         @RequestParam("id") long id,
                                         @RequestParam("name") String name,
                                         @RequestParam("index") Integer index){
        return addMap(cacheMapCaffeine, key, id, name, index);
    }
    @GetMapping("/caffeine/key/{key}/{id}")
    public CacheTestModel getCaffeineHashList(@PathVariable("key") String key,
                                              @PathVariable("id") long id){
        Mono<CacheTestModel> mono = cacheMapCaffeine.get(key, id);
        return mono.block();
    }
    @GetMapping("/caffeine/del/{key}")
    public Mono<Long> delCaffeineHash(@PathVariable("key") String key){
        return cacheMapCaffeine.delete(key);
    }
    @GetMapping("/caffeine/del/{key}/{id}")
    public Mono<Long> delCaffeineHashKey(@PathVariable("key") String key,
                                         @PathVariable("id") Long id){
        return cacheMapCaffeine.remove(key, id);
    }
    @GetMapping("/caffeine/has/{key}/{id}")
    public Mono<Boolean> hasCaffeineHash(@PathVariable("key") String key,
                                         @PathVariable("id") Long id){
        return cacheMapCaffeine.exist(key, id);
    }
    @GetMapping("/redis/put/{key}")
    public Flux<CacheTestModel> putRedisHash(@PathVariable("key") String key,
                                             @RequestParam("id") long[] id,
                                             @RequestParam("name") String[] name,
                                             @RequestParam("index") Integer[] index,
                                             @RequestParam(value = "duration", required = false) Long duration){
        return putMap(cacheMapRedis, key, id, name, index, duration);
    }
    @GetMapping("/redis/get/{key}")
    public Flux<CacheTestModel> getRedisHash(@PathVariable("key") String key){
        return cacheMapRedis.get(key).flatMapIterable(Map::values);
    }
    @GetMapping("/redis/add/{key}")
    public Mono<Boolean> addRedisHash(@PathVariable("key") String key,
                                      @RequestParam("id") long id,
                                      @RequestParam("name") String name,
                                      @RequestParam("index") Integer index){
        return addMap(cacheMapRedis, key, id, name, index);
    }
    @GetMapping("/redis/key/{key}/{id}")
    public CacheTestModel getRedisHashList(@PathVariable("key") String key,
                                           @PathVariable("id") long id){
        Mono<CacheTestModel> mono = cacheMapRedis.get(key, id);
        return mono.block();
    }
    @GetMapping("/redis/del/{key}/{id}")
    public Mono<Long> delRedisHashKey(@PathVariable("key") String key,
                                      @PathVariable("id") Long id){
        return cacheMapRedis.remove(key, id);
    }
    @GetMapping("/redis/del/{key}")
    public Mono<Long> delRedisHash(@PathVariable("key") String key){
        return cacheMapRedis.delete(key);
    }
    @GetMapping("/redis/has/{key}/{id}")
    public Mono<Boolean> hasRedisHash(@PathVariable("key") String key,
                                      @PathVariable("id") Long id){
        return cacheMapRedis.exist(key, id);
    }
}
