package cielo.printer.client

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.graphics.Bitmap
import android.os.IBinder
import cielo.printer.client.di.contract.PrinterClientContract
import cielo.printer.service.IPrinterCommunicationService
import cielo.printer.service.IPrinterCommunicationServiceCallback
import cielo.printer.service.IPrinterOperationCallback
import io.reactivex.disposables.CompositeDisposable
import io.reactivex.rxkotlin.subscribeBy
import io.reactivex.subjects.BehaviorSubject
import javax.inject.Inject

open class PrinterClientImpl @Inject
constructor(val context: Context) : PrinterClientContract {

    private var intent: Intent? = null

    private var printerServiceSubject: BehaviorSubject<IPrinterCommunicationService>
            = BehaviorSubject.create<IPrinterCommunicationService>()

    private var printerListener: PrinterClientContract.PrinterServiceListener? = null
    private val compositeDisposable: CompositeDisposable = CompositeDisposable()
    private val serviceCallback = object : IPrinterCommunicationServiceCallback.Stub() {

        override fun onSuccess() {
            printerListener?.onSuccess()
        }

        override fun onError(code: Int, msg: String?) {
            printerListener?.onError(Exception(msg))
        }

        override fun onWithoutPaper() {
            printerListener?.onWithoutPaper()
        }

    }

    private val serviceConnection = object : ServiceConnection {
        override fun onServiceDisconnected(name: ComponentName) {
            printerServiceSubject.onComplete()
        }

        override fun onServiceConnected(name: ComponentName, service: IBinder) {

            val printerService: IPrinterCommunicationService?
                    = IPrinterCommunicationService.Stub.asInterface(service)

            if (printerService != null) {
                printerServiceSubject.onNext(printerService)
            }

            setCallback()

        }
    }

    init {
        prepareServiceIntent()
    }

    private fun prepareServiceIntent() {
        this.intent = Intent()
        this.intent!!.`package` = "cielo.printer.manager"
        this.intent!!.action = "PRINT"
    }

    override fun bind() {
        context.startService(intent)
        context.bindService(intent, serviceConnection, Context.BIND_AUTO_CREATE)
    }

    override fun unbind() {
        context.unbindService(serviceConnection)
        compositeDisposable.dispose()
    }

    override fun printText(string: String) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printText(string)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printText(string: Array<String>, style: Map<String, Int>) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printColumnsTextWithAttributes(string, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printText(string: Array<String>, style: List<Map<String, Int>>) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printColumnsTextWithListAttributes(string, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printText(string: String, style: Map<String, Int>) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printTextWithAttributes(string, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printBitmap(bitmap: Bitmap) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printBitmap(bitmap)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printBitmap(bitmap: Bitmap, style: Map<String, Int>) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printBitmapWithAttributes(bitmap, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printQrCode(text: String, align: Int, size: Int) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printQRCode(text, align, size)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun printBarCode(text: String, align: Int, width: Int, height: Int, showContent: Boolean) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.printBarCode(text, align, width, height, showContent)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun setPrinterListener(printerListener: PrinterClientContract.PrinterServiceListener) {
        this.printerListener = printerListener
    }

    private fun setCallback() {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.setCallback(serviceCallback)
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun restartCounterLinesPrinted(printerListener: PrinterClientContract.PrinterOperationListener) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.restartCounterLinesPrinted(prepareOperationListener(printerListener))
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    override fun getCounterLinesPrinted(printerListener: PrinterClientContract.PrinterOperationListener) {
        val subscribe = printerServiceSubject.subscribeBy(onNext = {
            it.getCounterLinesPrinted(prepareOperationListener(printerListener))
        }, onError = {
            this.printerListener?.onError(it)
        })
        compositeDisposable.add(subscribe)
    }

    private fun prepareOperationListener(printerListener: PrinterClientContract.PrinterOperationListener): IPrinterOperationCallback.Stub {
        return object : IPrinterOperationCallback.Stub() {
            override fun onSuccess() {
                printerListener.onSuccess()
            }

            override fun onResult(result: Long) {
                printerListener.onResult(result)
            }

            override fun onError(code: Int, msg: String?) {
                printerListener.onError(Exception(msg))
            }
        }
    }

}