package cayte.frame.http;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.Thread.State;

import cayte.frame.log.D;
import okhttp3.Response;
import okhttp3.ResponseBody;

public class CayteResponse {

    public static final int ERR_NETWORK = -1;
    public static final int ERR_SERVER = -2;
    public static final int ERR_EXCEPTION = -3;
    public static final int ERR_UNKNOW = -4;
    public static final int ERR_TIMEOUT = -5;

    private String charsetName = CayteHttp.DEFAULT_CHARSETNAME;

    private Response response = null;

    private CayteResponseType responseType = CayteResponseType.STRING;
    private String path = null;

    private int code = -1;
    private String message = null;
    private ResponseBody responseBody = null;
    private InputStream in = null;
    private OutputStream out = null;
    private FileThread fileThread;

    private Object content = null;

    public CayteResponse(int code, String message) {
        this.code = code;
        this.message = message;
    }

    public CayteResponse(Response response, CayteResponseType responseType, String charsetName) {
        this.response = response;
        this.responseType = responseType;
        this.charsetName = charsetName;
    }

    public static CayteResponse createError(int code, String message) {
        return new CayteResponse(code, message);
    }

    public static CayteResponse create(Response response, CayteResponseType responseType, String charsetName) {
        return new CayteResponse(response, responseType, charsetName);
    }

    public Response response() {
        return response;
    }

    public boolean success() {
        return response != null && code == 200;
    }

    public CayteResponse path(String path) {
        this.path = path;
        return this;
    }

    public CayteResponse read() {
        return read(null);
    }

    public CayteResponse read(CayteProgress responseProgress) {
        readStatus();
        if (success()) {
            responseBody = response.body();
            if (responseProgress != null)
                responseBody = CayteResponseBody.create(responseBody, responseProgress);
            readContent();
        }
        return this;
    }

    public <T> T content() {
        return content == null ? null : (T) content;
    }

    private void readStatus() {
        if (response == null) {
            code = ERR_UNKNOW;
            message = "error : unknown";
            return;
        }
        if (!response.isSuccessful()) {
            code = response.code() > 0 ? response.code() : ERR_SERVER;
            message = response.message() == null ? "error : server" : response.message();
            return;
        }
        code = response.code();
        message = response.message();
        D.d(CayteHttp.TAG, "response code : " + code);
        D.d(CayteHttp.TAG, "response message : " + message);
    }

    private void readContent() {
        switch (responseType) {
            case BYTES:
                content = bytes();
                break;
            case STRING:
            default:
                content = string();
                break;
            case FILE:
                content = file(path);
                break;
        }
    }

    public int code() {
        return code;
    }

    public String message() {
        return message;
    }


    private byte[] bytes() {
        if (responseBody == null) {
            D.e(CayteHttp.TAG, "Exception : is not success");
            return null;
        }
        try {
            byte[] bytes = responseBody.bytes();
            D.d(CayteHttp.TAG, "response bytes : " + bytes.length);
            return bytes;
        } catch (Exception e) {
            code = ERR_EXCEPTION;
            message = "error : " + e.getLocalizedMessage();
            D.e(CayteHttp.TAG, "bytes Exception : " + e.getMessage(), e);
        }
        return null;
    }

    private String string() {
        if (responseBody == null) {
            D.e(CayteHttp.TAG, "Exception : is not success");
            return null;
        }
        try {
            String string = new String(responseBody.bytes(), charsetName);
            D.d(CayteHttp.TAG, "response content : " + string);
            return string;
        } catch (Exception e) {
            code = ERR_EXCEPTION;
            message = "error : " + e.getLocalizedMessage();
            D.e(CayteHttp.TAG, "content Exception : " + e.getMessage(), e);
        }
        return null;
    }

    private File file(String path) {
        if (responseBody == null) {
            D.e(CayteHttp.TAG, "Exception : is not success");
            return null;
        }
        try {
            in = responseBody.byteStream();

            File tempFile = new File(path + ".temp");
            if (tempFile.exists())
                tempFile.delete();
            tempFile.createNewFile();

            out = new FileOutputStream(tempFile);
            fileThread = new FileThread();
            fileThread.start();
            fileThread.join();

            if (fileThread.isInterrupted() || fileThread.isException()) {
                clearFile(path);
                code = ERR_EXCEPTION;
                D.e(CayteHttp.TAG, "file Exception : " + fileThread.getException().getMessage(), fileThread.getException());
                return null;
            }
            File file = new File(path);
            tempFile.renameTo(file);
            D.d(CayteHttp.TAG, "response file : " + file.getAbsolutePath());
            return file;
        } catch (Exception e) {
            code = ERR_EXCEPTION;
            message = "error : " + e.getLocalizedMessage();
            D.e(CayteHttp.TAG, "file Exception : " + e.getMessage(), e);
        }
        return null;
    }

    private void interruptThread() {
        if (fileThread != null)
            if (fileThread.getState() != State.TERMINATED)
                fileThread.interrupt();
    }

    private void clearFile(String path) {
        new File(path + ".temp").delete();
        new File(path).delete();
    }

    public void close() {
        interruptThread();
        if (in != null) {
            try {
                in.close();
            } catch (Exception e) {
            }
        }
        if (out != null) {
            try {
                out.close();
            } catch (Exception e) {
            }
        }
    }

    private class FileThread extends Thread {

        private boolean isException = false;
        private Exception exception = null;

        @Override
        public void run() {
            try {
                byte[] buffer = new byte[1024 * 8];
                int n = 0;
                while (!isInterrupted() && (n = in.read(buffer)) != -1) {
                    out.write(buffer, 0, n);
                    out.flush();
                }
            } catch (Exception e) {
                exception = e;
                isException = true;
            } finally {
                close();
            }
        }

        public boolean isException() {
            return isException;
        }

        public Exception getException() {
            return exception;
        }
    }

}
