package cayte.frame.http;

import java.io.File;
import java.io.InterruptedIOException;
import java.net.ConnectException;
import java.net.FileNameMap;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import cayte.frame.http.async.CayteHttpAsyncMode;
import cayte.frame.log.D;
import okhttp3.Call;
import okhttp3.CookieJar;
import okhttp3.Headers;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class CayteHttp {

    public static final String TAG = CayteHttp.class.getSimpleName();

    public static int DEFAULT_CONNECTTIMEOUT = 5;
    public static int DEFAULT_WRITETIMEOUT = 15;
    public static int DEFAULT_READTIMEOUT = 15;
    public static String DEFAULT_CHARSETNAME = "utf-8";
    public static boolean DEFAULT_RETRY_FAIL = false;
    public static boolean DEFAULT_POST_ENCODED = false;
    public static CayteHttpAsyncMode DEFAULT_HTTP_MODE = CayteHttpAsyncMode.POST;
    public static CookieJar DEFAULT_COOKIE_JAR = null;

    private OkHttpClient client = null;
    private Call call = null;

    private String url = null;
    private int connectTimeout = DEFAULT_CONNECTTIMEOUT;
    private int writeTimeout = DEFAULT_WRITETIMEOUT;
    private int readTimeout = DEFAULT_READTIMEOUT;
    private String requestCharsetName = DEFAULT_CHARSETNAME;
    private String responseCharsetName = DEFAULT_CHARSETNAME;
    private boolean retryOnFail = DEFAULT_RETRY_FAIL;
    private boolean postEncoded = DEFAULT_POST_ENCODED;
    private CookieJar cookieJar = DEFAULT_COOKIE_JAR;

    private Headers.Builder headers = new Headers.Builder();
    private Map<String, Object> params = new HashMap<String, Object>();
    private String content = null;
    private File file = null;

    private Request request = null;
    private CayteProgress requestProgress;
    private Response response = null;
    private CayteProgress responseProgress;
    private CayteResponseType responseType = CayteResponseType.STRING;
    private String path = null;
    private CayteResponse cayteResponse = null;

    private CookieJar session = null;
    private Throwable e = null;

    public CayteHttp() {
        D.i(TAG, "connect create");
    }

    public static CayteHttp create() {
        return new CayteHttp();
    }

    /*******************************************************************************************************/

    public static String decoding(String encode, String fromCharsetName, String toCharsetName) {
        try {
            return new String(encode.getBytes(fromCharsetName), toCharsetName);
        } catch (Exception e) {
            return encode;
        }
    }

    public static String getMimeType4URLConnection(String path) {
        try {
            FileNameMap fileNameMap = URLConnection.getFileNameMap();
            String type = fileNameMap.getContentTypeFor(path);
            if (type == null)
                return "application/octet-stream";
            return type;
        } catch (Exception e) {
            return "application/octet-stream";
        }
    }

    public CayteHttp url(String url) {
        this.url = url;
        D.i(TAG, "connect url : " + url.toString());
        return this;
    }

    public CayteHttp connectTimeout(int connectTimeout) {
        if (connectTimeout > 0)
            this.connectTimeout = connectTimeout;
        D.i(TAG, "client connectTimeout : " + connectTimeout);
        return this;
    }

    public CayteHttp writeTimeout(int writeTimeout) {
        if (writeTimeout > 0)
            this.writeTimeout = writeTimeout;
        D.i(TAG, "client writeTimeout : " + writeTimeout);
        return this;
    }

    public CayteHttp readTimeout(int readTimeout) {
        if (readTimeout > 0)
            this.readTimeout = readTimeout;
        D.i(TAG, "client readTimeout : " + readTimeout);
        return this;
    }

    public CayteHttp requestCharsetName(String requestCharsetName) {
        this.requestCharsetName = requestCharsetName;
        D.i(TAG, "client requestCharsetName : " + requestCharsetName);
        return this;
    }

    public CayteHttp responseCharsetName(String responseCharsetName) {
        this.responseCharsetName = responseCharsetName;
        D.i(TAG, "client responseCharsetName : " + responseCharsetName);
        return this;
    }

    public CayteHttp retryOnFail(boolean retryOnFail) {
        this.retryOnFail = retryOnFail;
        D.i(TAG, "client retryOnFail : " + retryOnFail);
        return this;
    }

    public CayteHttp postEncoded(boolean postEncoded) {
        this.postEncoded = postEncoded;
        D.i(TAG, "client postEncoded : " + postEncoded);
        return this;
    }

    public CayteHttp cookieJar(CookieJar cookieJar) {
        this.cookieJar = cookieJar;
        D.i(TAG, "client cookieJar");
        return this;
    }

    public CayteHttp requestProgress(CayteProgress requestProgress) {
        this.requestProgress = requestProgress;
        return this;
    }

    public CayteHttp responseProgress(CayteProgress responseProgress) {
        this.responseProgress = responseProgress;
        return this;
    }

    public CayteHttp responseType(CayteResponseType responseType) {
        this.responseType = responseType;
        return this;
    }

    public CayteHttp download(String path) {
        this.path = path;
        return this;
    }

    private void createClient() {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.connectTimeout(connectTimeout, TimeUnit.SECONDS);
        builder.writeTimeout(writeTimeout, TimeUnit.SECONDS);
        builder.readTimeout(readTimeout, TimeUnit.SECONDS);
        builder.retryOnConnectionFailure(retryOnFail);
        if (session == null) {
            if (cookieJar != null)
                builder.cookieJar(cookieJar);
        } else {
            builder.cookieJar(session);
        }
        client = builder.build();
    }

    public void close() {
        close(null);
    }

    public void close(Throwable e) {
        this.e = e;
        if (cayteResponse != null)
            cayteResponse.close();
        if (call != null)
            call.cancel();
        call = null;
        client = null;
    }

    private void doRequest() throws Exception {
        call = client.newCall(request);
        response = call.execute();
    }

    public CayteHttp get() {
        D.i(TAG, "get");
        try {
            createClient();
            request = new Request.Builder().url(url).headers(headers.build()).build();
            doRequest();
            D.i(TAG, "get finish");
        } catch (Exception e) {
            D.e(TAG, "get Exception : " + e.getMessage(), e);
            close(e);
        }
        return this;
    }

    public Headers.Builder getHeaders() {
        return headers;
    }

    public CayteHttp setHeaders(Headers.Builder headers) {
        this.headers = headers;
        return this;
    }

    public CayteHttp addHead(String name, String value) {
        headers.add(name, value);
        return this;
    }

    public CayteHttp setHead(String name, String value) {
        headers.set(name, value);
        return this;
    }

    public CayteHttp removeHead(String name) {
        headers.removeAll(name);
        return this;
    }

    public Map<String, Object> getParams() {
        return params;
    }

    public CayteHttp add(String name, Object value) {
        this.params.put(name, value);
        return this;
    }

    public CayteHttp addAll(Map<String, Object> params) {
        this.params.putAll(params);
        return this;
    }

    public CayteHttp set(Map<String, Object> params) {
        this.params = params;
        return this;
    }

    public CayteHttp clear() {
        this.params.clear();
        return this;
    }

    public CayteHttp content(String content) {
        this.content = content;
        return this;
    }

    public CayteHttp file(File file) {
        this.file = file;
        return this;
    }

    public CayteHttp post() {
        D.i(TAG, "post");
        try {
            CayteFormBuilder formBuilder = new CayteFormBuilder(requestCharsetName);
            if (params != null && !params.isEmpty()) {
                for (String key : params.keySet()) {
                    Object object = params.get(key);
                    if (object == null || object instanceof File)
                        continue;
                    D.d(TAG, "params :" + key + "=" + object.toString());
                    if (postEncoded)
                        formBuilder.addEncoded(key, object.toString());
                    else
                        formBuilder.add(key, object.toString());
                }
            }
            RequestBody body = formBuilder.build();
            if (requestProgress != null)
                body = CayteRequestBody.create(body, requestProgress);
            createClient();
            request = new Request.Builder().url(url).headers(headers.build()).post(body).build();
            doRequest();
            D.i(TAG, "post finish");
        } catch (Exception e) {
            D.e(TAG, "post Exception : " + e.getMessage(), e);
            close(e);
        }
        return this;
    }

    public CayteHttp post(String contentType) {
        D.i(TAG, "request : " + contentType);
        try {
            D.d(TAG, "content :" + content.toString());
            RequestBody body = RequestBody.create(MediaType.parse(contentType.toString() + ";charset=" + requestCharsetName), content);
            if (requestProgress != null)
                body = CayteRequestBody.create(body, requestProgress);
            createClient();
            request = new Request.Builder().url(url).headers(headers.build()).post(body).build();
            doRequest();
            D.i(TAG, "request finish");
        } catch (Exception e) {
            D.e(TAG, "request Exception : " + e.getMessage(), e);
            close(e);
        }
        return this;
    }

    public CayteHttp upload(String contentType) {
        D.i(TAG, "request : " + contentType);
        try {
            D.d(TAG, "file :" + file.getAbsolutePath());
            RequestBody body = RequestBody.create(MediaType.parse(contentType.toString()), file);
            if (requestProgress != null)
                body = CayteRequestBody.create(body, requestProgress);
            createClient();
            request = new Request.Builder().url(url).headers(headers.build()).post(body).build();
            doRequest();
            D.i(TAG, "request finish");
        } catch (Exception e) {
            D.e(TAG, "request Exception : " + e.getMessage(), e);
            close(e);
        }
        return this;
    }

    public CayteHttp multipart() {
        D.i(TAG, "multipart");
        try {
            MultipartBody.Builder builder = new MultipartBody.Builder().setType(MultipartBody.FORM);
            if (params != null && !params.isEmpty()) {
                for (String key : params.keySet()) {
                    Object object = params.get(key);
                    if (object == null)
                        continue;
                    if (object instanceof File) {
                        File file = (File) object;
                        D.d(TAG, "params :" + key + "=" + file.getAbsolutePath());
                        MediaType mediaType = MediaType.parse(getMimeType4URLConnection(file.getAbsolutePath()));
                        builder.addFormDataPart(key, file.getName(), RequestBody.create(mediaType, file));
                    } else if (object instanceof File[]) {
                        File[] files = (File[]) object;
                        for (File file : files) {
                            D.d(TAG, "params :" + key + "=" + file.getAbsolutePath());
                            MediaType mediaType = MediaType.parse(getMimeType4URLConnection(file.getAbsolutePath()));
                            builder.addFormDataPart(key, file.getName(), RequestBody.create(mediaType, file));
                        }
                    } else {
                        String content = object.toString();
                        D.d(TAG, "params :" + key + "=" + content);
                        if (!postEncoded)
                            content = URLEncoder.encode(content, requestCharsetName);
                        builder.addFormDataPart(key, content);
                    }
                }
            }
            RequestBody body = builder.build();
            if (requestProgress != null)
                body = CayteRequestBody.create(body, requestProgress);
            createClient();
            request = new Request.Builder().url(url).headers(headers.build()).post(body).build();
            doRequest();
            D.i(TAG, "multipart finish");
        } catch (Exception e) {
            D.e(TAG, "multipart Exception : " + e.getMessage(), e);
            close(e);
        }
        return this;
    }

    private CookieJar getSession4Connection() {
        if (client != null)
            return client.cookieJar();
        return null;
    }

    public CookieJar getSession() {
        return session;
    }

    public void setSession(CookieJar session) {
        this.session = session;
    }

    public CayteResponse response() {
        try {
            if (client == null) {
                if (e == null) {
                    cayteResponse = CayteResponse.createError(CayteResponse.ERR_UNKNOW, "error : unknow");
                } else {
                    if (e instanceof ConnectException || e instanceof UnknownHostException) {
                        cayteResponse = CayteResponse.createError(CayteResponse.ERR_NETWORK, "error : network");
                    } else if (e instanceof InterruptedIOException) {
                        cayteResponse = CayteResponse.createError(CayteResponse.ERR_TIMEOUT, "error : timeout");
                    } else {
                        cayteResponse = CayteResponse.createError(CayteResponse.ERR_EXCEPTION, "error : " + e.getLocalizedMessage());
                    }
                }
                return cayteResponse;
            }
            this.session = getSession4Connection();
            cayteResponse = CayteResponse.create(response, responseType, responseCharsetName).path(path).read(responseProgress);
            D.i(TAG, "response");
        } catch (Exception e) {
            D.e(TAG, "response Exception : " + e.getMessage(), e);
            cayteResponse = CayteResponse.createError(CayteResponse.ERR_EXCEPTION, "error : " + e.getLocalizedMessage());
        } finally {
            close();
        }
        return cayteResponse;
    }

    /*******************************************************************************************************/

    public CayteHttp postOctet() {
        return post("application/octet-stream");
    }

    public CayteHttp postJson() {
        return post("application/json");
    }

    public CayteHttp uploadOctet() {
        return upload("application/octet-stream");
    }

}
