package cayte.frame.http.async;

import java.io.File;
import java.util.Map;

import cayte.frame.http.CayteHttp;
import cayte.frame.http.CayteProgress;
import cayte.frame.http.CayteResponse;
import cayte.frame.http.CayteResponseType;
import cayte.frame.task.GenericThread;
import cayte.frame.task.TaskProgress;
import cayte.frame.task.TaskResult;
import okhttp3.CookieJar;
import okhttp3.Headers;

public class CayteHttpThread extends GenericThread<Object> implements CayteHttpSession {

    protected CayteHttpAsyncMode MODE = CayteHttpAsyncMode.DEFAULT;

    protected CayteHttp http = null;

    protected CayteHttpAsyncHandler handler = null;

    public CayteHttpThread() {
        this(CayteHttpAsyncMode.DEFAULT);
    }

    public CayteHttpThread(CayteHttpAsyncMode MODE) {
        this.MODE = MODE;
        this.http = CayteHttp.create();
        this.handler = new CayteHttpAsyncHandler();
    }

    public static CayteHttpThread create() {
        return new CayteHttpThread();
    }

    public static CayteHttpThread create(CayteHttpAsyncMode MODE) {
        return new CayteHttpThread(MODE);
    }

    public CayteHttpThread setCallback(CayteHttpAsyncCallback callback) {
        callback.setHttpSession(this);
        this.handler.setCallback(callback);
        return this;
    }

    public CayteHttp getCayteHttp() {
        return http;
    }

    public CayteHttpThread url(String url) {
        http.url(url);
        return this;
    }

    public CayteHttpThread connectTimeout(int connectTimeout) {
        http.connectTimeout(connectTimeout);
        return this;
    }

    public CayteHttpThread writeTimeout(int writeTimeout) {
        http.writeTimeout(writeTimeout);
        return this;
    }

    public CayteHttpThread readTimeout(int readTimeout) {
        http.readTimeout(readTimeout);
        return this;
    }

    public CayteHttpThread requestCharsetName(String requestCharsetName) {
        http.requestCharsetName(requestCharsetName);
        return this;
    }

    public CayteHttpThread responseCharsetName(String responseCharsetName) {
        http.responseCharsetName(responseCharsetName);
        return this;
    }

    public CayteHttpThread retryOnFail(boolean retryOnFail) {
        http.retryOnFail(retryOnFail);
        return this;
    }

    public CayteHttpThread postEncoded(boolean postEncoded) {
        http.postEncoded(postEncoded);
        return this;
    }

    public CayteHttpThread cookieManager(CookieJar cookieJar) {
        http.cookieJar(cookieJar);
        return this;
    }

    @Override
    public CookieJar getSession() {
        return http.getSession();
    }

    @Override
    public void setSession(CookieJar session) {
        http.setSession(session);
    }

    public Headers.Builder getHeaders() {
        return http.getHeaders();
    }

    public CayteHttpThread setHeaders(Headers.Builder headers) {
        http.setHeaders(headers);
        return this;
    }

    public CayteHttpThread addHead(String name, String value) {
        http.addHead(name, value);
        return this;
    }

    public CayteHttpThread setHead(String name, String value) {
        http.setHead(name, value);
        return this;
    }

    public CayteHttpThread removeHead(String name) {
        http.removeHead(name);
        return this;
    }

    public Map<String, Object> getParams() {
        return http.getParams();
    }

    public CayteHttpThread add(String name, Object value) {
        http.add(name, value);
        return this;
    }

    public CayteHttpThread addAll(Map<String, Object> params) {
        http.addAll(params);
        return this;
    }

    public CayteHttpThread set(Map<String, Object> params) {
        http.set(params);
        return this;
    }

    public CayteHttpThread clear() {
        http.clear();
        return this;
    }

    public CayteHttpThread content(String content) {
        http.content(content);
        return this;
    }

    public CayteHttpThread file(File file) {
        http.file(file);
        return this;
    }

    public CayteHttpThread download(String path) {
        http.download(path);
        return this;
    }

    public CayteHttpThread responseType(CayteResponseType responseType) {
        http.responseType(responseType);
        return this;
    }

    public CayteHttpThread requestProgress(final CayteProgress requestProgress) {
        http.requestProgress(new CayteProgress() {

            @Override
            public void progress(long progress, long length, int state) {
                if (requestProgress != null)
                    requestProgress.progress(progress, length, state);
                TaskProgress taskProgress = new TaskProgress();
                taskProgress.progress = progress;
                taskProgress.length = length;
                taskProgress.setInteger(state);
                publishProgress(taskProgress);
            }
        });
        return this;
    }

    public CayteHttpThread responseProgress(final CayteProgress responseProgress) {
        http.responseProgress(new CayteProgress() {

            @Override
            public void progress(long progress, long length, int state) {
                if (responseProgress != null)
                    responseProgress.progress(progress, length, state);
                TaskProgress taskProgress = new TaskProgress();
                taskProgress.progress = progress;
                taskProgress.length = length;
                taskProgress.setInteger(state);
                publishProgress(taskProgress);
            }
        });
        return this;
    }

    @Override
    protected void onPreExecute() {
        super.onPreExecute();
        handler.PRE();
    }

    protected CayteResponse doHttp() {
        if (MODE == CayteHttpAsyncMode.DEFAULT)
            MODE = CayteHttp.DEFAULT_HTTP_MODE;
        switch (MODE) {
            case GET:
                return http.get().response();
            case POST:
                return http.post().response();
            case JSON:
                return http.postJson().response();
            case OCTET:
                return http.postOctet().response();
            case MULTIPART:
                return http.multipart().response();
            case UP_OCTET:
                return http.uploadOctet().response();
            case DEFAULT:
            default:
                return null;
        }
    }

    @Override
    protected TaskResult _doInBackground(Object... params) {
        TaskResult result = new TaskResult();
        CayteResponse response = doHttp();
        result.success = true;
        result.setInteger(response.code());
        result.setString(response.message());
        result.setObject(response.content());
        return result;
    }

    @Override
    protected void onProgressUpdate(TaskProgress... values) {
        for (int i = 0; i < values.length; i++) {
            if (handler != null && values[i] != null)
                handler.PROGRESS(values[i].progress, values[i].length, values[i].getInteger());
        }
    }

    @Override
    protected void onPostExecute(TaskResult result) {
        super.onPostExecute(result);
        handler.POST();
        handler.CALLBACK(result);
    }

    @Override
    protected void onCancelled() {
        super.onCancelled();
        handler.CANCEL();
    }

    public void cancel() {
        try {
            super.cancel();
            if (http != null)
                http.close();
            http = null;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

}
