package cayte.frame.http;

import java.io.File;
import java.io.InterruptedIOException;
import java.net.ConnectException;
import java.net.CookieHandler;
import java.net.CookieManager;
import java.net.FileNameMap;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.squareup.okhttp.MediaType;
import com.squareup.okhttp.MultipartBuilder;
import com.squareup.okhttp.OkHttpClient;
import com.squareup.okhttp.Request;
import com.squareup.okhttp.RequestBody;
import com.squareup.okhttp.Response;

import cayte.frame.http.async.CayteHttpAsyncMode;
import cayte.frame.log.D;

public class CayteHttp {

	public static final String TAG = CayteHttp.class.getSimpleName();

	public static int DEFAULT_CONNECTTIMEOUT = 5;
	public static int DEFAULT_WRITETIMEOUT = 15;
	public static int DEFAULT_READTIMEOUT = 15;
	public static String DEFAULT_CHARSETNAME = "utf-8";
	public static boolean DEFAULT_RETRY_FAIL = false;
	public static boolean DEFAULT_POST_ENCODED = false;
	public static CayteHttpAsyncMode DEFAULT_HTTP_MODE = CayteHttpAsyncMode.POST;
	public static CookieManager DEFAULT_COOKIE_MANAGER = null;

	private OkHttpClient client = null;

	private String url = null;
	private int connectTimeout = DEFAULT_CONNECTTIMEOUT;
	private int writeTimeout = DEFAULT_WRITETIMEOUT;
	private int readTimeout = DEFAULT_READTIMEOUT;
	private String requestCharsetName = DEFAULT_CHARSETNAME;
	private String responseCharsetName = DEFAULT_CHARSETNAME;
	private boolean retryOnFail = DEFAULT_RETRY_FAIL;
	private boolean postEncoded = DEFAULT_POST_ENCODED;
	private CookieManager cookieManager = DEFAULT_COOKIE_MANAGER;

	private Map<String, Object> params = new HashMap<String, Object>();
	private String content = null;
	private File file = null;

	private String path = null;

	private Request request = null;
	private CayteProgress requestProgress;
	private Object tag = null;
	private Response response = null;
	private CayteProgress responseProgress;
	private CayteResponse cayteResponse = null;

	private CookieHandler session = null;

	public static CayteHttp create() {
		return new CayteHttp();
	}

	public CayteHttp() {
		D.i(TAG, "connect create");
	}

	public CayteHttp url(String url) {
		this.url = url;
		D.i(TAG, "connect url : " + url.toString());
		return this;
	}

	public CayteHttp connectTimeout(int connectTimeout) {
		if (connectTimeout > 0)
			this.connectTimeout = connectTimeout;
		D.i(TAG, "client connectTimeout : " + connectTimeout);
		return this;
	}

	public CayteHttp writeTimeout(int writeTimeout) {
		if (writeTimeout > 0)
			this.writeTimeout = writeTimeout;
		D.i(TAG, "client writeTimeout : " + writeTimeout);
		return this;
	}

	public CayteHttp readTimeout(int readTimeout) {
		if (readTimeout > 0)
			this.readTimeout = readTimeout;
		D.i(TAG, "client readTimeout : " + readTimeout);
		return this;
	}

	public CayteHttp requestCharsetName(String requestCharsetName) {
		this.requestCharsetName = requestCharsetName;
		D.i(TAG, "client requestCharsetName : " + requestCharsetName);
		return this;
	}

	public CayteHttp responseCharsetName(String responseCharsetName) {
		this.responseCharsetName = responseCharsetName;
		D.i(TAG, "client responseCharsetName : " + responseCharsetName);
		return this;
	}

	public CayteHttp retryOnFail(boolean retryOnFail) {
		this.retryOnFail = retryOnFail;
		D.i(TAG, "client retryOnFail : " + retryOnFail);
		return this;
	}

	public CayteHttp postEncoded(boolean postEncoded) {
		this.postEncoded = postEncoded;
		D.i(TAG, "client postEncoded : " + postEncoded);
		return this;
	}

	public CayteHttp cookieManager(CookieManager cookieManager) {
		this.cookieManager = cookieManager;
		D.i(TAG, "client cookieManager");
		return this;
	}

	public CayteHttp requestProgress(CayteProgress requestProgress) {
		this.requestProgress = requestProgress;
		return this;
	}

	public CayteHttp responseProgress(CayteProgress responseProgress) {
		this.responseProgress = responseProgress;
		return this;
	}

	private void createClient() {
		client = new OkHttpClient();
		client.setConnectTimeout(connectTimeout, TimeUnit.SECONDS);
		client.setWriteTimeout(writeTimeout, TimeUnit.SECONDS);
		client.setReadTimeout(readTimeout, TimeUnit.SECONDS);
		client.setRetryOnConnectionFailure(retryOnFail);
		if (cookieManager != null)
			client.setCookieHandler(cookieManager);
		tag = String.valueOf(System.currentTimeMillis());
	}

	public void close() {
		close(null);
	}

	private Throwable e = null;

	public void close(Throwable e) {
		this.e = e;
		if (cayteResponse != null)
			cayteResponse.close();
		if (client != null && tag != null) {
			client.cancel(tag);
		}
		client = null;
	}

	private void doRequest() throws Exception {
		setSession2Connection();
		response = client.newCall(request).execute();
	}

	public CayteHttp get() {
		D.i(TAG, "get");
		try {
			createClient();
			request = new Request.Builder().url(url).tag(tag).build();
			doRequest();
			D.i(TAG, "get finish");
		} catch (Exception e) {
			D.e(TAG, "get Exception : " + e.getMessage(), e);
			close(e);
		}
		return this;
	}

	public Map<String, Object> getParams() {
		return params;
	}

	public CayteHttp add(String name, Object value) {
		this.params.put(name, value);
		return this;
	}

	public CayteHttp addAll(Map<String, Object> params) {
		this.params.putAll(params);
		return this;
	}

	public CayteHttp set(Map<String, Object> params) {
		this.params = params;
		return this;
	}

	public CayteHttp clear() {
		this.params.clear();
		return this;
	}

	public CayteHttp content(String content) {
		this.content = content;
		return this;
	}

	public CayteHttp file(File file) {
		this.file = file;
		return this;
	}

	public CayteHttp post() {
		D.i(TAG, "post");
		try {
			CayteFormBuilder formBuilder = new CayteFormBuilder(requestCharsetName);
			if (params != null && !params.isEmpty()) {
				for (String key : params.keySet()) {
					Object object = params.get(key);
					if (object == null || object instanceof File)
						continue;
					D.d(TAG, "params :" + key + "=" + object.toString());
					if (postEncoded)
						formBuilder.addEncoded(key, object.toString());
					else
						formBuilder.add(key, object.toString());
				}
			}
			RequestBody body = formBuilder.build();
			if (requestProgress != null)
				body = CayteRequestBody.create(body, requestProgress);
			createClient();
			request = new Request.Builder().url(url).post(body).tag(tag).build();
			doRequest();
			D.i(TAG, "post finish");
		} catch (Exception e) {
			D.e(TAG, "post Exception : " + e.getMessage(), e);
			close(e);
		}
		return this;
	}

	public CayteHttp post(String contentType) {
		D.i(TAG, "request : " + contentType);
		try {
			D.d(TAG, "content :" + content.toString());
			RequestBody body = RequestBody
					.create(MediaType.parse(contentType.toString() + ";charset=" + requestCharsetName), content);
			if (requestProgress != null)
				body = CayteRequestBody.create(body, requestProgress);
			createClient();
			request = new Request.Builder().url(url).post(body).tag(tag).build();
			doRequest();
			D.i(TAG, "request finish");
		} catch (Exception e) {
			D.e(TAG, "request Exception : " + e.getMessage(), e);
			close(e);
		}
		return this;
	}

	public CayteHttp upload(String contentType) {
		D.i(TAG, "request : " + contentType);
		try {
			D.d(TAG, "file :" + file.getAbsolutePath());
			RequestBody body = RequestBody.create(MediaType.parse(contentType.toString()), file);
			if (requestProgress != null)
				body = CayteRequestBody.create(body, requestProgress);
			createClient();
			request = new Request.Builder().url(url).post(body).tag(tag).build();
			doRequest();
			D.i(TAG, "request finish");
		} catch (Exception e) {
			D.e(TAG, "request Exception : " + e.getMessage(), e);
			close(e);
		}
		return this;
	}

	public CayteHttp multipart() {
		D.i(TAG, "multipart");
		try {
			MultipartBuilder builder = new MultipartBuilder().type(MultipartBuilder.FORM);
			if (params != null && !params.isEmpty()) {
				for (String key : params.keySet()) {
					Object object = params.get(key);
					if (object == null)
						continue;
					if (object instanceof File) {
						File file = (File) object;
						D.d(TAG, "params :" + key + "=" + file.getAbsolutePath());
						MediaType mediaType = MediaType.parse(getMimeType4URLConnection(file.getAbsolutePath()));
						builder.addFormDataPart(key, file.getName(), RequestBody.create(mediaType, file));
					} else {
						String content = object.toString();
						D.d(TAG, "params :" + key + "=" + content);
						if (!postEncoded)
							content = URLEncoder.encode(content, requestCharsetName);
						builder.addFormDataPart(key, content);
					}
				}
			}
			RequestBody body = builder.build();
			if (requestProgress != null)
				body = CayteRequestBody.create(body, requestProgress);
			createClient();
			request = new Request.Builder().url(url).post(body).tag(tag).build();
			doRequest();
			D.i(TAG, "multipart finish");
		} catch (Exception e) {
			D.e(TAG, "multipart Exception : " + e.getMessage(), e);
			close(e);
		}
		return this;
	}

	private CookieHandler getSession4Connection() {
		if (client != null)
			return client.getCookieHandler();
		return null;
	}

	private void setSession2Connection() {
		if (client != null && session != null)
			client.setCookieHandler(session);
	}

	public CookieHandler getSession() {
		return session;
	}

	public void setSession(CookieHandler session) {
		this.session = session;
	}

	public CayteHttp download(String path) {
		this.path = path;
		return this;
	}

	public CayteResponse response() {
		try {
			if (client == null) {
				if (e == null) {
					cayteResponse = CayteResponse.createError(CayteResponse.ERR_UNKNOW, "error : unknow");
				} else {
					if (e instanceof ConnectException || e instanceof UnknownHostException) {
						cayteResponse = CayteResponse.createError(CayteResponse.ERR_NETWORK, "error : network");
					} else if (e instanceof InterruptedIOException) {
						cayteResponse = CayteResponse.createError(CayteResponse.ERR_TIMEOUT, "error : timeout");
					} else {
						cayteResponse = CayteResponse.createError(CayteResponse.ERR_EXCEPTION,
								"error : " + e.getLocalizedMessage());
					}
				}
				return cayteResponse;
			}
			this.session = getSession4Connection();
			if (path == null) {
				cayteResponse = CayteResponse.create(response, responseCharsetName).read(responseProgress);
			} else {
				cayteResponse = CayteResponse.create(response, responseCharsetName).read(path, responseProgress);
			}
			D.i(TAG, "response");
		} catch (Exception e) {
			D.e(TAG, "response Exception : " + e.getMessage(), e);
			cayteResponse = CayteResponse.createError(CayteResponse.ERR_EXCEPTION,
					"error : " + e.getLocalizedMessage());
		} finally {
			close();
		}
		return cayteResponse;
	}

	/*******************************************************************************************************/

	public static String decoding(String encode, String fromCharsetName, String toCharsetName) {
		try {
			return new String(encode.getBytes(fromCharsetName), toCharsetName);
		} catch (Exception e) {
			return encode;
		}
	}

	public static String getMimeType4URLConnection(String path) {
		try {
			FileNameMap fileNameMap = URLConnection.getFileNameMap();
			String type = fileNameMap.getContentTypeFor(path);
			if (type == null)
				return "application/octet-stream";
			return type;
		} catch (Exception e) {
			return "application/octet-stream";
		}
	}

	/*******************************************************************************************************/

	public CayteHttp postOctet() {
		return post("application/octet-stream");
	}

	public CayteHttp postJson() {
		return post("application/json");
	}

	public CayteHttp uploadOctet() {
		return upload("application/octet-stream");
	}

}
