package cayte.frame.http;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.Thread.State;

import com.squareup.okhttp.Response;
import com.squareup.okhttp.ResponseBody;

import cayte.frame.log.D;

public class CayteResponse {

	public static final int ERR_NETWORK = -1;
	public static final int ERR_SERVER = -2;
	public static final int ERR_EXCEPTION = -3;
	public static final int ERR_UNKNOW = -4;
	public static final int ERR_TIMEOUT = -5;

	private String charsetName = CayteHttp.DEFAULT_CHARSETNAME;

	private Response response = null;

	private int code = -1;
	private String message = null;
	private String content = null;
	private File file = null;

	public static CayteResponse createError(int code, String message) {
		return new CayteResponse(code, message);
	}

	public static CayteResponse create(Response response, String charsetName) {
		return new CayteResponse(response, charsetName);
	}

	public CayteResponse(int code, String message) {
		this.code = code;
		this.message = message;
	}

	public CayteResponse(Response response, String charsetName) {
		this.response = response;
		this.charsetName = charsetName;
	}

	public Response response() {
		return response;
	}

	public boolean success() {
		return response != null && response.isSuccessful() && code == 200;
	}

	private void readStatus() {
		if (response == null) {
			code = ERR_UNKNOW;
			message = "error : unknown";
			return;
		}
		if (!response.isSuccessful()) {
			code = ERR_SERVER;
			message = "error : server";
			return;
		}
		code = response.code();
		message = response.message();
	}

	public CayteResponse read() {
		return read(null);
	}

	public CayteResponse read(CayteProgress responseProgress) {
		readStatus();
		if (success()) {
			try {
				ResponseBody responseBody = response.body();
				if (responseProgress != null)
					responseBody = CayteResponseBody.create(responseBody, responseProgress);
				content = new String(responseBody.bytes(), charsetName);
			} catch (Exception e) {
				code = ERR_EXCEPTION;
				message = "error : " + e.getLocalizedMessage();
				content = null;
				D.e(CayteHttp.TAG, "content Exception : " + e.getMessage(), e);
			}
		}
		D.d(CayteHttp.TAG, "response code : " + code);
		D.d(CayteHttp.TAG, "response message : " + message);
		if (content != null)
			D.d(CayteHttp.TAG, "response content : " + content);
		else
			D.d(CayteHttp.TAG, "response content : null");
		return this;
	}

	public CayteResponse read(String path, CayteProgress responseProgress) {
		readStatus();
		if (success()) {
			try {
				ResponseBody responseBody = response.body();
				if (responseProgress != null)
					responseBody = CayteResponseBody.create(responseBody, responseProgress);
				file = readFile(path, responseBody);
				if (file == null) {
					code = ERR_NETWORK;
					message = "error : network";
				}
			} catch (Exception e) {
				code = ERR_EXCEPTION;
				message = "error : " + e.getLocalizedMessage();
				file = null;
				D.e(CayteHttp.TAG, "file Exception : " + e.getMessage(), e);
			}
		}
		D.d(CayteHttp.TAG, "response code : " + code);
		D.d(CayteHttp.TAG, "response message : " + message);
		if (file != null)
			D.d(CayteHttp.TAG, "response file : " + file.getAbsolutePath());
		else
			D.d(CayteHttp.TAG, "response file : null");
		return this;
	}

	public int code() {
		return code;
	}

	public String message() {
		return message;
	}

	public String content() {
		return content;
	}

	public File file() {
		return file;
	}

	private InputStream in = null;
	private OutputStream out = null;

	private File readFile(String path, ResponseBody responseBody) throws Exception {
		in = responseBody.byteStream();

		File tempFile = new File(path + ".temp");
		if (tempFile.exists())
			tempFile.delete();
		tempFile.createNewFile();

		out = new FileOutputStream(tempFile);
		fileThread = new FileThread();
		fileThread.start();
		fileThread.join();

		if (fileThread.isInterrupted() || fileThread.isException()) {
			clearFile(path);
			return null;
		}
		File file = new File(path);
		tempFile.renameTo(file);
		return file;
	}

	private FileThread fileThread;

	private class FileThread extends Thread {

		private boolean isException = false;

		@Override
		public void run() {
			try {
				byte[] buffer = new byte[1024 * 8];
				int n = 0;
				while (!isInterrupted() && (n = in.read(buffer)) != -1) {
					out.write(buffer, 0, n);
					out.flush();
				}
			} catch (Exception e) {
				isException = true;
			} finally {
				close();
			}
		}

		public boolean isException() {
			return isException;
		}

	}

	private void interruptThread() {
		if (fileThread != null)
			if (fileThread.getState() != State.TERMINATED)
				fileThread.interrupt();
	}

	private void clearFile(String path) {
		new File(path + ".temp").delete();
		new File(path).delete();
	}

	public void close() {
		interruptThread();
		if (in != null) {
			try {
				in.close();
			} catch (Exception e) {
			}
		}
		if (out != null) {
			try {
				out.close();
			} catch (Exception e) {
			}
		}
	}

}
