package cayte.frame.ui;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;

import cayte.frame.act.ActivityStack;
import cayte.frame.act.OverridePendingTransition;
import cayte.frame.helper.MobileParams;
import cayte.frame.manager.IsRefreshManager;
import cayte.frame.task.GenericTask;
import cayte.frame.task.TaskManager;

public class UIActivity<T extends Activity, E extends UIInterface> {

    /**
     * LayoutInflater
     */
    public LayoutInflater inflater = null;
    /**
     * UI需要实现的相关接口
     */
    public E ai;
    /**
     * T extends Activity
     */
    protected T a;
    /**
     * Task管理
     */
    protected TaskManager taskManager = null;


    /**
     * 实例化
     *
     * @param a  a extends Activity
     * @param ai ai implements UIInterface
     */
    public UIActivity(T a, E ai) {
        this.a = a;
        this.ai = ai;
    }

    /**
     * 获取 T extends Activity 实例
     *
     * @return T extends Activity
     */
    public T getA() {
        return a;
    }

    /**
     * 在super.onCreate前调用,设置Activity载入动画
     */
    public void overrideIn() {
        OverridePendingTransition.overrideInPendingTransition(a, ai.overridePendingTransition());
    }

    /**
     * 在finish后调用,设置Activity关闭动画
     */
    public void overrideOut() {
        OverridePendingTransition.overrideOutPendingTransition(a, ai.overridePendingTransition());
    }

    /**
     * 在super.onCreate后调用<br>
     * 初始化MobileParams,taskManager,inflater
     */
    public void onCreate() {
        MobileParams.newInstance(a);
        taskManager = new TaskManager();
        inflater = (LayoutInflater) a.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
    }

    /**
     * 在super.onResume后调用<br>
     * ai.refresh(IsRefreshManager.instance().get(a.getClass()));
     */
    public void onResume() {
        String action = IsRefreshManager.instance().get(a.getClass());
        if (!TextUtils.isEmpty(action))
            ai.refresh(action);
    }

    /**
     * 在super.onConfigurationChanged后调用<br>
     * 更新MobileParams
     *
     * @param newConfig 传入onConfigurationChanged中的参数
     */
    public void onConfigurationChanged(Configuration newConfig) {
        MobileParams.newInstance(a);
    }

    /**
     * 在super.onDestroy后调用<br>
     * 取消所有Task<br>
     * 销毁对象taskManager,inflater
     */
    public void onDestroy() {
        cancelAllTask();
        taskManager.deleteObservers();
        taskManager = null;
        inflater = null;
    }

    /**
     * 将当前Activity压入栈中
     */
    public void pushActivity() {
        ActivityStack.instance().pushActivity(a, 0);
    }

    /**
     * 将当前Activity压入栈中
     *
     * @param flag 0:默认压入<br>
     *             1:如果队列中包含当前Activity,pop至该位置并替换该Activity<br>
     *             2:pop至根Activity并压入当前Activity<br>
     */
    public void pushActivity(int flag) {
        ActivityStack.instance().pushActivity(a, flag);
    }

    /**
     * 将当前Activity退出栈中
     */
    public void popActivity() {
        ActivityStack.instance().popActivity(a);
    }

    /**
     * 集合设置是否可见,setVisibility
     *
     * @param visible like View.VISIBLE
     * @param views   View...
     */
    public void setVisibility(int visible, View... views) {
        for (View v : views) {
            v.setVisibility(visible);
        }
    }

    /**
     * 集合设置点击事件,setOnClickListener
     *
     * @param click 点击事件View.OnClickListener
     * @param views View...
     */
    public void setClickListener(View.OnClickListener click, View... views) {
        for (View v : views) {
            v.setOnClickListener(click);
        }
    }

    /**
     * 执行线程并加入到TaskManager中
     *
     * @param genericTask GenericTask
     */
    public void execute(GenericTask<?> genericTask) {
        if (genericTask.getStatus() != android.os.AsyncTask.Status.PENDING)
            return;
        genericTask.execute();
        taskManager.addTask(genericTask);
    }

    /**
     * 执行线程并加入到TaskManager中
     *
     * @param genericTask GenericTask
     * @param params      参数集合
     * @param <T>         参数泛型
     */
    public <T> void execute(GenericTask<T> genericTask, T... params) {
        if (genericTask.getStatus() != android.os.AsyncTask.Status.PENDING)
            return;
        genericTask.execute(params);
        taskManager.addTask(genericTask);
    }

    /**
     * 取消TaskManager中所有线程(只能取消回调,不能中断线程)
     */
    public void cancelAllTask() {
        taskManager.cancelAll();
    }

}
