package buzz.getcoco.iot.android;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.annotations.SerializedName;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import buzz.getcoco.iot.Attribute;
import buzz.getcoco.iot.Capability;
import buzz.getcoco.iot.CapabilityHvacControl;
import buzz.getcoco.iot.CapabilityRemoteControl;
import buzz.getcoco.iot.Device;
import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.Resource;

public class AttributeEx extends Attribute {

  private static final String TAG = "AttributeEx";

  private transient MutableLiveData<String> nameObservable;
  private transient MutableLiveData<Object> minValueObservable;
  private transient MutableLiveData<Object> maxValueObservable;
  private transient MutableLiveData<Object> currentValueObservable;
  private transient MutableLiveData<Object> defaultValueObservable;

  private transient Identifier identifier;

  private transient ApplianceInfo[] applianceInfo;

  protected AttributeEx(int id, Capability parent) {
    super(id, parent);
  }

  protected Network getParentNetwork() {
    Resource resource = getParentResource();
    Device device = (null == resource) ? null : resource.getParent();

    return (null == device) ? null : device.getParent();
  }

  protected void syncCurrentValue() {
    FactoryUtility.postValue(currentValueObservable, getCurrentValue());
  }

  @Nullable
  public ResourceEx getParentResource() {
    Capability capability = getParent();

    return (null == capability) ? null : (ResourceEx) capability.getParent();
  }

  @Override
  protected void internalSetName(String name) {
    super.internalSetName(name);
    FactoryUtility.postValue(nameObservable, name);
  }

  @Override
  protected void internalSetMinValue(Object minValue) {
    Log.d(TAG, "internalSetMinValue: minVal: " + minValue);

    super.internalSetMinValue(minValue);
    FactoryUtility.postValue(minValueObservable, minValue);
  }

  @Override
  protected void internalSetMaxValue(Object maxValue) {
    Log.d(TAG, "internalSetMaxValue: maxVal: " + maxValue);

    super.internalSetMaxValue(maxValue);
    FactoryUtility.postValue(maxValueObservable, maxValue);
  }

  @Override
  protected void internalSetCurrentValue(Object currentValue) {
    super.internalSetCurrentValue(currentValue);

    Log.d(TAG, "internalSetCurrentValue: " + currentValue);
    Log.d(TAG, "internalSetCurrentValue: " + getCurrentValue());

    if (null != getParent() && CapabilityRemoteControl.AttributeId.APPLIANCE_LIST == getId()) {
      Log.d(TAG, "internalSetCurrentValue: updating the appliance info");

      reCalApplianceInfo();

      Log.d(TAG, "internalSetCurrentValue: updated appliance info: " + Arrays.toString(applianceInfo));
    }

    FactoryUtility.postValue(currentValueObservable, getCurrentValue());
  }

  @Override
  protected void internalSetDefaultValue(Object defaultValue) {
    Log.d(TAG, "internalSetDefaultValue: defVal: " + defaultValue);

    super.internalSetDefaultValue(defaultValue);
    FactoryUtility.postValue(defaultValueObservable, defaultValue);
  }

  @Override
  protected void internalSetParent(Capability parent) {
    super.internalSetParent(parent);

    syncCurrentValue();
  }

  public LiveData<String> getNameObservable() {
    return (null != nameObservable) ? nameObservable : (nameObservable = FactoryUtility.createLiveData(getName()));
  }

  public LiveData<Object> getCurrentValueObservable() {
    return (null != currentValueObservable) ? currentValueObservable : (currentValueObservable = FactoryUtility.createLiveData(getCurrentValue()));
  }

  public LiveData<Object> getDefaultValueObservable() {
    return (null != defaultValueObservable) ? defaultValueObservable : (defaultValueObservable = FactoryUtility.createLiveData(getDefaultValue()));
  }

  public static Capability.AttributeId getAttributeId(int attributeId, int capabilityId) {
    return Capability.CapabilityId.getEnum(capabilityId).getAttributeId(attributeId);
  }

  public Identifier getIdentifier() {
    return (null != identifier) ?  identifier : (identifier = Identifier.getIdentifier(this));
  }

  private synchronized void reCalApplianceInfo() {
    ApplianceInfo[] applianceInfo = null;

    try {
      ApplianceInfoContainer container = new Gson().fromJson(getAsString(), ApplianceInfoContainer.class);
      applianceInfo = (null == container) ? null : container.applianceInfo;
    } catch (JsonSyntaxException jse) {
      Log.d(TAG, "reCalApplianceInfo: error", jse);
    }

    if (null != applianceInfo)
      this.applianceInfo = applianceInfo;
    else
      this.applianceInfo = new ApplianceInfo[0];
  }

  public ApplianceInfo[] getAsApplianceInfo() {
    if (CapabilityRemoteControl.AttributeId.APPLIANCE_LIST != getId()) {
      throw new IllegalArgumentException("called getAsApplianceInfo on: " + this);
    }

    if (null == applianceInfo)
      reCalApplianceInfo();

    return applianceInfo;
  }

  private void trimMemory() {
    nameObservable                  = FactoryUtility.nullIfEmpty(nameObservable);
    minValueObservable              = FactoryUtility.nullIfEmpty(minValueObservable);
    maxValueObservable              = FactoryUtility.nullIfEmpty(maxValueObservable);
    currentValueObservable          = FactoryUtility.nullIfEmpty(currentValueObservable);
    defaultValueObservable          = FactoryUtility.nullIfEmpty(defaultValueObservable);
  }

  private static final class ApplianceInfoContainer {
    @SerializedName("appliances")
    public ApplianceInfo[] applianceInfo;
  }

  public static final class ApplianceInfo {
    @SerializedName("applianceId")
    public String applianceId;
    @SerializedName("applianceName")
    public String applianceName;
    @SerializedName("brandId")
    public int brandId;
    @SerializedName("pairingCode")
    public int pairingCode;

    @SerializedName("capCommandsArray")
    public CapabilityCommands[] capCommands;

    @SerializedName("capAttribValuesArray")
    public CapabilityAttributes[] attrs;

    @Nullable
    public Object getAttributeValue(@NonNull Capability.AttributeId attributeId) {
      int capId = attributeId.getCapabilityId().getInt();
      int attrId = attributeId.getInt();

      if (null == attrs)
        return null;

      for (CapabilityAttributes ca : attrs) {
        if (ca.capId == capId && ca.attrId == attrId)
          return ca.attrValue;
      }

      return null;
    }

    @NonNull
    public List<ModesAttributeValue> getAsSupportedModes() {

      Object oModesJson = getAttributeValue(CapabilityHvacControl.AttributeId.SUPPORTED_MODES_ARR);

      try {
        Gson gson = new Gson();
        return Arrays.asList(gson.fromJson(gson.toJson(oModesJson), ModesAttributeValue[].class));
      } catch (Exception e) {
        Log.d(TAG, "getAsSupportedModes: err", e);
      }

      return Collections.emptyList();
    }

    @NonNull
    @Override
    public String toString() {
      return "ApplianceInfo{" +
          "applianceId=" + applianceId +
          ", applianceName='" + applianceName + '\'' +
          ", brandId=" + brandId +
          ", pairingCode=" + pairingCode +
          ", capCommands=" + Arrays.toString(capCommands) +
          ", attrs=" + Arrays.toString(attrs) +
          '}';
    }
  }

  public static final class CapabilityCommands {
    @SerializedName("capabilityId")
    public int capId;
    @SerializedName("commandIdArray")
    public int[] supportedCmdArr;

    public Capability.CapabilityId getCapabilityId() {
      return Capability.CapabilityId.getEnum(capId);
    }

    @NonNull
    @Override
    public String toString() {
      return "CapabilityCommands{" +
          "capId=" + capId +
          ", supportedCmdArr=" + Arrays.toString(supportedCmdArr) +
          '}';
    }
  }

  public static final class CapabilityAttributes {
    @SerializedName("capabilityId")
    private int capId;
    @SerializedName("attributeId")
    private int attrId;
    @SerializedName("attributeValue")
    public Object attrValue;
    @SerializedName("attributeDataType")
    public int dataType;

    public CapabilityAttributes(int capId, int attrId, Object attrValue, int dataType) {
      this.capId = capId;
      this.attrId = attrId;
      this.attrValue = attrValue;
      this.dataType = dataType;
    }

    public CapabilityAttributes() {

    }

    public Capability.AttributeId getAttributeId() {
      return Capability.CapabilityId.getEnum(capId).getAttributeId(attrId);
    }

    public int getAttrId() {
      return attrId;
    }

    public int getCapId() {
      return capId;
    }

    @NonNull
    @Override
    public String toString() {
      return "CapabilityAttributes{" +
          "capId=" + capId +
          ", attrId=" + attrId +
          ", attrValue=" + attrValue +
          ", dataType=" + dataType +
          '}';
    }
  }

  public static final class ModesAttributeValue {
    @SerializedName("mode")
    private final int mode;
    @SerializedName("minTemperatureCelsius")
    private final int minTempCelsius;
    @SerializedName("maxTemperatureCelsius")
    private final int maxTempCelsius;
    @SerializedName("supportedFanSpeedsArr")
    private final int[] fanSpeeds;

    public ModesAttributeValue(int mode, int minTempCelsius, int maxTempCelsius, int[] fanSpeeds) {
      this.mode = mode;
      this.minTempCelsius = minTempCelsius;
      this.maxTempCelsius = maxTempCelsius;
      this.fanSpeeds = fanSpeeds;
    }

    public int getMode() {
      return mode;
    }

    public int getMinTempCelsius() {
      return minTempCelsius;
    }

    public int getMaxTempCelsius() {
      return maxTempCelsius;
    }

    @NonNull
    public int[] getFanSpeeds() {
      return fanSpeeds == null ? new int[0] : fanSpeeds;
    }

    @NonNull
    @Override
    public String toString() {
      return "ModesAttributeValue{" +
          "mode=" + mode +
          ", minTempCelsius=" + minTempCelsius +
          ", maxTempCelsius=" + maxTempCelsius +
          ", fanSpeeds=" + Arrays.toString(fanSpeeds) +
          '}';
    }
  }
}
