package buzz.getcoco.iot.android;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.common.collect.ImmutableList;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import buzz.getcoco.iot.ConnectivityStateManager;
import buzz.getcoco.iot.Device;
import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.Resource;
import buzz.getcoco.iot.Rule;
import buzz.getcoco.iot.Scene;
import buzz.getcoco.iot.Zone;

class BaseNetworkEx<D extends Device, Z extends Zone, S extends Scene, R extends Rule> extends Network {

  private static final String TAG = "BaseNetworkEx";

  private transient Identifier identifier;

  private transient MutableLiveData<List<D>> deviceListObservable;
  private transient MutableLiveData<List<Z>> zoneListObservable;
  private transient MutableLiveData<List<R>> ruleListObservable;
  private transient MutableLiveData<List<S>> sceneListObservable;

  private transient MutableLiveData<String> networkNameObservable;
  private transient MutableLiveData<Boolean> removeNetworkObservable;
  private transient MutableLiveData<Network.State> stateObservable;
  private transient MutableLiveData<ConnectivityStateManager.State> conStateObservable;

  private transient ImmutableList<D> deviceList;
  private transient ImmutableList<Z> zoneList;
  private transient ImmutableList<R> ruleList;
  private transient ImmutableList<S> sceneList;

  private transient boolean isRemoved;

  protected BaseNetworkEx(String id) {
    super(id);
  }

  @SuppressWarnings("unchecked")
  private static <T, U extends T> U castUp(T t) {
    return (U) t;
  }

  @Override
  protected void internalAddDevice(Device device) {
    super.internalAddDevice(device);

    if (null == deviceList)
      return;

    postSortedDevices();
  }

  @Override
  protected void internalRemoveDevice(long deviceNodeId) {
    super.internalRemoveDevice(deviceNodeId);

    if (null == deviceList)
      return;

    postSortedDevices();
  }

  @Override
  protected void internalAddZone(Zone zone) {
    super.internalAddZone(zone);

    if (null == zoneList)
      return;

    postSortedZones();
  }

  @Override
  protected void internalRemoveZone(int zoneId) {
    super.internalRemoveZone(zoneId);

    if (null == zoneList)
      return;

    postSortedZones();
  }

  @Override
  protected void internalAddRule(Rule rule) {
    super.internalAddRule(rule);

    if (null == ruleList)
      return;

    postSortedRules();
  }

  @Override
  protected void internalRemoveRule(int ruleId) {
    super.internalRemoveRule(ruleId);

    if (null == ruleList)
      return;

    postSortedRules();
  }

  @Override
  protected void internalAddScene(Scene scene) {
    super.internalAddScene(scene);

    if (null == sceneList)
      return;

    postSortedScenes();
  }

  @Override
  protected void internalRemoveScene(int sceneId) {
    SceneEx scene = getScene(sceneId);

    super.internalRemoveScene(sceneId);

    if (null != scene)
      scene.onSceneRemoved();

    if (null == sceneList)
      return;

    postSortedScenes();
  }

  @Override
  protected void internalSetName(String name) {
    super.internalSetName(name);
    Log.d(TAG, "internalSetName: newName: " + name + ", ID: " + this.getId());

    FactoryUtility.postValue(networkNameObservable, name);
  }

  @Override
  protected void internalSetState(Network.State state) {
    super.internalSetState(state);
    Log.d(TAG, "internalSetState: state: " + state);

    FactoryUtility.postValue(stateObservable, state);
  }

  @Override
  protected void internalRemoveNetwork() {
    super.internalRemoveNetwork();
    Log.d(TAG, "internalRemoveNetwork: networkName: " + this.getName() + ", isRemoved: " + isRemoved);

    FactoryUtility.postValue(removeNetworkObservable, (isRemoved = true));
  }

  protected void onZoneUpdated(@NonNull ZoneEx zone) {
    // if default zone has changed. It might affect the addition and removal from list
    // else, this update can be ignored

    if (Zone.DEFAULT_ZONE_ID != zone.getId())
      return;

    postSortedZones();
  }

  protected void onSceneNameUpdated(Scene scene) {
    Log.d(TAG, "onSceneNameUpdated: scene: " + scene);

    postSortedScenes();
  }

  protected void onRuleNameUpdated(Rule rule) {
    Log.d(TAG, "onRuleNameUpdated: rule: " + rule);

    postSortedRules();
  }

  protected void onZoneNameUpdated(Zone zone) {
    Log.d(TAG, "onZoneNameUpdated: zone: " + zone);

    postSortedZones();
  }

  protected void onDeviceNameUpdated(Device device) {
    Log.d(TAG, "onDeviceNameUpdated: device: " + device);

    postSortedDevices();
  }

  @Override
  protected void internalSetConnectivityState(ConnectivityStateManager.State connectivityState) {
    ConnectivityStateManager.State currentState = getConnectivityManagerState();

    super.internalSetConnectivityState(connectivityState);

    if (currentState == connectivityState)
      return;

    for (Device d : this) {
      for (Resource r : d) {
        ResourceEx rex = (ResourceEx) r;

        rex.onStateChange();
      }
    }

    FactoryUtility.postValue(conStateObservable, connectivityState);
  }

  public Identifier getIdentifier() {
    return (null != identifier) ?  identifier : (identifier = Identifier.getIdentifier(this));
  }

  @SuppressWarnings("unchecked")
  public List<D> getSortedDeviceList() {
    ArrayList<D> devices = new ArrayList<>();

    for (Device device: getDeviceMap().values()) {
      devices.add((D) device);
    }

    Collections.sort(devices, Utils.DEVICE_COMPARATOR);

    return (deviceList = ImmutableList.copyOf(devices));
  }

  @SuppressWarnings("unchecked")
  public List<Z> getSortedZoneList() {
    Z defaultZone = getZone(Zone.DEFAULT_ZONE_ID);
    ArrayList<Z> zones = new ArrayList<>();

    for (Zone zone: getZoneMap().values()) {
      zones.add((Z) zone);
    }

    boolean defaultHasResources = !defaultZone.getResources().isEmpty();

    if (!defaultHasResources)
      zones.remove(defaultZone);

    Collections.sort(zones, Utils.ZONE_COMPARATOR);

    return (zoneList = ImmutableList.copyOf(zones));
  }

  @SuppressWarnings("unchecked")
  public List<R> getSortedRuleList() {
    ArrayList<R> rules = new ArrayList<>();

    for (Rule rule: getRuleMap().values()) {
      rules.add((R) rule);
    }

    Collections.sort(rules, Utils.RULE_COMPARATOR);

    return (ruleList = ImmutableList.copyOf(rules));
  }

  @SuppressWarnings("unchecked")
  public List<S> getSortedSceneList() {
    ArrayList<S> scenes = new ArrayList<>();

    for (Scene scene: getSceneMap().values()) {
      scenes.add((S) scene);
    }

    Collections.sort(scenes, Utils.SCENE_COMPARATOR);

    return (sceneList = ImmutableList.copyOf(scenes));
  }

  protected void postSortedZones() {
    if (null == zoneList)
      return;

    FactoryUtility.postValue(zoneListObservable, getSortedZoneList());
  }

  protected void postSortedScenes() {
    if (null == sceneList)
      return;

    FactoryUtility.postValue(sceneListObservable, getSortedSceneList());
  }

  protected void postSortedRules() {
    if (null == ruleList)
      return;

    FactoryUtility.postValue(ruleListObservable, getSortedRuleList());
  }

  protected void postSortedDevices() {
    if (null == deviceList)
      return;

    FactoryUtility.postValue(deviceListObservable, getSortedDeviceList());
  }

  public LiveData<List<D>> getDeviceListObservable() {
    return (null != deviceListObservable) ? deviceListObservable : (deviceListObservable = FactoryUtility.createLiveData(getSortedDeviceList()));
  }

  /**
   * NOTE: This will remove the default zone, if it has no resources
   */
  public LiveData<List<Z>> getZoneListObservable() {
    return ((null != zoneListObservable) ? zoneListObservable : (zoneListObservable = FactoryUtility.createLiveData(getSortedZoneList())));
  }

  public LiveData<List<R>> getRuleListObservable() {
    return (null != ruleListObservable) ? ruleListObservable : (ruleListObservable = FactoryUtility.createLiveData(getSortedRuleList()));
  }

  public LiveData<List<S>> getSceneListObservable() {
    return (null != sceneListObservable) ? sceneListObservable : (sceneListObservable = FactoryUtility.createLiveData(getSortedSceneList()));
  }

  public LiveData<String> getNetworkNameObservable() {
    return (null != networkNameObservable) ? networkNameObservable : (networkNameObservable = FactoryUtility.createLiveData(getName()));
  }

  public LiveData<Network.State> getStateObservable() {
    return (null != stateObservable) ? stateObservable : (stateObservable = FactoryUtility.createLiveData(getState()));
  }

  public LiveData<Boolean> getRemoveNetworkObservable() {
    return (null != removeNetworkObservable) ? removeNetworkObservable : (removeNetworkObservable = FactoryUtility.createLiveData(isRemoved));
  }

  public LiveData<ConnectivityStateManager.State> getConnectivityManageStateObservable() {
    return (null != conStateObservable) ? conStateObservable : (conStateObservable = FactoryUtility.createLiveData(getConnectivityManagerState()));
  }

  private void trimMemory() {
    deviceListObservable    = FactoryUtility.nullIfEmpty(deviceListObservable);
    zoneListObservable      = FactoryUtility.nullIfEmpty(zoneListObservable);
    ruleListObservable      = FactoryUtility.nullIfEmpty(ruleListObservable);
    sceneListObservable     = FactoryUtility.nullIfEmpty(sceneListObservable);
    networkNameObservable   = FactoryUtility.nullIfEmpty(networkNameObservable);
    removeNetworkObservable = FactoryUtility.nullIfEmpty(removeNetworkObservable);
    stateObservable         = FactoryUtility.nullIfEmpty(stateObservable);
    conStateObservable      = FactoryUtility.nullIfEmpty(conStateObservable);
  }

  @NonNull
  @Override
  public String toString() {
    return "NetworkEx{" +
        "isRemoved=" + isRemoved +
        ", super=" + super.toString() +
        '}';
  }
}
