package buzz.getcoco.iot.android;

import android.util.Log;

import androidx.annotation.CallSuper;
import androidx.annotation.NonNull;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Set;

import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.Resource;
import buzz.getcoco.iot.Zone;

public class ZoneEx extends Zone {

  private static final String TAG = "ZoneEx";

  private transient Identifier identifier;
  private transient MutableLiveData<String> nameObservable;
  private transient ImmutableList<ResourceEx> resources;
  private transient MutableLiveData<List<ResourceEx>> resourcesObservable;

  protected ZoneEx(int zoneId, Network parent) {
    super(zoneId, parent);
  }

  @Override
  protected void internalSetName(String name) {
    NetworkEx parent;
    super.internalSetName(name);

    if (null != (parent = (NetworkEx) getParent())) {
      parent.onZoneNameUpdated(this);
    }

    for (Resource resource : this) {
      ((ResourceEx)resource).onZoneNameChanged(name);
    }

    FactoryUtility.postValue(nameObservable, name);
  }

  @Override
  protected void internalRemoveResource(Resource resource) {
    super.internalRemoveResource(resource);

    notifyNetworkForChange();

    Log.d(TAG, "internalRemoveResource: "
        + "zone: " + this
        + ", resource: " + resource);

    compileAndPostResources();
  }

  @Override
  protected void internalAddResource(Resource resource) {
    super.internalAddResource(resource);

    notifyNetworkForChange();

    Log.d(TAG, "internalRemoveResource: "
        + "zone: " + this
        + ", resource: " + resource);

    compileAndPostResources();
  }

  private void compileAndPostResources() {
    if (null == resources)
      return;

    FactoryUtility.postValue(resourcesObservable, getCopyOfResources());
  }

  @CallSuper
  protected void onResourceUpdated(@NonNull ResourceEx res) {
    notifyNetworkForChange();
  }

  public Identifier getIdentifier() {
    return (null != identifier) ?  identifier : (identifier = Identifier.getIdentifier(this));
  }

  public LiveData<String> getNameObservable() {
    return (null != nameObservable) ? nameObservable : (nameObservable = FactoryUtility.createLiveData(getName()));
  }

  public List<ResourceEx> getCopyOfResources() {
    Set<Resource> resources;
    ImmutableList.Builder<ResourceEx> resourcesCopy = new ImmutableList.Builder<>();

    synchronized (resources = getResources()) {
      for (Resource r : resources) {
        resourcesCopy.add((ResourceEx) r);
      }
    }

    return (this.resources = resourcesCopy.build());
  }

  public LiveData<List<ResourceEx>> getResourcesObservable() {
    return (null != resourcesObservable) ? resourcesObservable : (resourcesObservable = FactoryUtility.createLiveData(getCopyOfResources()));
  }

  private void notifyNetworkForChange() {
    NetworkEx parent = (NetworkEx) getParent();

    if (null != parent)
      parent.onZoneUpdated(this);
  }

  private void trimMemory() {
    nameObservable = FactoryUtility.nullIfEmpty(nameObservable);
    resourcesObservable = FactoryUtility.nullIfEmpty(resourcesObservable);
  }
}
