package buzz.getcoco.iot.android;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.common.collect.ImmutableList;

import java.util.List;

import buzz.getcoco.iot.Attribute;
import buzz.getcoco.iot.Capability;
import buzz.getcoco.iot.ConnectivityStateManager;
import buzz.getcoco.iot.Device;
import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.PowerSource;
import buzz.getcoco.iot.ReceiverType;
import buzz.getcoco.iot.Resource;
import buzz.getcoco.iot.Zone;

public class ResourceEx extends Resource {

  private static final String TAG = "ResourceEx";

  private transient MutableLiveData<String> nameObservable;
  private transient MutableLiveData<String> zoneNameObservable;
  private transient MutableLiveData<String> metadataObservable;
  private transient MutableLiveData<String> manufacturerObservable;
  private transient MutableLiveData<String> modelObservable;
  private transient MutableLiveData<String> firmWareObservable;
  private transient MutableLiveData<PowerSource> powerSourceObservable;
  private transient MutableLiveData<ReceiverType> receiverTypeObservable;
  private transient MutableLiveData<Zone> parentZoneObservable;
  private transient MutableLiveData<Boolean> resourceRemovalObservable;

  // both stateChangeObservable & resourceReadyObservable serve the same purpose.
  // ready being less informative and state being more informative
  private transient MutableLiveData<State> stateChangeObservable;
  private transient MutableLiveData<Boolean> resourceReadyObservable;

  private transient MutableLiveData<List<Capability>> capabilitiesObservable;

  private transient State currentState = State.OFFLINE;

  private transient ImmutableList<Capability> capabilities;

  /**
   * This is a one time allocation for speed ups
   * This can be used to replace all {@link Identifier#getIdentifier(Resource)} calls
   */
  private transient Identifier id;

  public enum State {
    ONLINE,
    LOCAL,
    OFFLINE
  }

  protected ResourceEx(String eui, Device parentDevice, Zone parentZone) {
    super(eui, parentDevice, parentZone);
  }

  @Override
  protected void internalMarkAsReady() {
    super.internalMarkAsReady();
    onStateChange();
  }

  /**
   * @see #internalMarkAsReady()
   * @see DeviceEx#internalSetReady
   */
  protected void onStateChange() {
    if (!isReady()) {
      this.currentState = State.OFFLINE;
    } else {
      Network network = getParent().getParent();
      ConnectivityStateManager.State networkState = network.getConnectivityManagerState();

      switch (networkState) {
        case ONLINE:
          this.currentState = State.ONLINE;
          break;
        case LOCAL:
          this.currentState = State.LOCAL;
          break;
        case OFFLINE:
        case CONNECTING:
        default:
          this.currentState = State.OFFLINE;
      }

      Log.d(TAG, "setting: " + this.currentState + ", for networkState: " + networkState);
    }

    postCurrentState();

    for (Capability c : this) {
      for (Attribute a : c) {
        AttributeEx aEx = (AttributeEx) a;
        aEx.syncCurrentValue();
      }
    }
  }

  protected void postCurrentState() {
    Log.d(TAG, "postCurrentState: ready: " + isReady() + ", currentState: " + currentState + ", resource: " + getName());

    FactoryUtility.postValue(resourceReadyObservable, isReady());
    FactoryUtility.postValue(stateChangeObservable, currentState);
  }

  public State getState() {
    return currentState;
  }

  @Override
  protected void internalSetName(String name) {
    super.internalSetName(name);
    FactoryUtility.postValue(nameObservable, name);
  }

  @Override
  protected void internalSetMetadata(String metadata) {
    super.internalSetMetadata(metadata);
    FactoryUtility.postValue(metadataObservable, metadata);
  }

  @Override
  protected void internalSetManufacturer(String manufacturer) {
    super.internalSetManufacturer(manufacturer);
    FactoryUtility.postValue(manufacturerObservable, manufacturer);
  }

  @Override
  protected void internalSetModel(String model) {
    super.internalSetModel(model);
    FactoryUtility.postValue(modelObservable, model);
  }

  @Override
  protected void internalSetFirmware(String firmware) {
    super.internalSetFirmware(firmware);
    FactoryUtility.postValue(firmWareObservable, firmware);
  }

  @Override
  protected void internalSetPowerSource(PowerSource powerSource) {
    super.internalSetPowerSource(powerSource);
    FactoryUtility.postValue(powerSourceObservable, powerSource);
  }

  @Override
  protected void internalAddCapability(Capability capability) {
    super.internalAddCapability(capability);
    ZoneEx parentZone = getParentZone();

    postCapabilities();

    if (null != parentZone) {
      parentZone.onResourceUpdated(this);
    }
  }

  @Override
  protected void internalRemoveCapability(Capability.CapabilityId capabilityId) {
    super.internalRemoveCapability(capabilityId);

    ZoneEx parentZone = getParentZone();

    postCapabilities();

    if (null != parentZone) {
      parentZone.onResourceUpdated(this);
    }
  }

  protected void onCapabilitiesChanged(CapabilityWrapperEx capability) {
    ZoneEx parentZone = getParentZone();

    if (null != parentZone) {
      parentZone.onResourceUpdated(this);
    }
  }

  @Override
  protected void internalSetReceiverType(ReceiverType receiverType) {
    super.internalSetReceiverType(receiverType);

    FactoryUtility.postValue(receiverTypeObservable, receiverType);
  }

  @Override
  protected void internalSetParentZone(Zone parentZone) {
    super.internalSetParentZone(parentZone);

    FactoryUtility.postValue(parentZoneObservable, parentZone);
    FactoryUtility.postValue(zoneNameObservable, getParentZone().getName());
  }

  protected void onZoneNameChanged(String zoneName) {
    FactoryUtility.postValue(zoneNameObservable, zoneName);
  }

  @Override
  protected void internalRemoveResource() {
    super.internalRemoveResource();

    FactoryUtility.postValue(resourceRemovalObservable, true);
  }

  private void postCapabilities() {
    if (null == capabilities)
      return;

    FactoryUtility.postValue(capabilitiesObservable, getCapabilities());
  }

  public List<Capability> getCapabilities() {
    return (capabilities = ImmutableList.copyOf(getCapabilityMap().values()));
  }

  public LiveData<String> getNameObservable() {
    return (null != nameObservable) ? nameObservable : (nameObservable = FactoryUtility.createLiveData(getName()));
  }

  public LiveData<String> getMetadataObservable() {
    return (null != metadataObservable) ? metadataObservable : (metadataObservable = FactoryUtility.createLiveData(getMetadata()));
  }

  public LiveData<String> getManufacturerObservable() {
    return (null != manufacturerObservable) ? manufacturerObservable : (manufacturerObservable = FactoryUtility.createLiveData(getManufacturer()));
  }

  public LiveData<String> getModelObservable() {
    return (null != modelObservable) ? modelObservable : (modelObservable = FactoryUtility.createLiveData(getModel()));
  }

  public LiveData<String> getFirmWareObservable() {
    return (null != firmWareObservable) ? firmWareObservable : (firmWareObservable = FactoryUtility.createLiveData(getFirmware()));
  }

  public LiveData<PowerSource> getPowerSourceObservable() {
    return (null !=powerSourceObservable) ? powerSourceObservable : (powerSourceObservable = FactoryUtility.createLiveData(getPowerSource()));
  }

  public LiveData<List<Capability>> getCapabilitiesObservable() {
   return (null != capabilitiesObservable) ? capabilitiesObservable : (capabilitiesObservable = FactoryUtility.createLiveData(getCapabilities()));
  }

  public LiveData<ReceiverType> getReceiverTypeObservable() {
    return (null != receiverTypeObservable) ? receiverTypeObservable : (receiverTypeObservable = FactoryUtility.createLiveData(getReceiverType()));
  }

  public LiveData<Zone> getParentZoneObservable() {
    return (null != parentZoneObservable) ? parentZoneObservable : (parentZoneObservable = FactoryUtility.createLiveData(getParentZone()));
  }


  public LiveData<String> getZoneNameObservable() {
    return (null != zoneNameObservable) ? zoneNameObservable : (zoneNameObservable = FactoryUtility.createLiveData(getParentZone().getName()));
  }

  public LiveData<Boolean> getResourceRemovalObservable() {
    return (null != resourceRemovalObservable) ? resourceRemovalObservable : (resourceRemovalObservable = FactoryUtility.createLiveData(false));
  }

  public LiveData<State> getStateObservable() {
    return (null != stateChangeObservable) ? stateChangeObservable : (stateChangeObservable = FactoryUtility.createLiveData(getState()));
  }

  public MutableLiveData<Boolean> getResourceReadyObservable() {
    return (null != resourceReadyObservable) ? resourceReadyObservable : (resourceReadyObservable = FactoryUtility.createLiveData(isReady()));
  }

  public Identifier getIdentifier() {
    return (null != id) ? id : (id = Identifier.getIdentifier(this));
  }

  @Nullable
  public Network getParentNetwork() {
    Device d = getParent();
    return (null == d) ? null : d.getParent();
  }

  @NonNull
  @Override
  public String toString() {
    return "ResourceEx{" +
        "identifier=" + id +
        ", currentState=" + currentState +
        ", super=" + super.toString() +
        '}';
  }

  private void trimMemory() {
    id = null;

    nameObservable            =  FactoryUtility.nullIfEmpty(nameObservable);
    zoneNameObservable        =  FactoryUtility.nullIfEmpty(zoneNameObservable);
    metadataObservable        =  FactoryUtility.nullIfEmpty(metadataObservable);
    manufacturerObservable    =  FactoryUtility.nullIfEmpty(manufacturerObservable);
    modelObservable           =  FactoryUtility.nullIfEmpty(modelObservable);
    firmWareObservable        =  FactoryUtility.nullIfEmpty(firmWareObservable);
    powerSourceObservable     =  FactoryUtility.nullIfEmpty(powerSourceObservable);
    receiverTypeObservable    =  FactoryUtility.nullIfEmpty(receiverTypeObservable);
    parentZoneObservable      =  FactoryUtility.nullIfEmpty(parentZoneObservable);
    resourceRemovalObservable =  FactoryUtility.nullIfEmpty(resourceRemovalObservable);
    stateChangeObservable     =  FactoryUtility.nullIfEmpty(stateChangeObservable);
    resourceReadyObservable   =  FactoryUtility.nullIfEmpty(resourceReadyObservable);
    capabilitiesObservable =  FactoryUtility.nullIfEmpty(capabilitiesObservable);
  }
}
