/*
 * @file      Identifier.java
 * @brief     A class to transfer buzz.getcoco.iot.* objects betweeen activities
 *
 * @details
 *
 * @see
 * @author    Aryasomayajula Mrunal Krishna Sarma, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */
/*************************************************************************************/
/*===================================================================================*/
package buzz.getcoco.iot.android;

import android.os.Bundle;
import android.os.Parcel;
import android.os.Parcelable;

import androidx.annotation.Keep;

import buzz.getcoco.iot.Attribute;
import buzz.getcoco.iot.Capability;
import buzz.getcoco.iot.Device;
import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.Resource;
import buzz.getcoco.iot.Rule;
import buzz.getcoco.iot.Scene;
import buzz.getcoco.iot.Zone;

@Keep
public class Identifier extends buzz.getcoco.iot.Identifier implements Parcelable {

  @Keep
  public static Creator<Identifier> CREATOR = new Creator<Identifier>() {
    @Override
    public Identifier createFromParcel(Parcel parcel) {
      Bundle bundle = parcel.readBundle(getClass().getClassLoader());
      Identifier identifier;
      long deviceNodeId;
      String networkId, resourceEui;
      int capabilityId, attributeId, zoneId, sceneId, ruleId;

      if (null != bundle) {
        networkId = bundle.getString(Constants.NETWORK_ID);
        deviceNodeId = bundle.getLong(Constants.DEVICE_ID);
        resourceEui = bundle.getString(Constants.RESOURCE_ID);
        capabilityId = bundle.getInt(Constants.CAPABILITY_ID);
        attributeId = bundle.getInt(Constants.ATTRIBUTE_ID);
        zoneId = bundle.getInt(Constants.ZONE_ID);
        sceneId = bundle.getInt(Constants.SCENE_ID);
        ruleId = bundle.getInt(Constants.RULE_ID);

        identifier = new Identifier(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, zoneId, sceneId, ruleId);
      } else {
        identifier = new Identifier(null);
      }

      return identifier;
    }

    @Override
    public Identifier[] newArray(int size) {
      return new Identifier[size];
    }
  };

  private Identifier(String networkId, long deviceNodeId, String resourceEui, int capabilityId, int attributeId) {
    super(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, -1, -1, -1);
  }

  private Identifier(String networkId, long deviceNodeId, String resourceEui, int capabilityId, int attributeId, int zoneId, int sceneId, int ruleId) {
    super(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, zoneId, sceneId, ruleId);
  }

  private Identifier(buzz.getcoco.iot.Identifier id) {
    super(id);
  }

  @Override
  public int describeContents() {
    return 0;
  }

  @Override
  public void writeToParcel(Parcel parcel, int i) {
    Bundle bundle = new Bundle();

    bundle.putString(Constants.NETWORK_ID, networkId);
    bundle.putLong(Constants.DEVICE_ID, deviceNodeId);
    bundle.putString(Constants.RESOURCE_ID, resourceEui);
    bundle.putInt(Constants.CAPABILITY_ID, capabilityId);
    bundle.putInt(Constants.ATTRIBUTE_ID, attributeId);
    bundle.putInt(Constants.ZONE_ID, zoneId);
    bundle.putInt(Constants.SCENE_ID, sceneId);
    bundle.putInt(Constants.RULE_ID, ruleId);

    parcel.writeBundle(bundle);
  }

  public static Identifier getIdentifier(String networkId) {
    return new Identifier(networkId, -1, null, -1, -1);
  }

  public static Identifier getIdentifier(Network network) {
    return new Identifier(network.getId(), -1, null, -1, -1);
  }

  public static Identifier getIdentifier(Device device) {
    return new Identifier(device.getParent().getId(), device.getId(), null, -1, -1);
  }

  public static Identifier getIdentifier(Resource resource) {
    return new Identifier(resource.getParent().getParent().getId(), resource.getParent().getId(), resource.getId(), -1, -1);
  }

  public static Identifier getIdentifier(Capability capability) {
    return new Identifier(capability.getParent().getParent().getParent().getId(), capability.getParent().getParent().getId(), capability.getParent().getId(), capability.getId().getInt(), -1);
  }

  public static Identifier getIdentifier(Attribute attribute) {
    return new Identifier(attribute.getParent().getParent().getParent().getParent().getId(), attribute.getParent().getParent().getParent().getId(), attribute.getParent().getParent().getId(), attribute.getParent().getId().getInt(), attribute.getMapKey());
  }

  public static Identifier getIdentifier(Zone zone) {
    return new Identifier(zone.getParent().getId(), -1, null, -1, -1, zone.getId(), -1, -1);
  }

  public static Identifier getIdentifier(Scene scene) {
    return new Identifier(scene.getParent().getId(), -1, null, -1, -1, -1, scene.getId(), -1);
  }

  public static Identifier getIdentifier(Rule rule) {
    return new Identifier(rule.getParent().getId(), -1, null, -1, -1, -1, -1, rule.getId());
  }

  public static Identifier fromJson(String json) {
    return new Identifier(buzz.getcoco.iot.Identifier.fromJson(json));
  }
}
