package buzz.getcoco.iot.android;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import buzz.getcoco.iot.ConnectivityStateManager;
import buzz.getcoco.iot.Device;
import buzz.getcoco.iot.Network;
import buzz.getcoco.iot.Resource;
import buzz.getcoco.iot.Rule;
import buzz.getcoco.iot.Scene;
import buzz.getcoco.iot.Zone;

class BaseNetworkEx<D extends Device, Z extends Zone, S extends Scene, R extends Rule> extends Network {

  private static final String TAG = "BaseNetworkEx";

  private transient Identifier identifier;

  private transient MutableLiveData<List<D>> deviceListObservable;
  private transient MutableLiveData<List<Z>> zoneListObservable;
  private transient MutableLiveData<List<R>> ruleListObservable;
  private transient MutableLiveData<List<S>> sceneListObservable;

  private transient MutableLiveData<String> networkNameObservable;
  private transient MutableLiveData<Boolean> removeNetworkObservable;
  private transient MutableLiveData<Network.State> stateObservable;
  private transient MutableLiveData<ConnectivityStateManager.State> conStateObservable;

  private transient List<D> deviceList;
  private transient List<Z> zoneList;
  private transient List<R> ruleList;
  private transient List<S> sceneList;

  private transient boolean isRemoved;

  protected BaseNetworkEx(String id) {
    super(id);
  }

  @SuppressWarnings("unchecked")
  private static <T, U extends T> U castUp(T t) {
    return (U) t;
  }

  @Override
  protected void internalAddDevice(Device device) {
    super.internalAddDevice(device);

    if (null == deviceList)
      return;

    deviceList.add(castUp(device));
    sortDevices();

    FactoryUtility.postValue(deviceListObservable, deviceList);
  }

  @Override
  protected void internalRemoveDevice(long deviceNodeId) {
    super.internalRemoveDevice(deviceNodeId);

    if (null == deviceList)
      return;

    for(int i = 0; i < deviceList.size(); i++) {
      if (deviceList.get(i).getId() == deviceNodeId) {
        deviceList.remove(i);
        break;
      }
    }
    sortDevices();

    FactoryUtility.postValue(deviceListObservable, deviceList);
  }

  @Override
  protected void internalAddZone(Zone zone) {
    super.internalAddZone(zone);

    if (null == zoneList)
      return;

    zoneList.add(castUp(zone));
    sortZonesAndCorrectDefaultZoneValidity();

    FactoryUtility.postValue(zoneListObservable, zoneList);
  }

  @Override
  protected void internalRemoveZone(int zoneId) {
    super.internalRemoveZone(zoneId);

    if (null == zoneList)
      return;

    for (int i = 0; i < zoneList.size(); i++) {
      if (zoneId == zoneList.get(i).getId()) {
        zoneList.remove(i);
        break;
      }
    }
    sortZonesAndCorrectDefaultZoneValidity();

    FactoryUtility.postValue(zoneListObservable, zoneList);
  }

  @Override
  protected void internalAddRule(Rule rule) {
    super.internalAddRule(rule);

    if (null == ruleList)
      return;

    ruleList.add(castUp(rule));
    sortRules();

    FactoryUtility.postValue(ruleListObservable, ruleList);
  }

  @Override
  protected void internalRemoveRule(int ruleId) {
    super.internalRemoveRule(ruleId);

    if (null == ruleList)
      return;

    for (int i = 0; i < ruleList.size(); i++) {
      if (ruleList.get(i).getId() == ruleId) {
        ruleList.remove(i);
        break;
      }
    }
    sortRules();

    FactoryUtility.postValue(ruleListObservable, ruleList);
  }

  @Override
  protected void internalAddScene(Scene scene) {
    S sceneEx = castUp(scene);

    super.internalAddScene(scene);

    if (null == sceneList)
      return;

    if (!sceneList.contains(sceneEx)) {
      sceneList.add(sceneEx);
      sortScenes();
    }

    FactoryUtility.postValue(sceneListObservable, sceneList);
  }

  @Override
  protected void internalRemoveScene(int sceneId) {
    SceneEx scene = getScene(sceneId);

    super.internalRemoveScene(sceneId);

    if (null != scene)
      scene.onSceneRemoved();

    if (null == sceneList)
      return;

    for (int i = 0; i < sceneList.size(); i++) {
      if (sceneList.get(i).getId() == sceneId) {
        sceneList.remove(i);
        break;
      }
    }

    FactoryUtility.postValue(sceneListObservable, sceneList);
  }

  @Override
  protected void internalSetName(String name) {
    super.internalSetName(name);
    Log.d(TAG, "internalSetName: newName: " + name + ", ID: " + this.getId());

    FactoryUtility.postValue(networkNameObservable, name);
  }

  @Override
  protected void internalSetState(Network.State state) {
    super.internalSetState(state);
    Log.d(TAG, "internalSetState: state: " + state);

    FactoryUtility.postValue(stateObservable, state);
  }

  @Override
  protected void internalRemoveNetwork() {
    super.internalRemoveNetwork();
    Log.d(TAG, "internalRemoveNetwork: networkName: " + this.getName() + ", isRemoved: " + isRemoved);

    FactoryUtility.postValue(removeNetworkObservable, (isRemoved = true));
  }

  protected void onSceneNameUpdated(Scene scene) {
    Log.d(TAG, "onSceneNameUpdated: scene: " + scene);

    sortScenes();
    FactoryUtility.postValue(sceneListObservable, getSceneList());
  }

  protected void onRuleNameUpdated(Rule rule) {
    Log.d(TAG, "onRuleNameUpdated: rule: " + rule);

    sortRules();
    FactoryUtility.postValue(ruleListObservable, getRuleList());
  }

  protected void onZoneNameUpdated(Zone zone) {
    Log.d(TAG, "onZoneNameUpdated: zone: " + zone);

    sortZonesAndCorrectDefaultZoneValidity();
    FactoryUtility.postValue(zoneListObservable, getZoneList());
  }

  protected void onDeviceNameUpdated(Device device) {
    Log.d(TAG, "onDeviceNameUpdated: device: " + device);

    sortDevices();
    FactoryUtility.postValue(deviceListObservable, deviceList);
  }

  @Override
  protected void internalSetConnectivityState(ConnectivityStateManager.State connectivityState) {
    ConnectivityStateManager.State currentState = getConnectivityManagerState();

    super.internalSetConnectivityState(connectivityState);

    if (currentState == connectivityState)
      return;

    for (Device d : this) {
      for (Resource r : d) {
        ResourceEx rex = (ResourceEx) r;

        rex.onStateChange();
      }
    }

    FactoryUtility.postValue(conStateObservable, connectivityState);
  }

  public Identifier getIdentifier() {
    return (null != identifier) ?  identifier : (identifier = Identifier.getIdentifier(this));
  }

  public List<D> getDeviceList() {
    if (null == deviceList) {
      deviceList = Collections.synchronizedList(new ArrayList<>());

      for (Device device : getDeviceIterable()) {
        deviceList.add(castUp(device));
      }

      sortDevices();
    }

    return deviceList;
  }

  public List<Z> getZoneList() {
    if (null == zoneList) {
      zoneList = Collections.synchronizedList(new ArrayList<>());

      for (Zone zone : getZoneIterable()) {
        zoneList.add(castUp(zone));
      }

      sortZonesAndCorrectDefaultZoneValidity();
    }

    return zoneList;
  }

  public List<R> getRuleList() {
    if (null == ruleList) {
      ruleList = Collections.synchronizedList(new ArrayList<>());

      for (Rule rule : getRuleIterable()) {
        ruleList.add(castUp(rule));
      }

      sortRules();
    }

    return ruleList;
  }

  public List<S> getSceneList() {
    if (null == sceneList) {
      sceneList = Collections.synchronizedList(new ArrayList<>());

      for (Scene scene : getSceneIterable()) {
        sceneList.add(castUp(scene));
      }

      sortScenes();
    }

    return sceneList;
  }

  @SuppressWarnings("SynchronizeOnNonFinalField")
  protected void sortZonesAndCorrectDefaultZoneValidity() {
    if (null == zoneList)
      return;

    synchronized (zoneList) {
      Z defaultZone = getZone(Zone.DEFAULT_ZONE_ID);

      if (null != defaultZone) {
        // NOTE: comparator always pushes the default zone to the end
        boolean defaultHasResources = !defaultZone.getResources().isEmpty();
        boolean containsDefaultZone = zoneList.lastIndexOf(defaultZone) >= 0;

        if (containsDefaultZone && !defaultHasResources) {
          zoneList.remove(defaultZone);
        }

        if (!containsDefaultZone && defaultHasResources) {
          zoneList.add(defaultZone);
        }
      } else {
        Log.w(TAG, "sortZonesAndCorrectDefaultZoneValidity: default zone missing");
      }

      Collections.sort(zoneList, Utils.ZONE_COMPARATOR);
    }
  }

  @SuppressWarnings("SynchronizeOnNonFinalField")
  protected void sortScenes() {
    if (null == sceneList)
      return;

    synchronized (sceneList) {
      Collections.sort(sceneList, Utils.SCENE_COMPARATOR);
    }
  }

  @SuppressWarnings("SynchronizeOnNonFinalField")
  protected void sortRules() {
    if (null == ruleList)
      return;

    synchronized (ruleList) {
      Collections.sort(ruleList, Utils.RULE_COMPARATOR);
    }
  }

  @SuppressWarnings("SynchronizeOnNonFinalField")
  protected void sortDevices() {
    if (null == deviceList)
      return;

    synchronized (deviceList) {
      Collections.sort(deviceList, Utils.DEVICE_COMPARATOR);
    }
  }

  public LiveData<List<D>> getDeviceListObservable() {
    return (null != deviceListObservable) ? deviceListObservable : (deviceListObservable = FactoryUtility.createLiveData(getDeviceList()));
  }

  /**
   * NOTE: This will remove the default zone, if it has no resources
   */
  public LiveData<List<Z>> getZoneListObservable() {
    return ((null != zoneListObservable) ? zoneListObservable : (zoneListObservable = FactoryUtility.createLiveData(getZoneList())));
  }

  public LiveData<List<R>> getRuleListObservable() {
    return (null != ruleListObservable) ? ruleListObservable : (ruleListObservable = FactoryUtility.createLiveData(getRuleList()));
  }

  public LiveData<List<S>> getSceneListObservable() {
    return (null != sceneListObservable) ? sceneListObservable : (sceneListObservable = FactoryUtility.createLiveData(getSceneList()));
  }

  public LiveData<String> getNetworkNameObservable() {
    return (null != networkNameObservable) ? networkNameObservable : (networkNameObservable = FactoryUtility.createLiveData(getName()));
  }

  public LiveData<Network.State> getStateObservable() {
    return (null != stateObservable) ? stateObservable : (stateObservable = FactoryUtility.createLiveData(getState()));
  }

  public LiveData<Boolean> getRemoveNetworkObservable() {
    return (null != removeNetworkObservable) ? removeNetworkObservable : (removeNetworkObservable = FactoryUtility.createLiveData(isRemoved));
  }

  public LiveData<ConnectivityStateManager.State> getConnectivityManageStateObservable() {
    return (null != conStateObservable) ? conStateObservable : (conStateObservable = FactoryUtility.createLiveData(getConnectivityManagerState()));
  }

  private void trimMemory() {
    deviceListObservable    = FactoryUtility.nullIfEmpty(deviceListObservable);
    zoneListObservable      = FactoryUtility.nullIfEmpty(zoneListObservable);
    ruleListObservable      = FactoryUtility.nullIfEmpty(ruleListObservable);
    sceneListObservable     = FactoryUtility.nullIfEmpty(sceneListObservable);
    networkNameObservable   = FactoryUtility.nullIfEmpty(networkNameObservable);
    removeNetworkObservable = FactoryUtility.nullIfEmpty(removeNetworkObservable);
    stateObservable         = FactoryUtility.nullIfEmpty(stateObservable);
    conStateObservable      = FactoryUtility.nullIfEmpty(conStateObservable);
  }

  @NonNull
  @Override
  public String toString() {
    return "NetworkEx{" +
        "isRemoved=" + isRemoved +
        ", super=" + super.toString() +
        '}';
  }
}
