/*
 * Copyright 2016 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.rio.dhcp;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;

import br.com.objectos.core.util.ImmutableList;
import br.com.objectos.rio.net.ConfiguredNetworkAdapter;
import br.com.objectos.rio.net.HardwareAddress;
import br.com.objectos.rio.net.IpAddress;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public class DhcpServerBuilder {

  private final ImmutableList.Builder<Host> hostList = ImmutableList.builder();
  private final ImmutableList.Builder<DhcpServerListener> listenerList = ImmutableList.builder();

  ConfiguredNetworkAdapter adapter;
  private IpAddress ipAddress;
  private int port;

  public DhcpServerBuilder() {
  }

  public DhcpServerBuilderDsl bindTo(ConfiguredNetworkAdapter adapter) {
    return bindTo(adapter, 67);
  }

  public DhcpServerBuilderDsl bindTo(ConfiguredNetworkAdapter adapter, int port) {
    this.adapter = Objects.requireNonNull(adapter);
    ipAddress = adapter.ipAddress();
    this.port = port;
    return new NetworkAdapterDsl();
  }

  public DhcpServerBuilderDsl localhost(int port) {
    this.port = port;
    ipAddress = IpAddress.localhost();
    return new LocalhostDsl();
  }

  DhcpServerConfiguration configuration() {
    return new DhcpServerConfiguration(
        ipAddress,
        port,
        configuredAdapterMap());
  }

  List<DhcpServerListener> listenerList() {
    return listenerList.build();
  }

  private Map<HardwareAddress, ServerConfiguredAdapter> configuredAdapterMap() {
    return hostList.build()
        .stream()
        .map(host -> host.toConfiguredAdapter(ipAddress))
        .collect(Collectors.toMap(
            ServerConfiguredAdapter::hardwareAddress,
            Function.identity(),
            throwingMerger(),
            LinkedHashMap::new));
  }

  private static <T> BinaryOperator<T> throwingMerger() {
    return (u,v) -> { throw new IllegalStateException(String.format("Duplicate key %s", u)); };
}

  private class LocalhostDsl extends Dsl {
    @Override
    public DhcpServer build() {
      return new LocalhostDhcpServer(DhcpServerBuilder.this);
    }
  }

  private class NetworkAdapterDsl extends Dsl {
    @Override
    public DhcpServer build() {
      return new NetworkAdapterDhcpServer(DhcpServerBuilder.this);
    }
  }

  private abstract class Dsl
      implements
      DhcpServerBuilderDsl,
      DhcpServerBuilderDsl.AddHost,
      DhcpServerBuilderDsl.WithListener {

    @Override
    public AddHost addHost(Host host) {
      hostList.add(host);
      return this;
    }

    @Override
    public final WithListener withListener(DhcpServerListener listener) {
      listenerList.add(listener);
      return this;
    }

  }

}