/*
 * Copyright 2016 Objectos, Fábrica de Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package br.com.objectos.rio.dhcp;

import java.util.List;
import java.util.Objects;
import java.util.logging.Logger;

import br.com.objectos.rio.net.HardwareAddress;
import br.com.objectos.rio.net.IpAddress;
import br.com.objectos.rio.net.NetInteger;
import br.com.objectos.rio.udp.Packet;
import br.com.objectos.rio.udp.UdpException;

/**
 * @author marcio.endo@objectos.com.br (Marcio Endo)
 */
public abstract class DhcpServer extends AbstractServer {

  private static final Logger logger = Logger.getLogger(DhcpServer.class.getName());

  private final DhcpServerConfiguration configuration;
  private final List<DhcpServerListener> listenerList;

  DhcpServer(DhcpServerBuilder builder) {
    configuration = builder.configuration();
    listenerList = builder.listenerList();
  }

  public DhcpServerConfiguration configuration() {
    return configuration;
  }

  @Override
  public void onPacket(Packet packet) throws UdpException {
    if (!packet.fromPort(clientPort())) {
      return;
    }

    DhcpDatagram datagram = packet.decode(DhcpDatagram::read);
    logger.info("Received: " + datagram);

    MessageType.clientMessageTypes()
        .stream()
        .filter(type -> datagram.matches(type))
        .findFirst()
        .map(type -> type.readClientMessage(datagram))
        .filter(in -> in.matches(this))
        .map(in -> in.nextMessage(this))
        .ifPresent(out -> {
          try {
            out.broadcastOrSend(this);
          } catch (UdpException e) {
            e.printStackTrace();
          }
        });
  }

  @Override
  public DhcpServer start() throws UdpException {
    return (DhcpServer) super.start();
  }

  @Override
  int clientPort() {
    return port() + 1;
  }

  @Override
  ServerConfiguredAdapter configuredAdapter(HardwareAddress chaddr) {
    return configuration.configuredAdapter(chaddr);
  }

  @Override
  final IpAddress ipAddress() {
    return configuration.ipAddress();
  }

  @Override
  boolean matches(IpAddress ipAddress) {
    return Objects.equals(ipAddress(), ipAddress);
  }

  @Override
  void onAckSent(AckMessage message) {
    for (DhcpServerListener listener : listenerList) {
      listener.onAckSent(message);
    }
  }

  @Override
  void onDiscovery(DiscoveryMessage message) {
    for (DhcpServerListener listener : listenerList) {
      listener.onDiscoveryReceived(message);
    }
  }

  @Override
  void onOfferSent(OfferMessage message) {
    for (DhcpServerListener listener : listenerList) {
      listener.onOfferSent(message);
    }
  }

  @Override
  void onRequest(RequestMessage message) {
    for (DhcpServerListener listener : listenerList) {
      listener.onRequestReceived(message);
    }
  }

  final int port() {
    return configuration.port();
  }

  IpAddress.Array domainNameServer(HardwareAddress chaddr) {
    return IpAddress.Array.of(IpAddress.of(8, 8, 8, 8), IpAddress.of(8, 8, 4, 4));
  }

  @Override
  IpAddress ipAddress(HardwareAddress chaddr) {
    return IpAddress.of(192, 168, 0, 102);
  }

  NetInteger leaseTime(HardwareAddress chaddr) {
    return NetInteger.of(86400);
  }

  IpAddress.Array router(HardwareAddress chaddr) {
    return IpAddress.Array.of(ipAddress());
  }

  IpAddress subnetMask(HardwareAddress chaddr) {
    return IpAddress.of(255, 255, 255, 0);
  }

}