/*
 * Copyright (C) 2011-2021 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.testing.fs;

import br.com.objectos.collections.list.ImmutableList;
import br.com.objectos.fs.Directory;
import br.com.objectos.fs.JavaIoTmpdir;
import java.io.IOException;

/**
 * Provides {@code static} methods for working with directories in testing
 * environments.
 *
 * @since 2
 */
public final class TestingFs {

  private TestingFs() {}

  /**
   * Creates a randomly named directory that is automatically removed during the
   * JVM shutdown process.
   *
   * <p>
   * The returned directory will be a direct child of the directory returned by
   * the {@link JavaIoTmpdir} class.
   *
   * @return a randomly named temporary directory
   *
   * @throws IOException
   *         if an I/O error occurs
   */
  public static Directory createTempDirectory() throws IOException {
    return TempDirectories.INSTANCE.next();
  }

  /**
   * Returns the parent directory of the resource named
   * {@code TEST-INF/.marker}.
   *
   * <p>
   * This is intended to ease working with test resources (as defined by Maven
   * build tool) in modular project.
   *
   * @return the parent directory of the resource named {@code TEST-INF/.marker}
   *
   * @throws IOException
   *         if the marker cannot be found, the marker is not a regular file or
   *         another I/O error occurs
   */
  public static Directory getTestInf() throws IOException {
    return TestInf.get();
  }

  /**
   * Recursively lists the relative pathnames of all entries of the specified
   * directory; the result is both sorted and immutable.
   *
   * @param directory
   *        the directory to list
   *
   * @return sorted immutable list of all entries
   *
   * @throws IOException
   *         if an I/O error occurs
   */
  public static ImmutableList<String> list(Directory directory) throws IOException {
    return DirectoryListMode.ALL.list(directory);
  }

  /**
   * Recursively lists the relative pathnames of all (regular) files of the
   * specified directory; the result is both sorted and immutable.
   *
   * @param directory
   *        the directory to list
   *
   * @return sorted immutable list of all files
   *
   * @throws IOException
   *         if an I/O error occurs
   */
  public static ImmutableList<String> listFiles(Directory directory) throws IOException {
    return DirectoryListMode.FILES.list(directory);
  }

}