package br.com.edsilfer.android.messenger.service.softkeyboard

import java.util.ArrayList
import java.util.concurrent.atomic.AtomicBoolean

import android.os.Handler
import android.os.Message
import android.util.Log
import android.view.View
import android.view.ViewGroup
import android.view.inputmethod.InputMethodManager
import android.widget.EditText

// TODO: ADD THIS CLASS TO TEXTVIEW KOTLIN API
class SoftKeyboard(private val layout: ViewGroup, private val im: InputMethodManager) : View.OnFocusChangeListener {

    companion object {
        private val CLEAR_FOCUS = 0
    }

    private var mBottomLayout: Int = 0
    private val mCoordinates: IntArray
    private var isVisible: Boolean = false
    private var mInputList: MutableList<EditText>? = null
    private val mKeyboardThread: SoftKeyboardChangesThread
    // References to a focused EditText
    private var mTempView: View? = null

    init {
        keyboardHideByDefault()
        initEditTexts(layout)
        mCoordinates = IntArray(2)
        isVisible = false
        mKeyboardThread = SoftKeyboardChangesThread()
        mKeyboardThread.start()
    }

    fun setSoftKeyboardCallback(mCallback: SoftKeyboardListener) {
        mKeyboardThread.setCallback(mCallback)
    }

    fun unRegisterSoftKeyboardCallback() {
        mKeyboardThread.stopThread()
    }

    private val layoutCoordinates: Int
        get() {
            layout.getLocationOnScreen(mCoordinates)
            return mCoordinates[1] + layout.height
        }

    private fun keyboardHideByDefault() {
        layout.isFocusable = true
        layout.isFocusableInTouchMode = true
    }

    /*
     * InitEditTexts now handles EditTexts in nested views
     * Thanks to Francesco Verheye (verheye.francesco@gmail.com)
     */
    private fun initEditTexts(viewGroup: ViewGroup) {
        if (mInputList == null)
            mInputList = ArrayList<EditText>()

        val childCount = viewGroup.childCount
        for (i in 0..childCount - 1) {
            val v = viewGroup.getChildAt(i)

            if (v is ViewGroup) {
                initEditTexts(v)
            }

            if (v is EditText) {
                v.onFocusChangeListener = this
                v.isCursorVisible = true
                mInputList!!.add(v)
            }
        }
    }

    /*
     * OnFocusChange does update mTempView correctly now when keyboard is still shown
     * Thanks to Israel Dominguez (dominguez.israel@gmail.com)
     */
    override fun onFocusChange(v: View, hasFocus: Boolean) {
        if (hasFocus) {
            mTempView = v
            if (!isVisible) {
                mBottomLayout = layoutCoordinates
                mKeyboardThread.keyboardOpened()
                isVisible = true
            }
        }
    }

    /**
     * Clear selected EditText focus
     */
    private val mHandler = object : Handler() {
        override fun handleMessage(m: Message) {
            when (m.what) {
                CLEAR_FOCUS -> if (mTempView != null) {
                    mTempView!!.clearFocus()
                    mTempView = null
                }
            }
        }
    }

    private inner class SoftKeyboardChangesThread : Thread() {

        private val mStarted: AtomicBoolean
        private var mCallback: SoftKeyboardListener? = null

        init {
            mStarted = AtomicBoolean(true)
        }

        fun setCallback(callback: SoftKeyboardListener) {
            mCallback = callback
        }

        override fun run() {
            while (mStarted.get()) {
                // Wait until keyboard is requested to open
                synchronized(this) {
                    try {
                        (this as Object).wait()
                    } catch (e: InterruptedException) {
                        e.printStackTrace()
                    }
                }

                var currentBottomLocation = layoutCoordinates

                // There is some lag between open soft-keyboard function and when it really appears.
                while (currentBottomLocation == mBottomLayout && mStarted.get()) {
                    currentBottomLocation = layoutCoordinates
                }

                if (mStarted.get())
                    mCallback!!.onSoftKeyboardShown()

                // When keyboard is opened from EditText, initial bottom location is greater than mBottomLayout
                // and at some moment equals mBottomLayout.
                // That broke the previous logic, so I added this new loop to handle this.
                while (currentBottomLocation >= mBottomLayout && mStarted.get()) {
                    currentBottomLocation = layoutCoordinates
                }

                // Now Keyboard is shown, keep checking layout dimensions until keyboard is gone
                while (currentBottomLocation != mBottomLayout && mStarted.get()) {
                    synchronized(this) {
                        try {
                            (this as Object).wait(500)
                        } catch (e: InterruptedException) {
                            Log.e("SoftKeyboardThread", e.message)
                        }
                    }
                    currentBottomLocation = layoutCoordinates
                }

                if (mStarted.get())
                    mCallback!!.onSoftKeyboardHidden()

                // if keyboard has been opened clicking and EditText.
                if (isVisible && mStarted.get())
                    isVisible = false

                // if an EditText is focused, remove its focus (on UI thread)
                if (mStarted.get())
                    mHandler.obtainMessage(CLEAR_FOCUS).sendToTarget()
            }
        }

        fun keyboardOpened() {
            synchronized(this) {
                (this as Object).notify()
            }
        }

        fun stopThread() {
            synchronized(this) {
                mStarted.set(false)
                (this as Object).notify()
            }
        }
    }

    interface SoftKeyboardListener {
        fun onSoftKeyboardHidden()
        fun onSoftKeyboardShown()
    }
}
