/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import aQute.bnd.make.*;
import aQute.bnd.service.*;
import aQute.libg.header.*;
import aQute.libg.reporter.*;

public class Processor implements Reporter, Constants, Closeable {

    public String              DEFAULT_PLUGINS       = "";                      // "aQute.lib.spring.SpringComponent";

    public final static String DEFAULT_BND_EXTENSION = ".bnd";
    public final static String DEFAULT_JAR_EXTENSION = ".jar";
    public final static String DEFAULT_BAR_EXTENSION = ".bar";

    private List<String>       errors                = new ArrayList<String>();
    private List<String>       warnings              = new ArrayList<String>();
    boolean                    pedantic;
    boolean                    trace                 = false;
    boolean                    exceptions;
    String[]                   METAPACKAGES          = { "META-INF",
            "OSGI-INF", "OSGI-OPT"                  };

    List<Object>               plugins;
    private File               base                  = new File("")
                                                             .getAbsoluteFile();
    private List<Closeable>    toBeClosed            = newList();

    public void getInfo(Processor processor, String prefix) {
        if (isFailOk())
            addAll(warnings, processor.getErrors(), prefix + ": ");
        else
            addAll(errors, processor.getErrors(), prefix + ": ");
        addAll(warnings, processor.getWarnings(), prefix + ": ");
    }

    public void getInfo(Processor processor) {
        getInfo(processor, "");
    }

    private <T> void addAll(List<String> to, List<? extends T> from,
            String prefix) {
        for (T x : from) {
            to.add(prefix + x);
        }
    }

    public void warning(String string) {
        warnings.add(string);
    }

    public void error(String string) {
        if (isFailOk())
            warning(string);
        else
            errors.add(string);
    }

    public void error(String string, Throwable t) {
        if (isFailOk())
            warning(string + ": " + t);
        else
            errors.add(string + " : " + t);
        if (exceptions)
            t.printStackTrace();
    }

    public List<String> getWarnings() {
        return warnings;
    }

    public List<String> getErrors() {
        return errors;
    }

    public Map<String, Map<String, String>> parseHeader(String value) {
        return parseHeader(value, this);
    }

    /**
     * Standard OSGi header parser.
     * 
     * @param value
     * @return
     */
    @SuppressWarnings("unchecked")
    static public Map<String, Map<String, String>> parseHeader(String value,
            Processor logger) {
        return OSGiHeader.parseHeader(value, logger);
    }

    public Map<String, Clazz> analyzeBundleClasspath(Jar dot,
            Map<String, Map<String, String>> bundleClasspath,
            Map<String, Map<String, String>> contained,
            Map<String, Map<String, String>> referred,
            Map<String, Set<String>> uses) throws IOException {
        Map<String, Clazz> classSpace = new HashMap<String, Clazz>();

        if (bundleClasspath.isEmpty()) {
            analyzeJar(dot, "", classSpace, contained, referred, uses);
        } else {
            for (String path : bundleClasspath.keySet()) {
                if (path.equals(".")) {
                    analyzeJar(dot, "", classSpace, contained, referred, uses);
                    continue;
                }
                //
                // There are 3 cases:
                // - embedded JAR file
                // - directory
                // - error
                //

                Resource resource = dot.getResource(path);
                if (resource != null) {
                    try {
                        Jar jar = new Jar(path);
                        addClose(jar);
                        EmbeddedResource.build(jar, resource);
                        analyzeJar(jar, "", classSpace, contained, referred,
                                uses);
                    } catch (Exception e) {
                        warning("Invalid bundle classpath entry: " + path + " "
                                + e);
                    }
                } else {
                    if (dot.getDirectories().containsKey(path)) {
                        analyzeJar(dot, path, classSpace, contained, referred,
                                uses);
                    } else {
                        warning("No sub JAR or directory " + path);
                    }
                }
            }
        }
        return classSpace;
    }

    public void addClose(Closeable jar) {
        toBeClosed.add(jar);
    }

    /**
     * We traverse through all the classes that we can find and calculate the
     * contained and referred set and uses. This method ignores the Bundle
     * classpath.
     * 
     * @param jar
     * @param contained
     * @param referred
     * @param uses
     * @throws IOException
     */
    private void analyzeJar(Jar jar, String prefix,
            Map<String, Clazz> classSpace,
            Map<String, Map<String, String>> contained,
            Map<String, Map<String, String>> referred,
            Map<String, Set<String>> uses) throws IOException {
        next: for (String path : jar.getResources().keySet()) {
            if (path.startsWith(prefix)) {
                String relativePath = path.substring(prefix.length());
                String pack = getPackage(relativePath);

                if (pack != null && !contained.containsKey(pack)) {
                    if (!(pack.equals(".") || isMetaData(relativePath))) {

                        Map<String, String> map = new LinkedHashMap<String, String>();
                        contained.put(pack, map);
                        Resource pinfo = jar.getResource(prefix
                                + pack.replace('.', '/') + "/packageinfo");
                        if (pinfo != null) {
                            InputStream in = pinfo.openInputStream();
                            String version = parsePackageInfo(in);
                            in.close();
                            if (version != null)
                                map.put("version", version);
                        }
                    }
                }

                if (path.endsWith(".class")) {
                    Resource resource = jar.getResource(path);
                    Clazz clazz;

                    try {
                        InputStream in = resource.openInputStream();
                        clazz = new Clazz(relativePath, in);
                        in.close();
                    } catch (Throwable e) {
                        errors.add("Invalid class file: " + relativePath + " "
                                + e.getMessage());
                        e.printStackTrace();
                        continue next;
                    }

                    String calculatedPath = clazz.getClassName() + ".class";
                    if (!calculatedPath.equals(relativePath))
                        error("Class in different directory than declared. Path from class name is "
                                + calculatedPath
                                + " but the path in the jar is "
                                + relativePath
                                + " from " + jar);

                    classSpace.put(relativePath, clazz);
                    referred.putAll(clazz.getReferred());

                    // Add all the used packages
                    // to this package
                    Set<String> t = uses.get(pack);
                    if (t == null)
                        uses.put(pack, t = new LinkedHashSet<String>());
                    t.addAll(clazz.getReferred().keySet());
                    t.remove(pack);
                }
            }
        }
        for (AnalyzerPlugin plugin : getPlugins(AnalyzerPlugin.class)) {
            if (plugin instanceof AnalyzerPlugin) {
                AnalyzerPlugin analyzer = (AnalyzerPlugin) plugin;
                try {
                    analyzer.analyzeJar(this, jar, prefix, classSpace,
                            contained, referred, uses);
                } catch (Exception e) {
                    error("Plugin Analyzer " + analyzer + " throws exception "
                            + e);
                    e.printStackTrace();
                }
            }
        }

    }

    /**
     * Decide if the package is a metadata package.
     * 
     * @param pack
     * @return
     */
    boolean isMetaData(String pack) {
        for (int i = 0; i < METAPACKAGES.length; i++) {
            if (pack.startsWith(METAPACKAGES[i]))
                return true;
        }
        return false;
    }

    public String getPackage(String clazz) {
        int n = clazz.lastIndexOf('/');
        if (n < 0)
            return ".";
        return clazz.substring(0, n).replace('/', '.');
    }

    //
    // We accept more than correct OSGi versions because in a later
    // phase we actually cleanup maven versions. But it is a bit yucky
    //
    static Pattern packageinfo = Pattern.compile("version\\s+([^\\s]*)\\s*");

    static String parsePackageInfo(InputStream jar) throws IOException {
        try {
            byte[] buf = EmbeddedResource.collect(jar);
            String line = new String(buf).trim();
            Matcher m = packageinfo.matcher(line);
            if (m.matches()) {
                return m.group(1);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Remove all entries from a map that start with a specific prefix
     * 
     * @param <T>
     * @param source
     * @param prefix
     * @return
     */
    static <T> Map<String, T> removeKeys(Map<String, T> source, String prefix) {
        Map<String, T> temp = new TreeMap<String, T>(source);
        for (Iterator<String> p = temp.keySet().iterator(); p.hasNext();) {
            String pack = (String) p.next();
            if (pack.startsWith(prefix))
                p.remove();
        }
        return temp;
    }

    public void progress(String s) {
        // System.out.println(s);
    }

    public boolean isPedantic() {
        return pedantic;
    }

    public void setPedantic(boolean pedantic) { // System.out.println("Set
        // pedantic: " + pedantic + " "
        // + this );
        this.pedantic = pedantic;
    }

    public static File getFile(File base, String file) {
        File f = new File(file);
        if (f.isAbsolute())
            return f;
        int n;

        f = base.getAbsoluteFile();
        while ((n = file.indexOf('/')) > 0) {
            String first = file.substring(0, n);
            file = file.substring(n + 1);
            if (first.equals(".."))
                f = f.getParentFile();
            else
                f = new File(f, first);
        }
        return new File(f, file).getAbsoluteFile();
    }

    public File getFile(String file) {
        return getFile(base, file);
    }

    /**
     * Return a list of plugins that implement the given class.
     * 
     * @param clazz
     *            Each returned plugin implements this class/interface
     * @return A list of plugins
     */
    public <T> List<T> getPlugins(Class<T> clazz) {
        List<T> l = new ArrayList<T>();
        List<Object> all = getPlugins();
        for (Object plugin : all) {
            if (clazz.isInstance(plugin))
                l.add(clazz.cast(plugin));
        }
        return l;
    }

    /**
     * Return a list of plugins. Plugins are defined with the -plugin command.
     * They are class names, optionally associated with attributes. Plugins can
     * implement the Plugin interface to see these attributes.
     * 
     * Any object can be a plugin.
     * 
     * @return
     */
    public List<Object> getPlugins() {
        if (this.plugins != null)
            return this.plugins;

        String spe = getProperty(Analyzer.PLUGIN, DEFAULT_PLUGINS);
        Map<String, Map<String, String>> plugins = parseHeader(spe);
        List<Object> list = new ArrayList<Object>();

        // Add the default plugins. Only if non is specified
        // will they be removed.
        list.add(new MakeBnd());
        list.add(new MakeCopy());

        for (Map.Entry<String, Map<String, String>> entry : plugins.entrySet()) {
            String key = (String) entry.getKey();
            if (key.equals("none"))
                return this.plugins = newList();

            try {
                if (isPedantic()) {
                    warning("Using plugin " + key);
                }

                // Plugins could use the same class with different
                // parameters so we could have duplicate names Remove
                // the ! added by the parser to make each name unique.
                while (key.endsWith(Constants.DUPLICATE_MARKER))
                    key = key.substring(0, key.length() - 1);

                Class<?> c = (Class<?>) getClass().getClassLoader().loadClass(
                        key);
                Object plugin = c.newInstance();
                if (plugin instanceof Plugin) {
                    ((Plugin) plugin).setProperties(entry.getValue());
                    ((Plugin) plugin).setReporter(this);
                }
                list.add(plugin);
            } catch (Exception e) {
                error("Problem loading the plugin: " + key + " exception: " + e);
            }
        }
        return this.plugins = list;
    }

    protected String getProperty(String key, String deflt) {
        return deflt;
    }

    public boolean isFailOk() {
        String v = getProperty(Analyzer.FAIL_OK, null);
        return v != null && v.equalsIgnoreCase("true");
    }

    public File getBase() {
        return base;
    }

    public void setBase(File base) {
        this.base = base;
    }

    public void clear() {
        errors.clear();
        warnings.clear();
    }

    public void trace(String msg) {
        if (trace) {
            System.out.println("# " + msg);
        }
    }

    public Map<String, Map<String, String>> newClauses() {
        return new LinkedHashMap<String, Map<String, String>>();
    }

    public <T> List<T> newList() {
        return new ArrayList<T>();
    }

    public <T> Set<T> newSet() {
        return new TreeSet<T>();
    }

    public static <K, V> Map<K, V> newMap() {
        return new LinkedHashMap<K, V>();
    }

    public <T> List<T> newList(Collection<T> t) {
        return new ArrayList<T>(t);
    }

    public <T> Set<T> newSet(Collection<T> t) {
        return new TreeSet<T>(t);
    }

    public <K, V> Map<K, V> newMap(Map<K, V> t) {
        return new LinkedHashMap<K, V>(t);
    }

    public void close() {
        for (Closeable c : toBeClosed) {
            try {
                c.close();
            } catch (IOException e) {
                // Who cares?
            }
        }
    }

    /**
     * Clean up version parameters. Other builders use more fuzzy definitions of
     * the version syntax. This method cleans up such a version to match an OSGi
     * version.
     * 
     * @param VERSION_STRING
     * @return
     */
    static Pattern fuzzyVersion      = Pattern
                                             .compile(
                                                     "(\\d+)(\\.(\\d+)(\\.(\\d+))?)?([^a-zA-Z0-9](.*))?",
                                                     Pattern.DOTALL);
    static Pattern fuzzyVersionRange = Pattern
                                             .compile(
                                                     "(\\(|\\[)\\s*([-\\da-zA-Z.]+)\\s*,\\s*([-\\da-zA-Z.]+)\\s*(\\]|\\))",
                                                     Pattern.DOTALL);
    static Pattern fuzzyModifier     = Pattern.compile("(\\d+[.-])*(.*)",
                                             Pattern.DOTALL);

    static Pattern nummeric          = Pattern.compile("\\d*");

    static public String cleanupVersion(String version) {
        Matcher m = fuzzyVersionRange.matcher(version);
        if (m.matches()) {
            String prefix = m.group(1);
            String first = m.group(2);
            String last = m.group(3);
            String suffix = m.group(4);
            return prefix + cleanupVersion(first) + "," + cleanupVersion(last) + suffix;
        } else {
            m = fuzzyVersion.matcher(version);
            if (m.matches()) {
                StringBuffer result = new StringBuffer();
                String d1 = m.group(1);
                String d2 = m.group(3);
                String d3 = m.group(5);
                String qualifier = m.group(7);

                if (d1 != null) {
                    result.append(d1);
                    if (d2 != null) {
                        result.append(".");
                        result.append(d2);
                        if (d3 != null) {
                            result.append(".");
                            result.append(d3);
                            if (qualifier != null) {
                                result.append(".");
                                cleanupModifier(result, qualifier);
                            }
                        } else if (qualifier != null) {
                            result.append(".0.");
                            cleanupModifier(result, qualifier);
                        }
                    } else if (qualifier != null) {
                        result.append(".0.0.");
                        cleanupModifier(result, qualifier);
                    }
                    return result.toString();
                }
            }
        }
        return version;
    }

    static void cleanupModifier(StringBuffer result, String modifier) {
        Matcher m = fuzzyModifier.matcher(modifier);
        if (m.matches())
            modifier = m.group(2);

        for (int i = 0; i < modifier.length(); i++) {
            char c = modifier.charAt(i);
            if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z')
                    || (c >= 'A' && c <= 'Z') || c == '_' || c == '-')
                result.append(c);
        }
    }

}
