/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.lang.reflect.*;
import java.text.*;
import java.util.*;
import java.util.regex.*;

import aQute.libg.reporter.*;

/**
 * Provide a macro processor. This processor can replace variables in strings
 * based on a properties and a domain. The domain can implement functions that
 * start with a "_" and take args[], the names of these functions are available
 * as functions in the macro processor (without the _). Macros can nest to any
 * depth but may not contain loops.
 * 
 */
public class Macro {
    Properties properties;
    Reporter   domain;

    public Macro(Properties properties, Reporter domain) {
        this.properties = properties;
        this.domain = domain;
    }

    public Macro(Reporter processor) {
        this(new Properties(), processor);
    }

    public String process(String line) {
        return process(line, null);
    }

    String process(String line, Link link) {
        StringBuffer sb = new StringBuffer();
        process(line, 0, '\u0000', '\u0000', sb, link);
        return sb.toString();
    }

    int process(String line, int index, char begin, char end,
            StringBuffer result, Link link) {
        int nesting = 1;

        StringBuffer variable = new StringBuffer();
        outer: while (index < line.length()) {
            char c1 = line.charAt(index++);
            if (c1 == end) {
                if (--nesting == 0) {
                    result.append(replace(variable.toString(), link));
                    return index;
                }
            } else if (c1 == begin)
                nesting++;
            else if (c1 == '$' && index < line.length() - 2) {
                char c2 = line.charAt(index);
                char terminator = getTerminator(c2);
                if (terminator != 0) {
                    index = process(line, index + 1, c2, terminator, variable,
                            link);
                    continue outer;
                }
            }
            variable.append(c1);
        }
        result.append(variable);
        return index;
    }

    char getTerminator(char c) {
        switch (c) {
        case '(':
            return ')';
        case '[':
            return ']';
        case '{':
            return '}';
        case '<':
            return '>';
        case '\u00ab': // Guillemet double << >>
            return '\u00bb';
        case '‹': // Guillemet single
            return '›';
        }
        return 0;
    }

    protected String replace(String key, Link link) {
        if (link != null && link.contains(key))
            return "${infinite:" + link.toString() + "}";

        if (key != null) {
            key = key.trim();
            if (key.length() > 0) {
                String value = (String) properties.getProperty(key);
                if (value != null)
                    return process(value, new Link(link, key));

                value = doCommands(key);
                if (value != null)
                    return process(value, new Link(link, key));

                if (key != null && key.trim().length() > 0) {
                    value = System.getProperty(key);
                    if (value != null)
                        return value;
                }
                domain.warning("No translation found for macro: " + key);
            } else {
                domain.warning("Found empty macro key");
            }
        } else {
            domain.warning("Found null macro key");
        }
        return "${" + key + "}";
    }

    /**
     * Parse the key as a command. A command consist of parameters separated by
     * ':'.
     * 
     * @param key
     * @return
     */
    static Pattern commands = Pattern.compile(";");

    private String doCommands(String key) {
        String[] args = commands.split(key);
        if (args == null || args.length == 0)
            return null;

        String result = doCommand(domain, args);
        if (result != null)
            return result;

        return doCommand(this, args);
    }

    private String doCommand(Object target, String[] args) {
        String cname = "_" + args[0].replaceAll("-", "_");
        try {
            Method m = target.getClass().getMethod(cname,
                    new Class[] { String[].class });
            return (String) m.invoke(target, new Object[] { args });
        } catch (NoSuchMethodException e) {
            // Ignore
        } catch (InvocationTargetException e) {
            domain.warning("Exception in replace: " + e.getCause());
        } catch (Exception e) {
            domain.warning("Exception in replace: " + e);
        }
        return null;
    }

    public String _filter(String args[]) {
        if (args.length != 3) {
            domain.warning("Invalid nr of arguments to filter "
                    + Arrays.asList(args));
            return null;
        }

        String list[] = args[1].split("\\s*,\\s*");
        StringBuffer sb = new StringBuffer();
        String del = "";
        for (int i = 0; i < list.length; i++) {
            if (list[i].matches(args[2])) {
                sb.append(del);
                sb.append(list[i]);
                del = ", ";
            }
        }
        return sb.toString();
    }

    public String _filterout(String args[]) {
        if (args.length != 3) {
            domain.warning("Invalid nr of arguments to filterout "
                    + Arrays.asList(args));
            return null;
        }

        String list[] = args[1].split("\\s*,\\s*");
        StringBuffer sb = new StringBuffer();
        String del = "";
        for (int i = 0; i < list.length; i++) {
            if (!list[i].matches(args[2])) {
                sb.append(del);
                sb.append(list[i]);
                del = ", ";
            }
        }
        return sb.toString();
    }

    public String _sort(String args[]) {
        if (args.length != 2) {
            domain.warning("Invalid nr of arguments to join "
                    + Arrays.asList(args));
            return null;
        }

        String list[] = args[1].split("\\s*,\\s*");
        StringBuffer sb = new StringBuffer();
        String del = "";
        Arrays.sort(list);
        for (int i = 0; i < list.length; i++) {
            sb.append(del);
            sb.append(list[i]);
            del = ", ";
        }
        return sb.toString();
    }

    public String _join(String args[]) {
        if (args.length == 1) {
            domain.warning("Invalid nr of arguments to join "
                    + Arrays.asList(args));
            return null;
        }

        StringBuffer sb = new StringBuffer();
        String del = "";
        for (int i = 1; i < args.length; i++) {
            String list[] = args[i].split("\\s*,\\s*");
            for (int j = 0; j < list.length; j++) {
                sb.append(del);
                sb.append(list[j]);
                del = ", ";
            }
        }
        return sb.toString();
    }

    public String _if(String args[]) {
        if (args.length < 3) {
            domain.warning("Invalid nr of arguments to if "
                    + Arrays.asList(args));
            return null;
        }

        if (args[1].trim().length() != 0)
            return args[2];
        if (args.length > 3)
            return args[3];
        else
            return "";
    }

    public String _now(String args[]) {
        return new Date().toString();
    }

    public String _fmodified(String args[]) throws Exception {
        if (args.length != 2) {
            domain.warning("Fmodified takes only 1 parameter "
                    + Arrays.asList(args));
            return null;
        }
        long time = 0;
        String list[] = args[1].split("\\s*,\\s*");
        for (int i = 0; i < list.length; i++) {
            File f = new File(list[i].trim());
            if (f.exists() && f.lastModified() > time)
                time = f.lastModified();
        }
        return "" + time;
    }

    public String _long2date(String args[]) {
        try {
            return new Date(Long.parseLong(args[1])).toString();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "not a valid long";
    }

    public String _literal(String args[]) {
        if (args.length != 2)
            throw new RuntimeException(
                    "Need a value for the ${literal;<value>} macro");
        return "${" + args[1] + "}";
    }

    public String _def(String args[]) {
        if (args.length != 2)
            throw new RuntimeException(
                    "Need a value for the ${def;<value>} macro");

        String value = (String) properties.get(args[1]);
        if (value == null)
            return "";
        else
            return value;
    }

    /**
     * 
     * replace ; <list> ; regex ; replace
     * 
     * @param args
     * @return
     */
    public String _replace(String args[]) {
        if (args.length != 4) {
            domain.warning("Invalid nr of arguments to replace "
                    + Arrays.asList(args));
            return null;
        }

        String list[] = args[1].split("\\s*,\\s*");
        StringBuffer sb = new StringBuffer();
        String del = "";
        for (int i = 0; i < list.length; i++) {
            String element = list[i].trim();
            if (!element.equals("")) {
                sb.append(del);
                sb.append(element.replaceAll(args[2], args[3]));
                del = ", ";
            }
        }

        return sb.toString();
    }

    public String _warning(String args[]) {
        for (int i = 1; i < args.length; i++) {
            domain.warning(process(args[i]));
        }
        return "";
    }

    public String _error(String args[]) {
        for (int i = 1; i < args.length; i++) {
            domain.error(process(args[i]));
        }
        return "";
    }

    /**
     * toclassname ; <path>.class ( , <path>.class ) *
     * 
     * @param args
     * @return
     */
    public String _toclassname(String args[]) {
        if (args.length != 2) {
            domain.warning("Invalid nr of arguments to toclassname "
                    + Arrays.asList(args));
            return null;
        }

        String list[] = args[1].split("\\s*,\\s*");
        StringBuffer sb = new StringBuffer();
        String del = "";
        for (int i = 0; i < list.length; i++) {
            String element = list[i];
            if (element.endsWith(".class")) {
                element = element.substring(0, element.length() - 6).replace(
                        '/', '.');
                sb.append(del);
                sb.append(element);
                del = ", ";
            } else {
                domain
                        .warning("in toclassname, "
                                + args[1]
                                + " is not a class path because it does not end in .class");
            }
        }
        return sb.toString();
    }

    /**
     * toclassname ; <path>.class ( , <path>.class ) *
     * 
     * @param args
     * @return
     */
    public String _toclasspath(String args[]) {
        if (args.length != 2) {
            domain.warning("Invalid nr of arguments to toclasspath "
                    + Arrays.asList(args));
            return null;
        }

        String list[] = args[1].split("\\s*,\\s*");
        StringBuffer sb = new StringBuffer();
        String del = "";
        for (int i = 0; i < list.length; i++) {
            String element = list[i];
            element = element.replace('.', '/') + ".class";
            sb.append(del);
            sb.append(element);
            del = ", ";
        }
        return sb.toString();
    }

    public String _dir(String args[]) {
        if (args.length < 2) {
            domain.warning("Need at least one file name for ${dir;...}");
            return null;
        } else {
            String del = "";
            StringBuffer sb = new StringBuffer();
            for (int i = 1; i < args.length; i++) {
                File f = new File(args[i]).getAbsoluteFile();
                if (f.exists() && f.getParentFile().exists()) {
                    sb.append(del);
                    sb.append(f.getParentFile().getAbsolutePath());
                    del = ",";
                }
            }
            return sb.toString();
        }

    }

    public String _basename(String args[]) {
        if (args.length < 2) {
            domain.warning("Need at least one file name for ${basename;...}");
            return null;
        } else {
            String del = "";
            StringBuffer sb = new StringBuffer();
            for (int i = 1; i < args.length; i++) {
                File f = new File(args[i]).getAbsoluteFile();
                if (f.exists() && f.getParentFile().exists()) {
                    sb.append(del);
                    sb.append(f.getName());
                    del = ",";
                }
            }
            return sb.toString();
        }

    }

    public String _isfile(String args[]) {
        if (args.length < 2) {
            domain.warning("Need at least one file name for ${isfile;...}");
            return null;
        } else {
            boolean isfile = true;
            for (int i = 1; i < args.length; i++) {
                File f = new File(args[i]).getAbsoluteFile();
                isfile &= f.isFile();
            }
            return isfile ? "true" : "false";
        }

    }

    public String _isdir(String args[]) {
        if (args.length < 2) {
            domain.warning("Need at least one file name for ${isdir;...}");
            return null;
        } else {
            boolean isdir = true;
            for (int i = 1; i < args.length; i++) {
                File f = new File(args[i]).getAbsoluteFile();
                isdir &= f.isDirectory();
            }
            return isdir ? "true" : "false";
        }

    }

    public String _tstamp(String args[]) {
        String format = "yyyyMMddhhmm";
        long now = System.currentTimeMillis();

        if (args.length > 1) {
            format = args[1];
            if (args.length > 2) {
                now = Long.parseLong(args[2]);
                if (args.length > 3) {
                    domain.warning("Too many arguments for tstamp: "
                            + Arrays.toString(args));
                }
            }
        }
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(new Date(now));
    }

    // Helper class to track expansion of variables
    // on the stack.
    static class Link {
        Link   previous;
        String key;

        public Link(Link previous, String key) {
            this.previous = previous;
            this.key = key;
        }

        public boolean contains(String key) {
            if (this.key.equals(key))
                return true;

            if (previous == null)
                return false;

            return previous.contains(key);
        }

        public String toString() {
            StringBuffer sb = new StringBuffer();
            String del = "[";
            for (Link r = this; r != null; r = r.previous) {
                sb.append(del);
                sb.append(r.key);
                del = ",";
            }
            sb.append("]");
            return sb.toString();
        }
    };

}
