/*
 *  This file is part of Bracket Properties
 *  Copyright 2011 David R. Smith
 *
 */
package asia.redact.bracket.properties.mgmt;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * <pre>
 * 
 * Suppose you have a set of server-localized properties files conforming to the schemata 
 * described in ServerLocale class and located in various places on the server. Then, to find 
 * all the associated files and put them in effective order, use something like:
 * 
 * List<File> folders = new ArrayList<File>();
 * folders.add("/Users/dave");
 * folders.add("/etc/properties");
 * folders.add("/opt/app/myapp");
 * ...
 * ServerLocalizedFileFinder finder = new ServerLocalizedFileFinder(folders, "myprops");
 * finder.findFiles();
 * 
 * </pre>
 * 
 * @author Dave
 *
 */
public class ServerLocalizedFileFinder {

	List<File> parentFolders;
	String baseName;
	
	private ServerLocalizedFileFinder(){
		parentFolders = new ArrayList<File>();
	}
	
	public ServerLocalizedFileFinder(File parentFolder, String baseName) {
		this();
		parentFolders.add(parentFolder);
		// trim if needed
		if(baseName.endsWith(".properties")){
			this.baseName = baseName.substring(0,baseName.length()-11);
		}else {
			this.baseName = baseName;
		}
	}
	
	public ServerLocalizedFileFinder(List<File> parentFolders, String baseName) {
		this();
		parentFolders.addAll(parentFolders);
		// trim baseName string if needed
		if(baseName.endsWith(".properties")){
			this.baseName = baseName.substring(0,baseName.length()-11);
		}else {
			this.baseName = baseName;
		}
	}
	
	/**
	 * Find all of the files in a given set for a baseName across a given set of parent folders. Return in appropriate order
	 * 
	 * @return
	 */
	public List<File> findFiles() {
		ArrayList<File> list = new ArrayList<File>();
		for(File folder:parentFolders){
		File [] files = folder.listFiles(new MyFileFilter(baseName));
			for(File f: files) list.add(f);
		}
		if(list.size()>1) Collections.sort(list, new MyComparator());
		return list;
	}
	
	
	/**
	 * Do the work of comparing properties files which are in the ServerLocale schemata
	 * 
	 * @author Dave
	 *
	 */
	class MyComparator implements Comparator<File> {

		public int compare(File o1, File o2) {
			String name0 = o1.getName();
			String name1 = o2.getName();
			int c0 = countUnderscores(name0);
			int c1 = countUnderscores(name1);
			int resultUnderscores = 0;
			if(c0>c1) resultUnderscores = 1;
			if(c0<c1) resultUnderscores = -1;
			
		//	if(resultUnderscores==0){
			//	return name0.compareTo(name1);
		//	}
			 return resultUnderscores;
			
		}
		
		private int countUnderscores(final String s) {
			int count = 0;
			for(int i = 0; i<s.length();i++){
				if(s.charAt(i)=='_')count++;
			}
			return count;
		}
		
	}
	
	/**
	 * Files must meet the conditions of matching baseName and ending in .properties to qualify
	 * 
	 * @author Dave
	 *
	 */
	class MyFileFilter implements FileFilter {
		
		String baseName;

		public MyFileFilter(String baseName) {
			super();
			this.baseName = baseName;
		}

		public boolean accept(File pathname) {
			if(pathname.isFile()&& 
					pathname.getName().startsWith(baseName)&&
					pathname.getName().endsWith(".properties")) return true;
			return false;
		}
	}
}
