/*
 *  This file is part of Bracket Properties
 *  Copyright 2011 David R. Smith
 *
 */
package asia.redact.bracket.properties.mgmt;

import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * <pre>
 * 
 * Builder to generate a ServerLocale.
 * 
 * This implementation uses the local host name to try to generate a
 * ServerLocale. For this to work, the host names of the servers involved
 * must obey a regular pattern that contains the environment, cluster, and
 * instance. A regular expression is submitted which sniffs for this pattern.
 * I think most admins name their servers with some kind of pattern, but it
 * may be nothing which is amenable to this code.
 * 
 * This class is very specific to servers I work with - your best bet
 * is to extend ServerLocaleBuilder to get your server name your way
 * 
 * </pre>
 * 
 * @author Dave
 *
 */
public class HostnameServerLocaleBuilder extends ServerLocaleBuilder {
	
	String pattern;

	public HostnameServerLocaleBuilder() {
		super();
		// this is the server name pattern generally in use at the Department of Education and Communities, NSW, Australia.
		pattern = "(u[dtap])\\-(\\w+)\\-(\\w+)(.+?)";
	}

	public HostnameServerLocaleBuilder(String pattern) {
		super();
		this.pattern = pattern;
	}

	public ServerLocale findServerLocale() {
		Pattern pat = Pattern.compile(pattern);
		String inet = fromInet();
		Matcher matcher = pat.matcher(inet);
		ServerLocale loc = null;
		if(matcher.matches()){
			loc = new ServerLocale();
			String env = matcher.group(1);
			if(env.equals("ud")){
				loc.setEnvironment("dev");
			}else if(env.equals("ut")){
				loc.setEnvironment("test");
			}else if(env.equals("ua")){
				loc.setEnvironment("acceptance");
			}else if(env.equals("up")){
				loc.setEnvironment("prod");
			}
			String cluster = matcher.group(2);
			loc.setCluster(cluster);
			String instance = matcher.group(3);
			loc.setInstance(instance);
		}else{
			throw new RuntimeException("Looks like either your pattern is wrong or your server's hostname is not amenable to this approach");
		}
		return loc;
	}

	public String getPattern() {
		return pattern;
	}

	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

}
