/*
 *  This file is part of Bracket Properties
 *  Copyright 2011 David R. Smith
 *
 */

package asia.redact.bracket.properties;

import java.util.ArrayList;
import java.util.List;

/**
 * <pre>
 * Representation of values which accounts better for the actual complexity of a potential properties value.
 * 
 * Note that we consider comments to be a part of the value. This is not really correct,
 * but it generally works because there is a one to many relation between a key and 
 * a set of comments. Only comments above the key/value pair are considered part of the set
 * 
 * 
 * </pre>
 * 
 * @author Dave
 *
 */
public class ValueModel {

	final static String lineSeparator = System.getProperty("line.separator");
	protected final List<String> comments;
	protected final List<String> values;
	protected char separator = '=';
	
	public ValueModel() {
		comments = new ArrayList<String>();
		values = new ArrayList<String>();
	}
	
	public ValueModel(String ... value) {
		this();
		for(String v: value){
			values.add(v);
		}
	}
	
    ValueModel(List<String> comments, List<String> values) {
		this.comments = comments;
		this.values = values;
	}

	@Override
	public synchronized int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((comments == null) ? 0 : comments.hashCode());
		result = prime * result + separator;
		result = prime * result + ((values == null) ? 0 : values.hashCode());
		return result;
	}

	@Override
	public synchronized boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ValueModel other = (ValueModel) obj;
		if (comments == null) {
			if (other.comments != null)
				return false;
		} else if (!comments.equals(other.comments))
			return false;
		if (separator != other.separator)
			return false;
		if (values == null) {
			if (other.values != null)
				return false;
		} else if (!values.equals(other.values))
			return false;
		return true;
	}

	public char getSeparator() {
		return separator;
	}

	public void setSeparator(char separator) {
		this.separator = separator;
	}

	public List<String> getComments() {
		return comments;
	}

	public List<String> getValues() {
		return values;
	}
	
	public void addValue(String value){
		values.add(value);
	}
	
	public void addComment(String comment){
		comments.add(comment);
	}
	
	public void clearComments() {
		comments.clear();
	}
	
	public String getValue(){
		StringBuilder b = new StringBuilder();
		for(String value:values) b.append(value);
		return b.toString();
	}
	
	public String toString(){
		return getValue();
	}
	
	public String toXML(int spaces) {
		
		StringBuilder sp = new StringBuilder();
		for(int i=0;i<=spaces;i++)sp.append(" ");
		
		StringBuilder builder = new StringBuilder();
		
		for(String comment: comments){
			builder.append(sp.toString());
			builder.append("<comment><![CDATA[").append(comment).append("]]></comment>").append(lineSeparator);
		}
		
		builder.append(sp.toString());
		builder.append("<separator>").append(separator).append("</separator>").append(lineSeparator);
		
		for(String value: values){
			builder.append(sp.toString());
			if(value != null && (!value.equals(""))) {
				builder.append("<value><![CDATA[").append(value).append("]]></value>").append(lineSeparator);
			}else{
				builder.append("<value/>").append(lineSeparator);
			}
			
		}
		
		return builder.toString();
	}
	
}
