package app.raybritton.elog.data

import app.raybritton.elog.BuildConfig
import app.raybritton.elog.ELog
import app.raybritton.elog.ELogConfig
import app.raybritton.elog.arch.LogModule
import okhttp3.*
import timber.log.Timber
import java.io.IOException
import java.net.URLEncoder
import java.text.SimpleDateFormat
import java.util.*
import java.util.concurrent.CopyOnWriteArraySet
import java.util.concurrent.TimeUnit

class LogUpload {
    //Ids of the logs currently being uploaded
    private var uploading = CopyOnWriteArraySet<String>()
    private val dateFormatter = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ", Locale.UK)

    private val okhttp by lazy {
        OkHttpClient.Builder()
            .readTimeout(ELogConfig.readTimeout, TimeUnit.SECONDS)
            .writeTimeout(ELogConfig.writeTimeout, TimeUnit.SECONDS)
            .connectTimeout(ELogConfig.connectTimeout, TimeUnit.SECONDS)
            .apply {
                ELogConfig.interceptor?.let {
                    addInterceptor(it)
                }
            }
            .build()
    }

    /**
     * if logId is not provided an id will be generated automatically using the ELog.idGen setting
     *
     * onComplete may be called from any thread
     */
    fun upload(log: LogFile, manual: Boolean, logTitle: String?, code: String = ELog.generateCode(), onComplete: (result: Result) -> Unit) {
        if (ELog.uploadServerUrl.isBlank()) {
            throw IllegalStateException("uploadServerUrl has not been set")
        }
        if (ELogConfig.apiPath.isBlank()) {
            throw IllegalStateException("apiPath has not been set")
        }
        if (uploading.contains(log.id)) {
            Timber.d("Already uploading ${log.id}")
            return
        }
        if (log.code != null) {
            Timber.d("Log already submitted")
            throw IllegalStateException("Log already submitted")
        }
        if (ELog.appBuild.isEmpty() || ELog.appPackage.isEmpty() || ELog.appVersionName.isEmpty()) {
            throw IllegalStateException("ELog.appBuild, appPackage, appVersion and appVersionName must be set before submitting a log")
        }
        uploading.add(log.id)
        val mediaType = MediaType.get("text/plain")
        val body = RequestBody.create(mediaType, log.file)
        val request = Request.Builder()
            .url(ELog.uploadServerUrl + ELogConfig.apiPath)
            .header("x-code", code)
            .header("x-version", ELog.appVersionName)
            .header("x-version-code", ELog.appVersion.toString())
            .header("x-app", ELog.appPackage)
            .header("x-build", ELog.appBuild)
            .header("x-created", dateFormatter.format(Date(log.created)))
            .header("x-device-id", LogPreferences.deviceId)
            .header("x-manual", manual.toString())
            .header("x-lib-ver", BuildConfig.ELOGS_LIB_VER)
            .header("x-addt-info", URLEncoder.encode(ELog.additionalInfo, "UTF-8"))
            .header("x-user-id", ELog.userId)
            .header("x-platform", "android")
            .apply {
                if (logTitle != null) {
                    header("x-auto-title", logTitle)
                }
                if (log.continuationCode != null) {
                    header("x-cont-code", log.continuationCode)
                }
            }
            .post(body)
            .build()

        okhttp.newCall(request)
            .enqueue(object : Callback {
                override fun onFailure(call: Call, e: IOException) {
                    onComplete(Result.Error(e))
                    uploading.remove(log.id)
                }

                override fun onResponse(call: Call, response: Response) {
                    if (response.isSuccessful) {
                        if (log.file.absolutePath == ELogTree.file.absolutePath) {
                            ELogTree.file = LogModule.logManager.createLog(code)
                            Timber.d("New file started")
                        }
                        LogModule.logManager.setCodeForLog(log, code)
                        onComplete(Result.Success(code))
                    } else {
                        onComplete(Result.Error(IOException("${response.code()} ${response.message()}")))
                    }
                    uploading.remove(log.id)
                }
            })
    }
}